import { useState, useEffect, useCallback } from 'react';

export interface FilterTab {
  id: string;
  label: string;
  token: string;
  isActive: boolean;
}

export interface UseFilterTabsConfig {
  planId: string;
  storageKeyPrefix: string;
}

export interface UseFilterTabsReturn {
  tabs: FilterTab[];
  activeTabs: FilterTab[];
  activeTokens: string[];
  addTab: (label: string) => void;
  toggleTab: (tabId: string) => void;
  removeTab: (tabId: string) => void;
  reorderTabs: (newOrder: FilterTab[]) => void;
  setTabsFromTokens: (tokens: string[]) => void;
}

function getStorageKey(prefix: string, planId: string): string {
  return `${prefix}:${planId}`;
}

function sanitizeToken(input: string): string {
  return input.trim().toUpperCase();
}

function loadTabsFromStorage(storageKey: string): FilterTab[] {
  if (typeof window === 'undefined') return [];
  
  try {
    const stored = localStorage.getItem(storageKey);
    
    if (!stored) return [];
    
    const parsed = JSON.parse(stored);
    
    if (!Array.isArray(parsed)) return [];
    
    return parsed.filter(
      (item): item is FilterTab =>
        typeof item === 'object' &&
        item !== null &&
        typeof item.id === 'string' &&
        typeof item.label === 'string' &&
        typeof item.token === 'string' &&
        typeof item.isActive === 'boolean'
    );
  } catch (error) {
    console.error('Error loading filter tabs from storage:', error);
    return [];
  }
}

function saveTabsToStorage(storageKey: string, tabs: FilterTab[]): void {
  if (typeof window === 'undefined') return;
  
  try {
    localStorage.setItem(storageKey, JSON.stringify(tabs));
  } catch (error) {
    console.error('Error saving filter tabs to storage:', error);
  }
}

export function useFilterTabs({ planId, storageKeyPrefix }: UseFilterTabsConfig): UseFilterTabsReturn {
  const storageKey = getStorageKey(storageKeyPrefix, planId);
  const [tabs, setTabs] = useState<FilterTab[]>(() => loadTabsFromStorage(storageKey));

  useEffect(() => {
    saveTabsToStorage(storageKey, tabs);
  }, [storageKey, tabs]);

  const addTab = useCallback((label: string) => {
    const token = sanitizeToken(label);
    
    if (!token) return;
    
    setTabs(prev => {
      const exists = prev.some(tab => tab.token === token);
      if (exists) return prev;
      
      const newTab: FilterTab = {
        id: `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        label: token,
        token: token,
        isActive: true
      };
      
      return [...prev, newTab];
    });
  }, []);

  const toggleTab = useCallback((tabId: string) => {
    setTabs(prev => prev.map(tab => 
      tab.id === tabId ? { ...tab, isActive: !tab.isActive } : tab
    ));
  }, []);

  const removeTab = useCallback((tabId: string) => {
    setTabs(prev => prev.filter(tab => tab.id !== tabId));
  }, []);

  const reorderTabs = useCallback((newOrder: FilterTab[]) => {
    setTabs(newOrder);
  }, []);

  const setTabsFromTokens = useCallback((tokens: string[]) => {
    const sanitizedTokens = tokens.map(t => sanitizeToken(t)).filter(Boolean);
    
    setTabs(prev => {
      const existingTokens = prev.map(t => t.token);
      const missingTokens = sanitizedTokens.filter(token => !existingTokens.includes(token));
      
      const newTabs = missingTokens.map(token => ({
        id: `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        label: token,
        token: token,
        isActive: true
      }));
      
      const updatedTabs = prev.map(tab => ({
        ...tab,
        isActive: sanitizedTokens.includes(tab.token)
      }));
      
      const hasChanges = 
        newTabs.length > 0 || 
        updatedTabs.some((tab, idx) => tab.isActive !== prev[idx].isActive);
      
      if (!hasChanges) return prev;
      
      return [...updatedTabs, ...newTabs];
    });
  }, []);

  const activeTabs = tabs.filter(t => t.isActive);
  const activeTokens = activeTabs.map(t => t.token);

  return {
    tabs,
    activeTabs,
    activeTokens,
    addTab,
    toggleTab,
    removeTab,
    reorderTabs,
    setTabsFromTokens
  };
}
