import { Pool } from "pg";
import type { ProductionComponentBatch, ProductionPlanLine } from "@shared/schema";

interface BatchGroupKey {
  color: string | null;
  edgingPattern: string | null;
  drillingPattern: string | null;
  boardType: string | null;
  componentType: string | null;
}

interface ComponentForBatching {
  id: number;
  generatedName: string;
  componentType: string;
  color: string | null;
  boardType: string | null;
  drillingPattern: string | null;
  edgingPattern: string | null;
  quantity: number;
}

interface BatchResult {
  batch: ProductionComponentBatch;
  componentIds: number[];
}

function generateBatchKey(component: ComponentForBatching): string {
  return JSON.stringify({
    color: component.color || '',
    edgingPattern: component.edgingPattern || '',
    drillingPattern: component.drillingPattern || '',
    boardType: component.boardType || '',
    componentType: component.componentType || '',
  });
}

function parseBatchKey(key: string): BatchGroupKey {
  const parsed = JSON.parse(key);
  return {
    color: parsed.color || null,
    edgingPattern: parsed.edgingPattern || null,
    drillingPattern: parsed.drillingPattern || null,
    boardType: parsed.boardType || null,
    componentType: parsed.componentType || null,
  };
}

export async function generateBatchNumber(pool: Pool): Promise<string> {
  const result = await pool.query(
    `SELECT batch_number FROM production.production_component_batches 
     WHERE batch_number ~ '^BATCH-[0-9]{6}$' 
     ORDER BY batch_number DESC LIMIT 1`
  );

  if (result.rows.length === 0) {
    return 'BATCH-000001';
  }

  const lastNumber = parseInt(result.rows[0].batch_number.split('-')[1]);
  return `BATCH-${String(lastNumber + 1).padStart(6, '0')}`;
}

/**
 * Generates component batches from a plan line.
 * Groups components by color, edging pattern, drilling pattern, board type, and component type.
 * 
 * @param pool - PostgreSQL connection pool
 * @param planLineId - ID of the production plan line
 * @returns Array of created component batches
 */
export async function generateBatchesForPlanLine(
  pool: Pool,
  planLineId: number
): Promise<BatchResult[]> {
  // TRANSACTION SAFETY: Wrap entire operation in transaction for atomicity
  const client = await pool.connect();
  
  try {
    await client.query('BEGIN');
    
    // Get plan line details
    const planLineResult = await client.query(
      'SELECT * FROM production.production_plan_lines WHERE id = $1',
      [planLineId]
    );

    if (planLineResult.rows.length === 0) {
      throw new Error(`Plan line not found: ${planLineId}`);
    }

  const planLine: ProductionPlanLine = {
    id: planLineResult.rows[0].id,
    planId: planLineResult.rows[0].plan_id,
    productId: planLineResult.rows[0].product_id,
    quantity: planLineResult.rows[0].quantity,
    sourceType: planLineResult.rows[0].source_type,
    sourceId: planLineResult.rows[0].source_id,
    sourceReference: planLineResult.rows[0].source_reference,
    productionOrderId: planLineResult.rows[0].production_order_id,
    routingId: planLineResult.rows[0].routing_id,
    routingOverride: planLineResult.rows[0].routing_override,
    bomId: planLineResult.rows[0].bom_id,
    plannedStartDate: planLineResult.rows[0].planned_start_date,
    plannedEndDate: planLineResult.rows[0].planned_end_date,
    status: planLineResult.rows[0].status,
    sequence: planLineResult.rows[0].sequence,
    notes: planLineResult.rows[0].notes,
    metadata: planLineResult.rows[0].metadata,
    createdAt: planLineResult.rows[0].created_at,
    updatedAt: planLineResult.rows[0].updated_at,
  };

  // Get BOM ID for the product
  let bomId = planLine.bomId;
  
  if (!bomId) {
    // Try to find active BOM for this product
    const bomResult = await client.query(
      `SELECT id FROM bom.product_boms 
       WHERE product_id = $1 AND is_active = true 
       ORDER BY created_at DESC LIMIT 1`,
      [planLine.productId]
    );

    if (bomResult.rows.length === 0) {
      throw new Error(`No active BOM found for product ${planLine.productId}`);
    }

    bomId = bomResult.rows[0].id;
  }

  // CRITICAL FIX: Get components for this SPECIFIC BOM, not just product_id
  // First, verify the BOM exists and matches the product
  const bomCheckResult = await client.query(
    `SELECT id, product_id FROM bom.product_boms WHERE id = $1`,
    [bomId]
  );

  if (bomCheckResult.rows.length === 0) {
    throw new Error(`BOM not found: ${bomId}`);
  }

  if (bomCheckResult.rows[0].product_id !== planLine.productId) {
    throw new Error(`BOM ${bomId} does not match product ${planLine.productId}`);
  }

  // Get all components for this SPECIFIC BOM
  const componentsResult = await client.query(
    `SELECT 
      pc.id, 
      pc.generated_name,
      pc.component_type,
      pc.color,
      pc.board_type,
      pc.drilling_pattern,
      pc.edging_pattern,
      pc.quantity
     FROM bom.product_components pc
     INNER JOIN bom.product_boms pb ON pb.product_id = pc.product_id
     WHERE pb.id = $1 AND pc.product_id = $2
     ORDER BY pc.id`,
    [bomId, planLine.productId]
  );

  if (componentsResult.rows.length === 0) {
    throw new Error(`No components found for product ${planLine.productId}`);
  }

  // IDEMPOTENCY FIX: Delete existing batches for this plan line before creating new ones
  await client.query(
    'DELETE FROM production.production_component_batches WHERE plan_line_id = $1',
    [planLineId]
  );

  // Group components by batch key
  const batchGroups = new Map<string, ComponentForBatching[]>();

  for (const row of componentsResult.rows) {
    const component: ComponentForBatching = {
      id: row.id,
      generatedName: row.generated_name,
      componentType: row.component_type,
      color: row.color,
      boardType: row.board_type,
      drillingPattern: row.drilling_pattern,
      edgingPattern: row.edging_pattern,
      quantity: row.quantity || 1,
    };

    const batchKey = generateBatchKey(component);
    
    if (!batchGroups.has(batchKey)) {
      batchGroups.set(batchKey, []);
    }
    
    batchGroups.get(batchKey)!.push(component);
  }

  // Create batches for each group
  const batches: BatchResult[] = [];

  for (const [batchKey, components] of Array.from(batchGroups.entries())) {
    const batchAttrs = parseBatchKey(batchKey);
    const batchNumber = await generateBatchNumber(pool);

    // Calculate total quantity for this batch
    // quantity = sum of component quantities * plan line quantity
    const totalQuantity = components.reduce((sum: number, c: ComponentForBatching) => sum + c.quantity, 0) * planLine.quantity;

    const result = await client.query(
      `INSERT INTO production.production_component_batches 
        (batch_number, plan_line_id, color, edging_pattern, drilling_pattern, 
         board_type, component_type, quantity, status, created_at, updated_at)
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, 'waiting', CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
       RETURNING *`,
      [
        batchNumber,
        planLineId,
        batchAttrs.color,
        batchAttrs.edgingPattern,
        batchAttrs.drillingPattern,
        batchAttrs.boardType,
        batchAttrs.componentType,
        totalQuantity,
      ]
    );

    const batch: ProductionComponentBatch = {
      id: result.rows[0].id,
      batchNumber: result.rows[0].batch_number,
      planLineId: result.rows[0].plan_line_id,
      productionOrderId: result.rows[0].production_order_id,
      workOrderId: result.rows[0].work_order_id,
      color: result.rows[0].color,
      edgingPattern: result.rows[0].edging_pattern,
      drillingPattern: result.rows[0].drilling_pattern,
      boardType: result.rows[0].board_type,
      componentType: result.rows[0].component_type,
      quantity: result.rows[0].quantity,
      quantityCompleted: result.rows[0].quantity_completed,
      quantityRejected: result.rows[0].quantity_rejected,
      currentStationId: result.rows[0].current_station_id,
      status: result.rows[0].status,
      routingId: result.rows[0].routing_id,
      currentOperationSequence: result.rows[0].current_operation_sequence,
      materialConsumption: result.rows[0].material_consumption,
      notes: result.rows[0].notes,
      metadata: result.rows[0].metadata,
      createdAt: result.rows[0].created_at,
      updatedAt: result.rows[0].updated_at,
    };

    batches.push({
      batch,
      componentIds: components.map((c: ComponentForBatching) => c.id),
    });
  }

    // COMMIT transaction
    await client.query('COMMIT');
    return batches;
    
  } catch (error) {
    // ROLLBACK on error
    await client.query('ROLLBACK');
    throw error;
  } finally {
    // Release connection back to pool
    client.release();
  }
}

/**
 * Gets all batches for a plan line.
 */
export async function getBatchesForPlanLine(
  pool: Pool,
  planLineId: number
): Promise<ProductionComponentBatch[]> {
  const result = await pool.query(
    `SELECT * FROM production.production_component_batches 
     WHERE plan_line_id = $1 
     ORDER BY created_at ASC`,
    [planLineId]
  );

  return result.rows.map(row => ({
    id: row.id,
    batchNumber: row.batch_number,
    planLineId: row.plan_line_id,
    productionOrderId: row.production_order_id,
    workOrderId: row.work_order_id,
    color: row.color,
    edgingPattern: row.edging_pattern,
    drillingPattern: row.drilling_pattern,
    boardType: row.board_type,
    componentType: row.component_type,
    quantity: row.quantity,
    quantityCompleted: row.quantity_completed,
    quantityRejected: row.quantity_rejected,
    currentStationId: row.current_station_id,
    status: row.status,
    routingId: row.routing_id,
    currentOperationSequence: row.current_operation_sequence,
    materialConsumption: row.material_consumption,
    notes: row.notes,
    metadata: row.metadata,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}
