/**
 * Material Breakdown API Routes
 * 
 * Endpoints dla obliczania rozk\u0142adu materia\u0142owego formatek
 */

import { Router } from 'express';
import { Pool } from 'pg';
import { 
  calculateMaterialBreakdown, 
  saveComponentMaterialBreakdown,
  getComponentMaterialBreakdown 
} from '../../services/bom/material-breakdown';

export function materialBreakdownRoutes(pool: Pool): Router {
  const router = Router();

  /**
   * POST /api/bom/material-breakdown/calculate
   * Oblicza breakdown dla podanych wymiar\u00f3w (bez zapisywania)
   */
  router.post('/calculate', async (req, res) => {
    try {
      const { width, height, thickness, edgingPattern } = req.body;
      
      if (!width || !height) {
        return res.status(400).json({ error: 'Width and height are required' });
      }
      
      const breakdown = await calculateMaterialBreakdown(pool, {
        componentId: 0, // temporary
        width: parseFloat(width),
        height: parseFloat(height),
        thickness: parseFloat(thickness || '18'),
        edgingPattern,
      });
      
      res.json(breakdown);
    } catch (error) {
      console.error('Error calculating material breakdown:', error);
      res.status(500).json({ error: 'Failed to calculate material breakdown' });
    }
  });

  /**
   * POST /api/bom/material-breakdown/components/:id/save
   * Oblicza i zapisuje breakdown dla komponentu
   */
  router.post('/components/:id/save', async (req, res) => {
    try {
      const componentId = parseInt(req.params.id);
      
      if (isNaN(componentId)) {
        return res.status(400).json({ error: 'Invalid component ID' });
      }
      
      const breakdownId = await saveComponentMaterialBreakdown(pool, componentId);
      const breakdown = await getComponentMaterialBreakdown(pool, componentId);
      
      res.json({ 
        message: 'Material breakdown saved successfully',
        breakdownId,
        breakdown 
      });
    } catch (error) {
      console.error('Error saving component material breakdown:', error);
      res.status(500).json({ 
        error: 'Failed to save component material breakdown',
        details: error instanceof Error ? error.message : 'Unknown error'
      });
    }
  });

  /**
   * GET /api/bom/material-breakdown/components/:id
   * Pobiera breakdown dla komponentu
   */
  router.get('/components/:id', async (req, res) => {
    try {
      const componentId = parseInt(req.params.id);
      
      if (isNaN(componentId)) {
        return res.status(400).json({ error: 'Invalid component ID' });
      }
      
      const breakdown = await getComponentMaterialBreakdown(pool, componentId);
      
      if (!breakdown) {
        return res.status(404).json({ error: 'Material breakdown not found for this component' });
      }
      
      res.json(breakdown);
    } catch (error) {
      console.error('Error fetching component material breakdown:', error);
      res.status(500).json({ error: 'Failed to fetch component material breakdown' });
    }
  });

  /**
   * POST /api/bom/material-breakdown/components/:id/recalculate
   * Przelicza breakdown dla komponentu (u\u017cywaj gdy zmieni\u0142y si\u0119 wymiary lub ceny)
   */
  router.post('/components/:id/recalculate', async (req, res) => {
    try {
      const componentId = parseInt(req.params.id);
      
      if (isNaN(componentId)) {
        return res.status(400).json({ error: 'Invalid component ID' });
      }
      
      // Recalculate and save
      const breakdownId = await saveComponentMaterialBreakdown(pool, componentId);
      const breakdown = await getComponentMaterialBreakdown(pool, componentId);
      
      res.json({ 
        message: 'Material breakdown recalculated successfully',
        breakdownId,
        breakdown 
      });
    } catch (error) {
      console.error('Error recalculating component material breakdown:', error);
      res.status(500).json({ 
        error: 'Failed to recalculate component material breakdown',
        details: error instanceof Error ? error.message : 'Unknown error'
      });
    }
  });

  return router;
}
