import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { pool } from './postgres';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

interface CSVRow {
  Name: string;
  nazwGen: string;
  ROZKROJE: string;
  'MAG BEZKOL (SYNC)': string;
  Ilość: string;
  'Długość (from MAG BEZKOL (SYNC))': string;
  'Szerokość (from MAG BEZKOL (SYNC))': string;
}

async function importRozkroje() {
  console.log('🚀 Starting rozkroje import...');
  
  // Read CSV file
  const csvPath = path.join(__dirname, '..', 'attached_assets', 'ROZKROJE FORMATKI-Grid view (1)_1762944449596.csv');
  const csvContent = fs.readFileSync(csvPath, 'utf-8');
  
  // Parse CSV (simple parser - assumes comma-separated with quotes)
  const lines = csvContent.split('\n');
  const headers = lines[0].replace(/^\uFEFF/, '').split(','); // Remove BOM
  
  const rows: CSVRow[] = [];
  for (let i = 1; i < lines.length; i++) {
    if (!lines[i].trim()) continue;
    
    // Simple CSV parser - doesn't handle complex quoted values perfectly
    // but good enough for this data
    const values = lines[i].split(',');
    const row: any = {};
    headers.forEach((header, index) => {
      row[header.trim()] = values[index]?.trim().replace(/^"/, '').replace(/"$/, '') || '';
    });
    
    if (row['Name'] && row['nazwGen (from ROZKROJE)']) {
      rows.push({
        Name: row['Name'],
        nazwGen: row['nazwGen (from ROZKROJE)'],
        ROZKROJE: row['ROZKROJE'],
        'MAG BEZKOL (SYNC)': row['MAG BEZKOL (SYNC)'],
        Ilość: row['Ilość'],
        'Długość (from MAG BEZKOL (SYNC))': row['Długość (from MAG BEZKOL (SYNC))'],
        'Szerokość (from MAG BEZKOL (SYNC))': row['Szerokość (from MAG BEZKOL (SYNC))']
      });
    }
  }
  
  console.log(`📊 Parsed ${rows.length} rows from CSV`);
  
  // Group by nazwGen (rozkroj code)
  const grouped = new Map<string, CSVRow[]>();
  for (const row of rows) {
    if (!grouped.has(row.nazwGen)) {
      grouped.set(row.nazwGen, []);
    }
    grouped.get(row.nazwGen)!.push(row);
  }
  
  console.log(`📦 Found ${grouped.size} unique rozkroje`);
  
  let patternsCreated = 0;
  let itemsCreated = 0;
  
  // Create cut patterns and items
  for (const [rozkrojCode, items] of Array.from(grouped.entries())) {
    // Get board dimensions from first item
    const firstItem = items[0];
    const length = parseFloat(firstItem['Długość (from MAG BEZKOL (SYNC))']) || 2800;
    const width = parseFloat(firstItem['Szerokość (from MAG BEZKOL (SYNC))']) || 2070;
    
    // Create cut pattern
    const patternResult = await pool.query(`
      INSERT INTO production.cut_patterns (
        code,
        board_length,
        board_width,
        board_thickness,
        kerf,
        status,
        notes,
        is_active
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
      RETURNING id
    `, [
      rozkrojCode,
      length || 2800,
      width || 2070,
      18,
      4,
      'planowany',
      `Import z CSV - formatki: ${firstItem.ROZKROJE}`,
      true
    ]);
    
    const patternId = patternResult.rows[0].id;
    patternsCreated++;
    
    console.log(`✅ Created pattern ${rozkrojCode} (ID: ${patternId}) - ${items.length} items`);
    
    // Create items for this pattern
    for (const item of items) {
      const itemLength = parseFloat(item['Długość (from MAG BEZKOL (SYNC))']);
      const itemWidth = parseFloat(item['Szerokość (from MAG BEZKOL (SYNC))']);
      const quantity = parseInt(item['Ilość']) || 0;
      
      if (!itemLength || !itemWidth) {
        console.log(`⚠️  Skipping item ${item.Name} - missing dimensions`);
        continue;
      }
      
      await pool.query(`
        INSERT INTO production.cut_pattern_items (
          pattern_id,
          item_name,
          length,
          width,
          thickness,
          quantity_requested,
          status,
          is_active
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
      `, [
        patternId,
        item['MAG BEZKOL (SYNC)'] || item.Name,
        itemLength,
        itemWidth,
        18,
        quantity,
        'zaplanowana',
        true
      ]);
      
      itemsCreated++;
    }
  }
  
  console.log(`\n✨ Import completed successfully!`);
  console.log(`📦 Created ${patternsCreated} cut patterns`);
  console.log(`📋 Created ${itemsCreated} cut pattern items`);
}

// Run import
importRozkroje()
  .then(() => {
    console.log('✅ Import finished');
    process.exit(0);
  })
  .catch((error) => {
    console.error('❌ Import failed:', error);
    process.exit(1);
  });
