import { useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useLocation } from "wouter";
import { ArrowLeft } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import { ProductionOrderForm, orderFormSchema, type OrderFormValues } from "@/features/production-orders/production-order-form";

export default function ProductionOrderNew() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  const form = useForm<OrderFormValues>({
    resolver: zodResolver(orderFormSchema),
    defaultValues: {
      orderNumber: "",
      productId: 0,
      quantityPlanned: 1,
      unitOfMeasure: "szt",
      status: "draft",
      priority: "normal",
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: OrderFormValues) => {
      const response = await fetch("/api/production/orders", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify(data),
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to create order" }));
        throw new Error(error.message);
      }
      return await response.json();
    },
    onSuccess: (createdOrder) => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/orders"] });
      toast({
        title: "Sukces",
        description: "Zlecenie produkcyjne zostało utworzone",
      });
      setLocation(`/production/orders/${createdOrder.id}`);
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć zlecenia",
      });
    },
  });

  const handleSubmit = (data: OrderFormValues) => {
    createMutation.mutate(data);
  };

  const handleCancel = () => {
    setLocation("/production/orders");
  };

  return (
    <div className="container mx-auto p-6 max-w-4xl">
      <div className="mb-6">
        <Button
          variant="ghost"
          onClick={handleCancel}
          className="mb-4"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Powrót do listy zleceń
        </Button>
        <h1 className="text-3xl font-bold" data-testid="text-page-title">Nowe Zlecenie Produkcyjne</h1>
        <p className="text-muted-foreground mt-2">Utwórz nowe zlecenie produkcyjne (ZLP)</p>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane Zlecenia</CardTitle>
          <CardDescription>Wypełnij formularz aby utworzyć nowe zlecenie produkcyjne</CardDescription>
        </CardHeader>
        <CardContent>
          <ProductionOrderForm
            form={form}
            onSubmit={handleSubmit}
            onCancel={handleCancel}
            isLoading={createMutation.isPending}
            mode="create"
          />
        </CardContent>
      </Card>
    </div>
  );
}
