import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Upload, Search, Filter, Download } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";

export default function ComponentTemplatesPage() {
  const { toast } = useToast();
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedFurnitureType, setSelectedFurnitureType] = useState<string>("all");
  const [csvFile, setCsvFile] = useState<File | null>(null);
  const [isUploading, setIsUploading] = useState(false);

  const { data: furnitureTypes } = useQuery<any[]>({
    queryKey: ['/api/component-templates/furniture-types'],
  });

  const { data: templatesData, isLoading: isLoadingTemplates } = useQuery({
    queryKey: ['/api/component-templates', { 
      furnitureType: selectedFurnitureType !== "all" ? selectedFurnitureType : undefined,
      search: searchTerm || undefined,
      limit: 100,
      offset: 0
    }],
    enabled: true,
  });

  const importMutation = useMutation({
    mutationFn: async (csvData: string) => {
      const response = await fetch('/api/component-templates/import-csv', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ csvData }),
        credentials: 'include'
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Import failed');
      }
      return response.json();
    },
    onSuccess: (data: any) => {
      toast({
        title: "Import zakończony",
        description: `Zaimportowano ${data.imported} szablonów, pominięto ${data.skipped}`,
      });
      queryClient.invalidateQueries({ queryKey: ['/api/component-templates'] });
      queryClient.invalidateQueries({ queryKey: ['/api/component-templates/furniture-types'] });
      setCsvFile(null);
      setIsUploading(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd importu",
        description: error.message || "Nie udało się zaimportować szablonów",
        variant: "destructive",
      });
      setIsUploading(false);
    },
  });

  const handleFileUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    if (!file.name.endsWith('.csv')) {
      toast({
        title: "Nieprawidłowy plik",
        description: "Proszę wybrać plik CSV",
        variant: "destructive",
      });
      return;
    }

    setCsvFile(file);
  };

  const handleImport = async () => {
    if (!csvFile) return;

    setIsUploading(true);

    try {
      const reader = new FileReader();
      reader.onload = async (event) => {
        const csvData = event.target?.result as string;
        await importMutation.mutateAsync(csvData);
      };
      reader.onerror = () => {
        toast({
          title: "Błąd odczytu pliku",
          description: "Nie udało się odczytać pliku CSV",
          variant: "destructive",
        });
        setIsUploading(false);
      };
      reader.readAsText(csvFile);
    } catch (error) {
      console.error("File upload error:", error);
      setIsUploading(false);
    }
  };

  const templates = (templatesData as any)?.templates || [];
  const total = (templatesData as any)?.total || 0;

  return (
    <div className="flex flex-col gap-6 p-6">
      <div className="flex flex-col gap-2">
        <h1 className="text-3xl font-bold" data-testid="text-page-title">
          Szablony Formatek
        </h1>
        <p className="text-muted-foreground">
          Zarządzaj matrycą formatek importowaną z Airtable
        </p>
      </div>

      <Tabs defaultValue="browse" className="w-full">
        <TabsList>
          <TabsTrigger value="browse" data-testid="tab-browse">
            Przeglądaj szablony
          </TabsTrigger>
          <TabsTrigger value="import" data-testid="tab-import">
            Import CSV
          </TabsTrigger>
        </TabsList>

        <TabsContent value="import" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Import matrycy formatek z Airtable</CardTitle>
              <CardDescription>
                Wybierz plik CSV wyeksportowany z Airtable (widok Grid)
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="csv-file">Plik CSV</Label>
                <Input
                  id="csv-file"
                  type="file"
                  accept=".csv"
                  onChange={handleFileUpload}
                  disabled={isUploading}
                  data-testid="input-csv-file"
                />
              </div>

              {csvFile && (
                <div className="flex items-center gap-2">
                  <Badge variant="outline" data-testid="badge-file-selected">
                    {csvFile.name}
                  </Badge>
                  <span className="text-sm text-muted-foreground">
                    ({(csvFile.size / 1024).toFixed(2)} KB)
                  </span>
                </div>
              )}

              <Button
                onClick={handleImport}
                disabled={!csvFile || isUploading}
                data-testid="button-import-csv"
              >
                <Upload className="mr-2 h-4 w-4" />
                {isUploading ? "Importowanie..." : "Importuj CSV"}
              </Button>

              <div className="mt-4 p-4 bg-muted rounded-lg">
                <h4 className="font-medium mb-2">Wymagane kolumny w pliku CSV:</h4>
                <ul className="text-sm text-muted-foreground space-y-1 list-disc list-inside">
                  <li>cz1, cz2 - Identyfikatory komponentu</li>
                  <li>Rodzaj / rodzajText - Typ mebla</li>
                  <li>Długość 50, Szerokość 30 - Bazowe wymiary</li>
                  <li>Grubość - Grubość płyty</li>
                  <li>Kolor, Oklejanie, Obrzeze1 - Specyfikacja materiałowa</li>
                  <li>Długość Mebla, Szerokość Mebla - Warunki zastosowania</li>
                </ul>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="browse" className="space-y-4">
          <Card>
            <CardHeader>
              <div className="flex flex-col gap-4">
                <div className="flex items-center justify-between gap-4">
                  <div className="flex-1 flex gap-2">
                    <div className="relative flex-1 max-w-sm">
                      <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
                      <Input
                        placeholder="Szukaj po cz1, cz2, kategorii..."
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                        className="pl-8"
                        data-testid="input-search-templates"
                      />
                    </div>
                    <Select value={selectedFurnitureType} onValueChange={setSelectedFurnitureType}>
                      <SelectTrigger className="w-[200px]" data-testid="select-furniture-type">
                        <SelectValue placeholder="Typ mebla" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all" data-testid="option-all-types">
                          Wszystkie typy
                        </SelectItem>
                        {furnitureTypes?.map((type: any) => (
                          <SelectItem key={type.furniture_type} value={type.furniture_type}>
                            {type.furniture_type} ({type.template_count})
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <Badge variant="outline" data-testid="badge-total-templates">
                    {total} szablonów
                  </Badge>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              {isLoadingTemplates ? (
                <div className="text-center py-8 text-muted-foreground">
                  Ładowanie szablonów...
                </div>
              ) : templates.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  Brak szablonów. Zaimportuj plik CSV z Airtable.
                </div>
              ) : (
                <div className="rounded-md border">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>cz1</TableHead>
                        <TableHead>cz2</TableHead>
                        <TableHead>Typ mebla</TableHead>
                        <TableHead>Wymiary bazowe</TableHead>
                        <TableHead>Grubość</TableHead>
                        <TableHead>Kolor</TableHead>
                        <TableHead>Oklejanie</TableHead>
                        <TableHead>Warunki</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {templates.map((template: any) => (
                        <TableRow key={template.id} data-testid={`row-template-${template.id}`}>
                          <TableCell className="font-medium">{template.cz1}</TableCell>
                          <TableCell>{template.cz2 || "-"}</TableCell>
                          <TableCell>
                            <Badge variant="secondary">{template.furniture_type}</Badge>
                          </TableCell>
                          <TableCell>
                            {template.base_length && template.base_width
                              ? `${template.base_length} × ${template.base_width}`
                              : "-"}
                          </TableCell>
                          <TableCell>{template.thickness || "-"}</TableCell>
                          <TableCell>{template.color || "-"}</TableCell>
                          <TableCell className="text-xs">{template.edging_pattern || "-"}</TableCell>
                          <TableCell className="text-xs">
                            {template.furniture_length_condition && template.furniture_width_condition
                              ? `${template.furniture_length_condition}×${template.furniture_width_condition}`
                              : template.furniture_length_condition
                              ? `D:${template.furniture_length_condition}`
                              : template.furniture_width_condition
                              ? `S:${template.furniture_width_condition}`
                              : "-"}
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
