import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Calendar } from "@/components/ui/calendar";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { ArrowRightLeft, Calendar as CalendarIcon, Filter, Loader2, MapPin, Package2, QrCode, X } from "lucide-react";
import type { ProductionLocation, ProductionCarrier } from "@shared/schema";

interface ProductionMaterialMovement {
  id: number;
  movementBatchId: string | null;
  materialType: string;
  materialCode: string;
  materialName: string | null;
  quantity: number;
  quantityRejected: number;
  unit: string;
  sourceLocationId: number | null;
  targetLocationId: number | null;
  carrierId: number | null;
  productionOrderId: number | null;
  workOrderId: number | null;
  workCenterId: number | null;
  routingOperationId: number | null;
  operatorId: number | null;
  scannedBarcode: string | null;
  scanMethod: string | null;
  status: string;
  movementType: string | null;
  movementDate: string;
  notes: string | null;
  metadata: any;
  createdAt: string;
  updatedAt: string;
}

const statusLabels: Record<string, string> = {
  planned: "Zaplanowany",
  in_progress: "W trakcie",
  completed: "Zakończony",
  cancelled: "Anulowany",
};

const statusColors: Record<string, string> = {
  planned: "bg-blue-500",
  in_progress: "bg-yellow-500",
  completed: "bg-green-500",
  cancelled: "bg-red-500",
};

export default function ProductionMovementsPage() {
  const [currentLocation, setLocation] = useLocation();
  
  // Initialize state from URL query params
  const initialParams = new URLSearchParams(window.location.search);
  const [materialCode, setMaterialCode] = useState(initialParams.get("materialCode") || "");
  const [locationId, setLocationId] = useState<string | undefined>(initialParams.get("locationId") || undefined);
  const [carrierId, setCarrierId] = useState<string | undefined>(initialParams.get("carrierId") || undefined);
  const [status, setStatus] = useState<string | undefined>(initialParams.get("status") || undefined);
  const [startDate, setStartDate] = useState<Date | undefined>(
    initialParams.get("startDate") ? new Date(initialParams.get("startDate")!) : undefined
  );
  const [endDate, setEndDate] = useState<Date | undefined>(
    initialParams.get("endDate") ? new Date(initialParams.get("endDate")!) : undefined
  );

  // Sync URL → state (for back/forward navigation)
  useEffect(() => {
    const params = new URLSearchParams(window.location.search);
    setMaterialCode(params.get("materialCode") || "");
    setLocationId(params.get("locationId") || undefined);
    setCarrierId(params.get("carrierId") || undefined);
    setStatus(params.get("status") || undefined);
    setStartDate(params.get("startDate") ? new Date(params.get("startDate")!) : undefined);
    setEndDate(params.get("endDate") ? new Date(params.get("endDate")!) : undefined);
  }, [currentLocation]); // Re-run when location changes (back/forward)

  // Sync state → URL (for filter changes)
  useEffect(() => {
    // Start with current URL params to preserve unrelated params
    const params = new URLSearchParams(window.location.search);
    
    // Update/remove filter params
    if (materialCode) {
      params.set("materialCode", materialCode);
    } else {
      params.delete("materialCode");
    }
    
    if (locationId) {
      params.set("locationId", locationId);
    } else {
      params.delete("locationId");
    }
    
    if (carrierId) {
      params.set("carrierId", carrierId);
    } else {
      params.delete("carrierId");
    }
    
    if (status) {
      params.set("status", status);
    } else {
      params.delete("status");
    }
    
    if (startDate) {
      params.set("startDate", startDate.toISOString());
    } else {
      params.delete("startDate");
    }
    
    if (endDate) {
      params.set("endDate", endDate.toISOString());
    } else {
      params.delete("endDate");
    }
    
    const newSearch = params.toString();
    const currentSearch = window.location.search.replace(/^\?/, '');
    
    // Only update if changed to avoid infinite loops
    if (newSearch !== currentSearch) {
      setLocation(`/production/movements${newSearch ? `?${newSearch}` : ''}`, { replace: true });
    }
  }, [materialCode, locationId, carrierId, status, startDate, endDate, setLocation]);

  // Build query params
  const queryParams = new URLSearchParams();
  if (materialCode) queryParams.set("materialCode", materialCode);
  if (locationId) queryParams.set("targetLocationId", locationId);
  if (carrierId) queryParams.set("carrierId", carrierId);
  if (status) queryParams.set("status", status);
  if (startDate) queryParams.set("startDate", startDate.toISOString());
  if (endDate) queryParams.set("endDate", endDate.toISOString());

  const queryString = queryParams.toString();

  // Fetch movements
  const { data: movements = [], isLoading } = useQuery<ProductionMaterialMovement[]>({
    queryKey: ["/api/production/material-movements", queryString],
    queryFn: async () => {
      const url = `/api/production/material-movements${queryString ? `?${queryString}` : ''}`;
      const res = await fetch(url);
      if (!res.ok) throw new Error("Failed to fetch movements");
      return res.json();
    },
  });

  // Fetch locations
  const { data: locations = [] } = useQuery<ProductionLocation[]>({
    queryKey: ["/api/production/locations"],
  });

  // Fetch carriers
  const { data: carriers = [] } = useQuery<ProductionCarrier[]>({
    queryKey: ["/api/production/carriers"],
  });

  const getLocationName = (id: number | null) => {
    if (!id) return '-';
    const location = locations.find(loc => loc.id === id);
    return location ? `${location.code} - ${location.name}` : '-';
  };

  const getCarrierName = (id: number | null) => {
    if (!id) return '-';
    const carrier = carriers.find(car => car.id === id);
    return carrier ? `${carrier.code}` : '-';
  };

  const clearFilters = () => {
    setMaterialCode("");
    setLocationId("");
    setCarrierId("");
    setStatus("");
    setStartDate(undefined);
    setEndDate(undefined);
  };

  const hasActiveFilters = materialCode || locationId || carrierId || status || startDate || endDate;

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold flex items-center gap-2" data-testid="text-page-title">
            <ArrowRightLeft className="h-8 w-8" />
            Ruchy Materiałów
          </h1>
          <p className="text-muted-foreground mt-1">
            Historia przepływu materiałów między lokalizacjami
          </p>
        </div>
        <div className="text-right">
          <div className="text-sm text-muted-foreground">Razem ruchów</div>
          <div className="text-2xl font-bold" data-testid="text-movements-count">
            {movements.length}
          </div>
        </div>
      </div>

      {/* Filters */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <Filter className="h-5 w-5" />
              Filtry
            </CardTitle>
            {hasActiveFilters && (
              <Button
                variant="ghost"
                size="sm"
                onClick={clearFilters}
                data-testid="button-clear-filters"
              >
                <X className="h-4 w-4 mr-2" />
                Wyczyść
              </Button>
            )}
          </div>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-3 lg:grid-cols-5">
            <div className="space-y-2">
              <Label htmlFor="filterMaterialCode">Kod materiału</Label>
              <Input
                id="filterMaterialCode"
                value={materialCode}
                onChange={(e) => setMaterialCode(e.target.value)}
                placeholder="Wpisz kod"
                data-testid="input-filter-material-code"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="filterLocation">Lokalizacja</Label>
              <Select value={locationId} onValueChange={(value) => setLocationId(value === "all" ? undefined : value)}>
                <SelectTrigger id="filterLocation" data-testid="select-filter-location">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  {locations.map((location) => (
                    <SelectItem key={location.id} value={location.id.toString()}>
                      {location.code} - {location.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="filterCarrier">Nośnik</Label>
              <Select value={carrierId} onValueChange={(value) => setCarrierId(value === "all" ? undefined : value)}>
                <SelectTrigger id="filterCarrier" data-testid="select-filter-carrier">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  {carriers.map((carrier) => (
                    <SelectItem key={carrier.id} value={carrier.id.toString()}>
                      {carrier.code} - {carrier.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="filterStatus">Status</Label>
              <Select value={status} onValueChange={(value) => setStatus(value === "all" ? undefined : value)}>
                <SelectTrigger id="filterStatus" data-testid="select-filter-status">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="planned">Zaplanowany</SelectItem>
                  <SelectItem value="in_progress">W trakcie</SelectItem>
                  <SelectItem value="completed">Zakończony</SelectItem>
                  <SelectItem value="cancelled">Anulowany</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label>Zakres dat</Label>
              <div className="flex gap-2">
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      variant="outline"
                      size="sm"
                      className="flex-1"
                      data-testid="button-filter-start-date"
                    >
                      <CalendarIcon className="h-4 w-4 mr-2" />
                      {startDate ? format(startDate, "dd.MM", { locale: pl }) : "Od"}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0">
                    <Calendar
                      mode="single"
                      selected={startDate}
                      onSelect={setStartDate}
                      locale={pl}
                    />
                  </PopoverContent>
                </Popover>
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      variant="outline"
                      size="sm"
                      className="flex-1"
                      data-testid="button-filter-end-date"
                    >
                      <CalendarIcon className="h-4 w-4 mr-2" />
                      {endDate ? format(endDate, "dd.MM", { locale: pl }) : "Do"}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0">
                    <Calendar
                      mode="single"
                      selected={endDate}
                      onSelect={setEndDate}
                      locale={pl}
                    />
                  </PopoverContent>
                </Popover>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Movements Table */}
      <Card>
        <CardHeader>
          <CardTitle>Lista ruchów</CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex justify-center p-8">
              <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-movements" />
            </div>
          ) : movements.length === 0 ? (
            <div className="text-center text-muted-foreground py-8">
              Brak ruchów materiałów
            </div>
          ) : (
            <div className="rounded-md border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Data</TableHead>
                    <TableHead>Kod materiału</TableHead>
                    <TableHead>Nazwa</TableHead>
                    <TableHead className="text-right">Ilość</TableHead>
                    <TableHead>Źródło</TableHead>
                    <TableHead>→</TableHead>
                    <TableHead>Cel</TableHead>
                    <TableHead>Nośnik</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Kod QR</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {movements.map((movement) => (
                    <TableRow key={movement.id} data-testid={`movement-row-${movement.id}`}>
                      <TableCell className="font-mono text-sm">
                        {format(new Date(movement.movementDate), "dd.MM.yyyy HH:mm", { locale: pl })}
                      </TableCell>
                      <TableCell className="font-mono font-semibold">
                        {movement.materialCode}
                      </TableCell>
                      <TableCell className="max-w-[200px] truncate">
                        {movement.materialName || '-'}
                      </TableCell>
                      <TableCell className="text-right font-mono">
                        {movement.quantity} {movement.unit}
                        {movement.quantityRejected > 0 && (
                          <span className="text-destructive ml-2">
                            (-{movement.quantityRejected})
                          </span>
                        )}
                      </TableCell>
                      <TableCell className="text-sm">
                        {movement.sourceLocationId ? (
                          <div className="flex items-center gap-1">
                            <MapPin className="h-3 w-3" />
                            {getLocationName(movement.sourceLocationId)}
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className="text-center">
                        <ArrowRightLeft className="h-4 w-4 text-muted-foreground mx-auto" />
                      </TableCell>
                      <TableCell className="text-sm">
                        {movement.targetLocationId ? (
                          <div className="flex items-center gap-1">
                            <MapPin className="h-3 w-3" />
                            {getLocationName(movement.targetLocationId)}
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className="text-sm">
                        {movement.carrierId ? (
                          <div className="flex items-center gap-1">
                            <Package2 className="h-3 w-3" />
                            <span className="font-mono">{getCarrierName(movement.carrierId)}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <Badge
                          className={statusColors[movement.status] || "bg-gray-500"}
                          data-testid={`status-${movement.id}`}
                        >
                          {statusLabels[movement.status] || movement.status}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-sm">
                        {movement.scannedBarcode ? (
                          <div className="flex items-center gap-1 font-mono">
                            <QrCode className="h-3 w-3" />
                            {movement.scannedBarcode}
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
