import { useQuery } from "@tanstack/react-query";
import { Banknote, CreditCard, Wallet } from "lucide-react";

interface PaymentSummaryBarProps {
  searchQuery?: string;
  paymentFilter?: string;
  fulfillmentFilter?: string;
  invoiceFilter?: string;
  paymentTypeFilter?: string;
  sourceFilter?: string;
  dateRangeType?: string;
  customDateFrom?: string;
  customDateTo?: string;
}

interface PaymentSummary {
  paid: { amount: number; count: number };
  unpaid: { amount: number; count: number };
  cod: { amount: number; count: number };
}

export function PaymentSummaryBar({
  searchQuery,
  paymentFilter,
  fulfillmentFilter,
  invoiceFilter,
  paymentTypeFilter,
  sourceFilter,
  dateRangeType,
  customDateFrom,
  customDateTo,
}: PaymentSummaryBarProps) {
  const params = new URLSearchParams();
  if (searchQuery) params.append('search', searchQuery);
  if (paymentFilter && paymentFilter !== 'all' && paymentFilter !== 'Wszystkie') params.append('paymentFilter', paymentFilter);
  if (fulfillmentFilter && fulfillmentFilter !== 'all' && fulfillmentFilter !== 'Wszystkie') params.append('fulfillmentFilter', fulfillmentFilter);
  if (invoiceFilter && invoiceFilter !== 'all' && invoiceFilter !== 'Wszystkie') params.append('invoiceFilter', invoiceFilter);
  if (paymentTypeFilter && paymentTypeFilter !== 'all' && paymentTypeFilter !== 'Wszystkie') params.append('paymentTypeFilter', paymentTypeFilter);
  if (sourceFilter && sourceFilter !== 'all' && sourceFilter !== 'Wszystkie') params.append('sourceFilter', sourceFilter);
  if (dateRangeType && dateRangeType !== 'all' && dateRangeType !== 'Wszystkie') params.append('dateRangeType', dateRangeType);
  if (customDateFrom) params.append('customDateFrom', customDateFrom);
  if (customDateTo) params.append('customDateTo', customDateTo);

  const { data: summary } = useQuery<PaymentSummary>({
    queryKey: [`/api/orders/payment-summary?${params.toString()}`],
  });

  if (!summary) return null;

  const formatAmount = (amount: number) => {
    return new Intl.NumberFormat('pl-PL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount);
  };

  return (
    <div className="flex items-center gap-4 py-2 px-4 bg-muted/30 rounded-md border text-sm" data-testid="payment-summary-bar">
      <div className="flex items-center gap-2">
        <div className="flex items-center gap-1.5 text-green-600 dark:text-green-400" data-testid="summary-paid">
          <CreditCard className="h-4 w-4" />
          <span className="font-medium">
            {formatAmount(summary.paid.amount)} PLN
          </span>
          <span className="text-muted-foreground">
            ({summary.paid.count})
          </span>
        </div>
      </div>

      <div className="h-4 w-px bg-border" />

      <div className="flex items-center gap-2">
        <div className="flex items-center gap-1.5 text-red-600 dark:text-red-400" data-testid="summary-unpaid">
          <Banknote className="h-4 w-4" />
          <span className="font-medium">
            {formatAmount(summary.unpaid.amount)} PLN
          </span>
          <span className="text-muted-foreground">
            ({summary.unpaid.count})
          </span>
        </div>
      </div>

      <div className="h-4 w-px bg-border" />

      <div className="flex items-center gap-2">
        <div className="flex items-center gap-1.5 text-orange-600 dark:text-orange-400" data-testid="summary-cod">
          <Wallet className="h-4 w-4" />
          <span className="font-medium">
            {formatAmount(summary.cod.amount)} PLN
          </span>
          <span className="text-muted-foreground">
            ({summary.cod.count})
          </span>
        </div>
      </div>
    </div>
  );
}
