import { Pool, PoolClient } from "pg";

type PoolOrClient = Pool | PoolClient;

/**
 * Reprezentuje formatkę znalezioną w magazynie (stock_panels)
 */
export interface FormatkaSearchResult {
  id: number;
  name: string;
  internalCode: string;
  length: number;
  width: number;
  thickness: number;
  color: string | null;
  cz1: string | null;
  cz2: string | null;
  furnitureType: string | null;
  quantityAvailable: number;
  quantityReserved: number;
  quantityTotal: number;
  unitOfMeasure: string;
}

/**
 * Wyszukuje formatkę w magazynie (warehouse.stock_panels) zgodnie z logiką:
 * 1. Najpierw szuka dedykowanej formatki (pełna nazwa z BOM, np. WD-VB-500x300-N1-WOTAN)
 * 2. Jeśli nie znajdzie, szuka uniwersalnej formatki (wymiary + kolor, np. SIEDZISKO-VB-500x300-SUROWA)
 * 
 * UWAGA: Formatki są przechowywane w warehouse.stock_panels, NIE w warehouse.materials
 * 
 * @param pool - Pool lub Client połączenia z bazą danych
 * @param generatedName - Wygenerowana nazwa komponentu z BOM (np. "WD-VB-596x337-HDF-BIALY")
 * @param length - Długość formatki (mm)
 * @param width - Szerokość formatki (mm)
 * @param thickness - Grubość formatki (mm)
 * @param color - Kod koloru (np. "WOTAN", "BIALY")
 * @returns Formatka jeśli znaleziona, null w przeciwnym wypadku
 */
export async function findFormatkaInWarehouse(
  pool: PoolOrClient,
  generatedName: string,
  length: number,
  width: number,
  thickness: number | null,
  color: string | null
): Promise<FormatkaSearchResult | null> {
  // Wymiary w stock_panels są w mm (nie w cm jak w materials)
  const lengthMm = length;
  const widthMm = width;

  // KROK 1: Szukaj dedykowanej formatki po pełnej nazwie (generated_name z BOM)
  // Szukamy w warehouse.stock_panels
  const dedicatedResult = await pool.query(`
    SELECT 
      sp.id,
      sp.generated_name as name,
      sp.generated_name as internal_code,
      COALESCE(sp.length, 0) as length,
      COALESCE(sp.width, 0) as width,
      COALESCE(sp.thickness, 0) as thickness,
      sp.color_code as color,
      sp.cz1,
      sp.cz2,
      sp.furniture_type,
      COALESCE(sp.quantity, 0) as quantity_total,
      COALESCE(SUM(CASE 
        WHEN r.status = 'ACTIVE' 
        THEN CAST(r.quantity_reserved AS decimal) - COALESCE(CAST(r.quantity_consumed AS decimal), 0)
        ELSE 0 
      END), 0) as quantity_reserved,
      COALESCE(sp.unit, 'szt') as unit_of_measure
    FROM warehouse.stock_panels sp
    LEFT JOIN production.production_buffer_reservations r 
      ON r.product_sku = sp.generated_name AND r.status = 'ACTIVE'
    WHERE sp.is_active = true
      AND sp.generated_name ILIKE $1
    GROUP BY sp.id, sp.generated_name, sp.length, sp.width, sp.thickness, 
             sp.color_code, sp.cz1, sp.cz2, sp.furniture_type, sp.quantity, sp.unit
    LIMIT 1
  `, [`%${generatedName}%`]);

  if (dedicatedResult.rows.length > 0) {
    const row = dedicatedResult.rows[0];
    const quantityTotal = parseFloat(row.quantity_total);
    const quantityReserved = parseFloat(row.quantity_reserved);
    
    return {
      id: row.id,
      name: row.name,
      internalCode: row.internal_code,
      length: parseFloat(row.length),
      width: parseFloat(row.width),
      thickness: parseFloat(row.thickness),
      color: row.color,
      cz1: row.cz1,
      cz2: row.cz2,
      furnitureType: row.furniture_type,
      quantityTotal,
      quantityReserved,
      quantityAvailable: quantityTotal - quantityReserved,
      unitOfMeasure: row.unit_of_measure || 'szt',
    };
  }

  // KROK 2: Szukaj uniwersalnej formatki (wymiary + kolor) - bez dedykowanego cz1/cz2
  // Np. SIEDZISKO-VB-500x300-SUROWA pasuje do dowolnej formatki 500x300 w kolorze SUROWA
  if (!color) {
    // Jeśli nie ma koloru, nie można szukać uniwersalnej formatki
    return null;
  }

  const universalResult = await pool.query(`
    SELECT 
      sp.id,
      sp.generated_name as name,
      sp.generated_name as internal_code,
      COALESCE(sp.length, 0) as length,
      COALESCE(sp.width, 0) as width,
      COALESCE(sp.thickness, 0) as thickness,
      sp.color_code as color,
      sp.cz1,
      sp.cz2,
      sp.furniture_type,
      COALESCE(sp.quantity, 0) as quantity_total,
      COALESCE(SUM(CASE 
        WHEN r.status = 'ACTIVE' 
        THEN CAST(r.quantity_reserved AS decimal) - COALESCE(CAST(r.quantity_consumed AS decimal), 0)
        ELSE 0 
      END), 0) as quantity_reserved,
      COALESCE(sp.unit, 'szt') as unit_of_measure
    FROM warehouse.stock_panels sp
    LEFT JOIN production.production_buffer_reservations r 
      ON r.product_sku = sp.generated_name AND r.status = 'ACTIVE'
    WHERE sp.is_active = true
      AND ABS(COALESCE(sp.length, 0) - $1) < 1
      AND ABS(COALESCE(sp.width, 0) - $2) < 1
      AND (
        ($3::decimal IS NULL) OR 
        ABS(COALESCE(sp.thickness, 0) - $3) < 1
      )
      AND (
        sp.color_code = $4 OR 
        sp.color_code ILIKE $4 || '%' OR
        sp.generated_name ILIKE '%' || $4
      )
      AND (sp.cz1 IS NULL OR sp.cz1 = '')
      AND (sp.cz2 IS NULL OR sp.cz2 = '')
    GROUP BY sp.id, sp.generated_name, sp.length, sp.width, sp.thickness, 
             sp.color_code, sp.cz1, sp.cz2, sp.furniture_type, sp.quantity, sp.unit
    ORDER BY (COALESCE(sp.quantity, 0) - COALESCE(SUM(CASE 
      WHEN r.status = 'ACTIVE' 
      THEN CAST(r.quantity_reserved AS decimal) - COALESCE(CAST(r.quantity_consumed AS decimal), 0)
      ELSE 0 
    END), 0)) DESC
    LIMIT 1
  `, [lengthMm, widthMm, thickness, color]);

  if (universalResult.rows.length > 0) {
    const row = universalResult.rows[0];
    const quantityTotal = parseFloat(row.quantity_total);
    const quantityReserved = parseFloat(row.quantity_reserved);
    
    return {
      id: row.id,
      name: row.name,
      internalCode: row.internal_code,
      length: parseFloat(row.length),
      width: parseFloat(row.width),
      thickness: parseFloat(row.thickness),
      color: row.color,
      cz1: row.cz1,
      cz2: row.cz2,
      furnitureType: row.furniture_type,
      quantityTotal,
      quantityReserved,
      quantityAvailable: quantityTotal - quantityReserved,
      unitOfMeasure: row.unit_of_measure || 'szt',
    };
  }

  // Nie znaleziono żadnej formatki
  return null;
}

/**
 * Pobiera dane o rezerwacjach dla konkretnej formatki z stock_panels
 */
export async function getFormatkaReservationStats(
  pool: PoolOrClient,
  generatedName: string
): Promise<{ total: number; reserved: number; available: number }> {
  const result = await pool.query(`
    SELECT 
      COALESCE(sp.quantity, 0) as total_quantity,
      COALESCE(SUM(CASE 
        WHEN r.status = 'ACTIVE' 
        THEN CAST(r.quantity_reserved AS decimal) - COALESCE(CAST(r.quantity_consumed AS decimal), 0)
        ELSE 0 
      END), 0) as reserved_quantity
    FROM warehouse.stock_panels sp
    LEFT JOIN production.production_buffer_reservations r 
      ON r.product_sku = sp.generated_name AND r.status = 'ACTIVE'
    WHERE sp.generated_name = $1
    GROUP BY sp.quantity
  `, [generatedName]);

  if (result.rows.length === 0) {
    return { total: 0, reserved: 0, available: 0 };
  }

  const row = result.rows[0];
  const total = parseFloat(row.total_quantity);
  const reserved = parseFloat(row.reserved_quantity);

  return {
    total,
    reserved,
    available: total - reserved,
  };
}

/*
 * STARA LOGIKA (zakomentowana) - szukanie w warehouse.materials
 * Formatki w naszym systemie są przechowywane w warehouse.stock_panels,
 * nie w warehouse.materials (tam są inne materiały jak płyty, obrzeża itp.)
 * 
 * Różnice:
 * - stock_panels: dedykowane dla formatek z pełnymi danymi (cz1, cz2, furniture_type, wymiary w mm)
 * - materials: ogólne materiały z JSONB specifications
 *
 * Stara logika szukała w:
 * FROM warehouse.materials m
 * LEFT JOIN warehouse.material_groups g ON m.group_id = g.id
 * WHERE g.category = 'formatki'
 */
