import type { Express } from "express";
import { z } from "zod";
import { pool } from "../../postgres";
import * as planningService from "../../services/production/planning";
import * as batchEngine from "../../services/production/batch-engine";
import { requirePermission } from "../../auth";
import { generateProductionOrdersFromPlan } from "../../services/production/production-order-generator";

const createPlanSchema = z.object({
  planNumber: z.string().optional(),
  name: z.string().min(1),
  description: z.string().optional().nullable(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  status: z.enum(['draft', 'approved', 'in_progress', 'completed', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  notes: z.string().optional().nullable(),
  metadata: z.any().optional(),
  createdBy: z.number().optional().nullable(),
});

const createPlanLineSchema = z.object({
  planId: z.number(),
  productId: z.number(),
  quantity: z.number().positive(),
  sourceType: z.enum(['sales_order', 'forecast', 'buffer_stock', 'manual', 'order_demand', 'catalog_internal', 'cutting_pattern']).optional().nullable(),
  sourceId: z.number().optional().nullable(),
  sourceReference: z.string().optional().nullable(),
  productionOrderId: z.number().optional().nullable(),
  routingId: z.number().optional().nullable(),
  routingOverride: z.any().optional(),
  bomId: z.number().optional().nullable(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  status: z.enum(['pending', 'scheduled', 'in_progress', 'completed', 'cancelled']).optional(),
  sequence: z.number().optional().nullable(),
  notes: z.string().optional().nullable(),
  metadata: z.any().optional(),
});

const filtersSchema = z.object({
  status: z.string().optional(),
  priority: z.string().optional(),
  createdBy: z.coerce.number().optional(),
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  search: z.string().optional(),
  limit: z.coerce.number().optional(),
  offset: z.coerce.number().optional(),
});

const demandFiltersSchema = z.object({
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  marketplace: z.string().optional(),
  orderStatus: z.string().optional(),
  paymentStatus: z.string().optional(),
});

const availableOrdersFiltersSchema = z.object({
  search: z.string().optional(),
  color: z.string().optional(),
  sku: z.string().optional(),
  minLength: z.coerce.number().optional(),
  maxLength: z.coerce.number().optional(),
  minWidth: z.coerce.number().optional(),
  maxWidth: z.coerce.number().optional(),
  orderNumber: z.string().optional(),
  customerName: z.string().optional(),
  marketplace: z.string().optional(),
  showSetsOnly: z.string().optional(), // 'true' or 'false'
  showCatalogLinked: z.string().optional(), // 'true' or 'false'
  showInPlans: z.string().optional(), // 'true' or 'false' - filter items already in any plan
  dateFilter: z.enum(['all', 'today', 'yesterday', 'day-before', 'custom-days']).optional().default('all'),
  customDays: z.coerce.number().optional(), // Number of days ago for custom-days filter
  limit: z.coerce.number().min(1).max(10000).optional().default(100), // Increased max to 10000 for full data loading
  offset: z.coerce.number().min(0).optional().default(0),
  sortBy: z.enum(['order_date', 'order_number', 'buyer_name', 'total_amount', 'product_sku']).optional().default('order_date'),
  sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
});

// Helper: Parse semicolon-delimited search tokens into structured filters
interface ParsedSearchTokens {
  marketplaces: string[];
  paymentStatuses: string[];
  searchTerms: string[];
}

function parseSearchTokens(rawSearch: string | undefined): ParsedSearchTokens {
  if (!rawSearch) {
    return { marketplaces: [], paymentStatuses: [], searchTerms: [] };
  }

  // Split by semicolon, trim, uppercase, deduplicate
  const tokens = Array.from(new Set(
    rawSearch.split(';')
      .map(t => t.trim().toUpperCase())
      .filter(Boolean)
  ));

  // Whitelist recognized tokens
  const MARKETPLACE_TOKENS = ['ALLEGRO', 'SHOPER'];
  const PAYMENT_STATUS_TOKENS = ['PAID', 'UNPAID', 'PENDING'];

  const marketplaces: string[] = [];
  const paymentStatuses: string[] = [];
  const searchTerms: string[] = [];

  tokens.forEach(token => {
    if (MARKETPLACE_TOKENS.includes(token)) {
      marketplaces.push(token.toLowerCase()); // Store as lowercase for DB comparison
    } else if (PAYMENT_STATUS_TOKENS.includes(token)) {
      paymentStatuses.push(token.toLowerCase());
    } else {
      searchTerms.push(token); // Keep original case for search
    }
  });

  return { marketplaces, paymentStatuses, searchTerms };
}

export function registerPlanningRoutes(app: Express) {
  // GET /api/production/planning/plans - Get all plans with filtering
  app.get("/api/production/planning/plans", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = filtersSchema.parse(req.query);
      
      const processedFilters = {
        ...filters,
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
      };

      const plans = await planningService.getPlans(pool, processedFilters);
      res.json(plans);
    } catch (error) {
      console.error("Error fetching production plans:", error);
      res.status(500).json({ message: "Failed to fetch production plans" });
    }
  });

  // GET /api/production/planning/plans/:id - Get plan by ID
  app.get("/api/production/planning/plans/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const plan = await planningService.getPlanById(pool, id);
      if (!plan) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.json(plan);
    } catch (error) {
      console.error("Error fetching production plan:", error);
      res.status(500).json({ message: "Failed to fetch production plan" });
    }
  });

  // GET /api/production/planning/plans/:id/items - Get plan lines with full details
  app.get("/api/production/planning/plans/:id/items", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const items = await planningService.getPlanLinesWithDetails(pool, id);
      
      // DEBUG: Log how many items we're returning
      console.log(`📦 [PLAN ITEMS] Returning ${items.length} items for plan ${id}`);
      if (items.length > 0) {
        console.log(`📦 [SAMPLE] First item:`, {
          id: items[0].id,
          product_sku: items[0].product_sku,
          product_title: items[0].product_title,
          order_number: items[0].order_number
        });
      }
      
      // Disable HTTP caching and ETag to ensure fresh data
      res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private, max-age=0');
      res.setHeader('Pragma', 'no-cache');
      res.setHeader('Expires', '0');
      res.removeHeader('ETag'); // Remove Express auto-generated ETag
      
      res.json(items);
    } catch (error) {
      console.error("Error fetching plan items:", error);
      res.status(500).json({ message: "Failed to fetch plan items" });
    }
  });

  // GET /api/production/demand/aggregate - Aggregate order demand by catalog product
  app.get("/api/production/demand/aggregate", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = demandFiltersSchema.parse(req.query);
      
      const processedFilters: planningService.DemandFilters = {
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
        marketplace: filters.marketplace,
        orderStatus: filters.orderStatus ? filters.orderStatus.split(',') : undefined,
        paymentStatus: filters.paymentStatus ? filters.paymentStatus.split(',') : undefined,
      };

      const aggregatedDemand = await planningService.aggregateDemand(pool, processedFilters);
      res.json(aggregatedDemand);
    } catch (error) {
      console.error("Error aggregating demand:", error);
      res.status(500).json({ message: "Failed to aggregate demand" });
    }
  });

  // POST /api/production/planning/plans - Create new plan
  app.post("/api/production/planning/plans", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createPlanSchema.parse(req.body);
      
      const plan = await planningService.createPlan(pool, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
      });
      
      res.status(201).json(plan);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Plan with this number already exists" });
      }
      console.error("Error creating production plan:", error);
      res.status(500).json({ message: "Failed to create production plan" });
    }
  });

  // PATCH /api/production/planning/plans/:id - Update plan
  app.patch("/api/production/planning/plans/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const data = createPlanSchema.partial().parse(req.body);
      const plan = await planningService.updatePlan(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!plan) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.json(plan);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error updating production plan:", error);
      res.status(500).json({ message: "Failed to update production plan" });
    }
  });

  // DELETE /api/production/planning/plans/:id - Delete plan
  app.delete("/api/production/planning/plans/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const success = await planningService.deletePlan(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting production plan:", error);
      res.status(500).json({ message: "Failed to delete production plan" });
    }
  });

  // POST /api/production/planning/plans/:id/generate-orders - Generate Production Orders (ZLP) from plan
  app.post("/api/production/planning/plans/:id/generate-orders", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      console.log(`🏭 [ZLP GENERATION] Starting generation for plan ${planId}`);
      
      const result = await generateProductionOrdersFromPlan(planId);

      if (!result.success) {
        console.error(`❌ [ZLP GENERATION] Failed for plan ${planId}:`, result.errors);
        return res.status(400).json({ 
          message: "Failed to generate production orders", 
          errors: result.errors,
          summary: result.summary,
        });
      }

      console.log(`✅ [ZLP GENERATION] Success for plan ${planId}:`, {
        totalOrders: result.summary.totalOrders,
        totalComponents: result.summary.totalComponents,
        colorBreakdown: result.summary.colorBreakdown,
      });

      res.status(201).json(result);
    } catch (error: any) {
      console.error("❌ [ZLP GENERATION] Error:", error);
      res.status(500).json({ 
        message: "Failed to generate production orders",
        error: error.message,
      });
    }
  });

  // GET /api/production/planning/plans/:id/lines - Get all lines for a plan
  app.get("/api/production/planning/plans/:id/lines", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const lines = await planningService.getPlanLines(pool, planId);
      res.json(lines);
    } catch (error) {
      console.error("Error fetching plan lines:", error);
      res.status(500).json({ message: "Failed to fetch plan lines" });
    }
  });

  // GET /api/production/planning/lines/:id - Get plan line by ID
  app.get("/api/production/planning/lines/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const line = await planningService.getPlanLineById(pool, id);
      if (!line) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.json(line);
    } catch (error) {
      console.error("Error fetching plan line:", error);
      res.status(500).json({ message: "Failed to fetch plan line" });
    }
  });

  // POST /api/production/planning/lines - Create new plan line
  app.post("/api/production/planning/lines", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createPlanLineSchema.parse(req.body);
      
      // Check for duplicate based on metadata type
      if (data.planId) {
        let duplicateCheck;
        
        // For components from sets: check by set_id + component_id + order_number
        // Note: Same component from DIFFERENT orders = DIFFERENT products (user requirement)
        if (data.metadata?.set_id && data.metadata?.component_id) {
          if (data.metadata.order_number) {
            // New behavior: Check only rows with same order_number
            // This allows same component from different orders to be added separately
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND (metadata->>'set_id')::int = $2
               AND (metadata->>'component_id')::int = $3
               AND (metadata->>'order_number') = $4
               AND deleted_at IS NULL
               LIMIT 1`,
              [data.planId, data.metadata.set_id, data.metadata.component_id, data.metadata.order_number]
            );
          } 
          // Legacy behavior: Check only set_id + component_id (for old plan_lines without order_number)
          else {
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND (metadata->>'set_id')::int = $2
               AND (metadata->>'component_id')::int = $3
               AND deleted_at IS NULL
               LIMIT 1`,
              [data.planId, data.metadata.set_id, data.metadata.component_id]
            );
          }
        }
        // For regular order items: check by catalog_product_id + order_number
        // Note: Same product from DIFFERENT orders = DIFFERENT products (user requirement)
        // Only check for order_demand source type (allow manual/forecast/buffer to add same product)
        else if (data.productId && data.sourceType === 'order_demand') {
          if (data.metadata?.order_number) {
            // Strict matching: Check only rows with same order_number
            // This allows same product from different orders to be added separately
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND product_id = $2
               AND source_type = 'order_demand'
               AND (metadata->>'order_number') = $3
               AND deleted_at IS NULL
               LIMIT 1`,
              [data.planId, data.productId, data.metadata.order_number]
            );
          } else {
            // Fallback: Check only product_id (should not happen as all new items have order_number)
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND product_id = $2
               AND source_type = 'order_demand'
               AND deleted_at IS NULL
               LIMIT 1`,
              [data.planId, data.productId]
            );
          }
        }
        
        if (duplicateCheck && duplicateCheck.rows.length > 0) {
          return res.status(409).json({ 
            message: "Product already in the plan" 
          });
        }
      }
      
      const result = await planningService.createPlanLine(pool, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
      });
      
      // Return plan line with reservation info for frontend display
      res.status(201).json({
        ...result.planLine,
        reservationInfo: result.reservationInfo,
      });
    } catch (error: any) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if (error?.code === '23503') {
        return res.status(400).json({ message: "Invalid plan ID or product ID" });
      }
      // Handle duplicate detection error from createPlanLine
      if (error?.message?.includes('już istnieje w planie') || error?.message?.includes('already exists in plan')) {
        return res.status(409).json({ message: error.message });
      }
      console.error("Error creating plan line:", error);
      res.status(500).json({ message: "Failed to create plan line" });
    }
  });

  // PATCH /api/production/planning/lines/:id - Update plan line
  app.patch("/api/production/planning/lines/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const data = createPlanLineSchema.partial().parse(req.body);
      const line = await planningService.updatePlanLine(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!line) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.json(line);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error updating plan line:", error);
      res.status(500).json({ message: "Failed to update plan line" });
    }
  });

  // DELETE /api/production/planning/lines/:id - Delete plan line

  // POST /api/production/planning/lines/:id/reserve - Manually reserve packed product
  app.post("/api/production/planning/lines/:id/reserve", requirePermission('manage_production'), async (req, res) => {
    try {
      const lineId = parseInt(req.params.id);
      if (isNaN(lineId)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const { quantity: requestedQty } = req.body;
      
      const client = await pool.connect();
      try {
        await client.query('BEGIN');
        
        // Get line info with row lock
        const lineResult = await client.query(
          `SELECT product_id, quantity, reserved_quantity FROM production.production_plan_lines WHERE id = $1 FOR UPDATE`,
          [lineId]
        );
        
        if (lineResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(404).json({ message: "Plan line not found" });
        }
        
        const { product_id, quantity: lineQty, reserved_quantity: currentReserved } = lineResult.rows[0];
        
        // Check if it's a packed product with row lock
        const packedProductResult = await client.query(
          `SELECT id, product_sku, quantity, reserved_quantity FROM warehouse.packed_products WHERE catalog_product_id = $1 FOR UPDATE`,
          [product_id]
        );
        
        if (packedProductResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "This product is not a packed product" });
        }
        
        const packedProduct = packedProductResult.rows[0];
        const availableQty = packedProduct.quantity - packedProduct.reserved_quantity;
        
        if (availableQty <= 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "No available stock to reserve" });
        }
        
        // Calculate quantity to reserve - use requested or calculate needed amount
        const neededQty = lineQty - (currentReserved || 0);
        const qtyToReserve = Math.min(requestedQty || neededQty, availableQty, neededQty);
        
        if (qtyToReserve <= 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "Invalid quantity to reserve" });
        }
        
        // Get plan info for audit history
        const planResult = await client.query(
          `SELECT pp.id as plan_id, pp.name as plan_name 
           FROM production.production_plan_lines ppl 
           JOIN production.production_plans pp ON pp.id = ppl.plan_id 
           WHERE ppl.id = $1`,
          [lineId]
        );
        const planInfo = planResult.rows[0];
        const username = (req.user as any)?.username || 'system';
        
        // Build audit entry for reservation
        const auditEntry = {
          action: 'reserved',
          timestamp: new Date().toISOString(),
          performedBy: username,
          previousStatus: 'available',
          planId: planInfo?.plan_id,
          planLineId: lineId
        };
        
        // Reserve individual items using FIFO (oldest first) with audit history
        const reservedItemsResult = await client.query(
          `UPDATE warehouse.packed_product_items
           SET status = 'reserved',
               reserved_for_plan_line_id = $1,
               reserved_at = CURRENT_TIMESTAMP,
               updated_at = CURRENT_TIMESTAMP,
               metadata = COALESCE(metadata, '{}'::jsonb) || jsonb_build_object(
                 'auditHistory', COALESCE(metadata->'auditHistory', '[]'::jsonb) || $4::jsonb
               )
           WHERE id IN (
             SELECT id FROM warehouse.packed_product_items
             WHERE packed_product_id = $2
               AND status = 'available'
             ORDER BY packed_at ASC
             LIMIT $3
             FOR UPDATE SKIP LOCKED
           )
           RETURNING id, serial_number`,
          [lineId, packedProduct.id, qtyToReserve, JSON.stringify([auditEntry])]
        );
        
        const actualReservedCount = reservedItemsResult.rows.length;
        
        if (actualReservedCount === 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "No available items to reserve" });
        }
        
        // Update summary table
        await client.query(
          `UPDATE warehouse.packed_products 
           SET reserved_quantity = reserved_quantity + $1 
           WHERE id = $2`,
          [actualReservedCount, packedProduct.id]
        );
        
        // Update plan line reserved_quantity
        await client.query(
          `UPDATE production.production_plan_lines
           SET reserved_quantity = reserved_quantity + $1, updated_at = CURRENT_TIMESTAMP
           WHERE id = $2`,
          [actualReservedCount, lineId]
        );
        
        // Auto-zwolnij rezerwacje formatek W TEJ SAMEJ TRANSAKCJI (produkt spakowany jest zarezerwowany, formatki niepotrzebne)
        const formatkaRelease = await planningService.releaseFormatkiReservationsForPlanLine(client, lineId);
        
        await client.query('COMMIT');
        
        const serialNumbers = reservedItemsResult.rows.map((r: any) => r.serial_number);
        console.log(`📦 [RĘCZNA REZERWACJA] Zarezerwowano ${actualReservedCount} szt. produktu ${packedProduct.product_sku} dla ZLP line #${lineId}`);
        console.log(`📦 [ZAREZERWOWANE SZTUKI]: ${serialNumbers.join(', ')}`);
        
        res.json({ 
          message: "Reservation created successfully",
          productSku: packedProduct.product_sku,
          quantityReserved: actualReservedCount,
          totalReserved: (currentReserved || 0) + actualReservedCount,
          reservedSerialNumbers: serialNumbers,
          formatkiReleased: formatkaRelease.releasedCount,
          formatkiLogs: formatkaRelease.logs
        });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Error creating reservation:", error);
      res.status(500).json({ message: "Failed to create reservation" });
    }
  });

  // POST /api/production/planning/lines/:id/unreserve - Manually release packed product reservation
  app.post("/api/production/planning/lines/:id/unreserve", requirePermission('manage_production'), async (req, res) => {
    try {
      const lineId = parseInt(req.params.id);
      if (isNaN(lineId)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const client = await pool.connect();
      try {
        await client.query('BEGIN');
        
        // Get line info
        const lineResult = await client.query(
          `SELECT product_id, quantity, reserved_quantity FROM production.production_plan_lines WHERE id = $1`,
          [lineId]
        );
        
        if (lineResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(404).json({ message: "Plan line not found" });
        }
        
        const { product_id, quantity, reserved_quantity } = lineResult.rows[0];
        
        // Check if it's a packed product
        const packedProductResult = await client.query(
          `SELECT id, product_sku, reserved_quantity FROM warehouse.packed_products WHERE catalog_product_id = $1`,
          [product_id]
        );
        
        if (packedProductResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "This product is not a packed product" });
        }
        
        const packedProduct = packedProductResult.rows[0];
        
        // Get plan info for audit history
        const planResult = await client.query(
          `SELECT pp.id as plan_id, pp.name as plan_name 
           FROM production.production_plan_lines ppl 
           JOIN production.production_plans pp ON pp.id = ppl.plan_id 
           WHERE ppl.id = $1`,
          [lineId]
        );
        const planInfo = planResult.rows[0];
        const username = (req.user as any)?.username || 'system';
        
        // Build audit entry for unreservation
        const auditEntry = {
          action: 'unreserved',
          timestamp: new Date().toISOString(),
          performedBy: username,
          previousStatus: 'reserved',
          planId: planInfo?.plan_id,
          planLineId: lineId
        };
        
        // Zwolnij pojedyncze sztuki zarezerwowane dla tej linii planu z historią audytu
        const releasedItemsResult = await client.query(
          `UPDATE warehouse.packed_product_items
           SET status = 'available',
               reserved_for_plan_line_id = NULL,
               reserved_at = NULL,
               updated_at = CURRENT_TIMESTAMP,
               metadata = COALESCE(metadata, '{}'::jsonb) || jsonb_build_object(
                 'auditHistory', COALESCE(metadata->'auditHistory', '[]'::jsonb) || $2::jsonb
               )
           WHERE reserved_for_plan_line_id = $1
             AND status = 'reserved'
           RETURNING id, serial_number`,
          [lineId, JSON.stringify([auditEntry])]
        );
        
        const releasedCount = releasedItemsResult.rows.length;
        
        // Aktualizuj licznik w tabeli agregacyjnej (dla kompatybilności wstecznej)
        await client.query(
          `UPDATE warehouse.packed_products 
           SET reserved_quantity = GREATEST(0, reserved_quantity - $1) 
           WHERE id = $2`,
          [releasedCount, packedProduct.id]
        );
        
        // Zero out plan line reserved_quantity
        await client.query(
          `UPDATE production.production_plan_lines
           SET reserved_quantity = 0, updated_at = CURRENT_TIMESTAMP
           WHERE id = $1`,
          [lineId]
        );
        
        // Auto-zarezerwuj formatki z BOM W TEJ SAMEJ TRANSAKCJI (produkt spakowany zwolniony, trzeba wyprodukować z formatek)
        const userId = (req.user as any)?.id;
        const formatkaReserve = await planningService.reserveFormatkiForPlanLine(client, lineId, userId);
        
        await client.query('COMMIT');
        
        const serialNumbers = releasedItemsResult.rows.map((r: any) => r.serial_number);
        console.log(`📦 [RĘCZNE ZWOLNIENIE] Zwolniono ${releasedCount} szt. rezerwacji produktu ${packedProduct.product_sku} dla ZLP line #${lineId}`);
        console.log(`📦 [ZWOLNIONE SZTUKI]: ${serialNumbers.join(', ')}`);
        
        res.json({ 
          message: "Reservation released successfully",
          productSku: packedProduct.product_sku,
          quantityReleased: quantity,
          previousReserved: packedProduct.reserved_quantity,
          formatkiReserved: formatkaReserve.success,
          formatkiLogs: formatkaReserve.logs
        });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Error releasing reservation:", error);
      res.status(500).json({ message: "Failed to release reservation" });
    }
  });

  // GET /api/production/planning/lines/:id/bom-formatki - Get BOM formatki status for plan line
  app.get("/api/production/planning/lines/:id/bom-formatki", requirePermission('view_production'), async (req, res) => {
    try {
      const lineId = parseInt(req.params.id);
      if (isNaN(lineId)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      // Get plan line with BOM and packed product reservation status
      const lineResult = await pool.query(`
        SELECT ppl.id, ppl.product_id, ppl.bom_id, ppl.quantity as line_quantity,
               ppl.reserved_quantity as packed_reserved_qty,
               cp.sku as product_sku, cp.title as product_title,
               wpp.id as packed_product_id
        FROM production.production_plan_lines ppl
        JOIN catalog.products cp ON cp.id = ppl.product_id
        LEFT JOIN warehouse.packed_products wpp ON wpp.catalog_product_id = ppl.product_id
        WHERE ppl.id = $1 AND ppl.deleted_at IS NULL
      `, [lineId]);

      if (lineResult.rows.length === 0) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      const line = lineResult.rows[0];
      
      // Check if product is reserved from packed_products warehouse (already produced)
      const isPackedProductReserved = line.packed_product_id && (line.packed_reserved_qty || 0) > 0;
      
      if (!line.bom_id) {
        return res.json({
          lineId,
          productSku: line.product_sku,
          productTitle: line.product_title,
          bomId: null,
          isPackedProductReserved,
          formatki: [],
          summary: { total: 0, produced: 0, toProduction: 0 }
        });
      }

      // Get BOM components (formatki)
      const componentsResult = await pool.query(`
        SELECT 
          pc.id as component_id,
          pc.generated_name,
          pc.component_type,
          pc.calculated_length,
          pc.calculated_width,
          pc.thickness,
          pc.quantity as component_qty,
          pc.color,
          pc.edging_pattern
        FROM bom.product_components pc
        WHERE pc.product_bom_id = $1
        ORDER BY pc.id
      `, [line.bom_id]);

      // Get color dictionary for badge colors
      const colorsResult = await pool.query(`
        SELECT code, name, color as hex_color
        FROM product_creator.dictionaries
        WHERE dictionary_type = 'color' AND is_active = true
      `);
      
      const colorDict = new Map<string, string>();
      for (const c of colorsResult.rows) {
        colorDict.set(c.code?.toUpperCase(), c.hex_color);
        colorDict.set(c.name?.toUpperCase(), c.hex_color);
      }

      // Get existing reservations for this plan line (ALL statuses for history)
      // UWAGA: Formatki są w warehouse.stock_panels, NIE w warehouse.materials
      const reservationsResult = await pool.query(`
        SELECT 
          r.id as reservation_id,
          r.product_sku,
          r.quantity_reserved,
          r.quantity_consumed,
          r.status,
          r.reserved_at,
          r.consumed_at,
          r.cancelled_at,
          r.notes,
          sp.id as material_id,
          sp.generated_name as material_name,
          sp.generated_name as internal_code
        FROM production.production_buffer_reservations r
        LEFT JOIN warehouse.stock_panels sp ON sp.generated_name = r.product_sku
        WHERE r.zlp_item_id = $1
        ORDER BY r.reserved_at DESC
      `, [lineId]);

      // Separate active reservations (for status) and all reservations (for history)
      const activeReservationsMap = new Map();
      const reservationHistoryMap = new Map<string, any[]>();
      
      for (const r of reservationsResult.rows) {
        // Track all reservations as history per SKU
        const historyList = reservationHistoryMap.get(r.product_sku) || [];
        historyList.push({
          id: r.reservation_id,
          status: r.status,
          quantityReserved: parseFloat(r.quantity_reserved),
          quantityConsumed: r.quantity_consumed ? parseFloat(r.quantity_consumed) : null,
          reservedAt: r.reserved_at,
          consumedAt: r.consumed_at,
          cancelledAt: r.cancelled_at,
          notes: r.notes
        });
        reservationHistoryMap.set(r.product_sku, historyList);
        
        // Only keep ACTIVE reservation in the main map for status determination
        if (r.status === 'ACTIVE') {
          activeReservationsMap.set(r.product_sku, r);
        }
      }

      // Build formatki list with production status
      // KEY LOGIC: If packed product is reserved from warehouse, all formatki are ALREADY PRODUCED
      // Otherwise, check individual formatki reservations from formatki warehouse
      const formatki = [];
      let totalComponents = 0;
      let producedCount = 0;

      for (const comp of componentsResult.rows) {
        // Check if this is a formatka (has dimensions)
        const isFormatka = comp.calculated_length && comp.calculated_width;
        if (!isFormatka) continue;

        totalComponents++;
        const componentQty = (comp.component_qty || 1) * line.line_quantity;

        // Extract color from generated_name (last part after last dash)
        const nameParts = (comp.generated_name || '').split('-');
        const color = nameParts[nameParts.length - 1]?.toUpperCase() || comp.color || 'UNKNOWN';

        // Helper function to fully normalize a string for comparison
        const normalizeForMatch = (str: string): string => {
          return str
            .toLowerCase()
            .replace(/formatka-/gi, '') // Remove ALL occurrences of formatka-
            .replace(/×/g, 'x')        // Normalize × to x
            .replace(/[-_\s]+/g, '')   // Remove hyphens, underscores, spaces
            .trim();
        };
        
        // Extract key parts from a SKU/component name
        const extractKeyParts = (str: string): { dims: string | null; color: string | null } => {
          const normalized = str.toLowerCase().replace(/×/g, 'x');
          
          // Extract dimensions: e.g., "500x300"
          const dimsMatch = normalized.match(/(\d+)x(\d+)/i);
          const dims = dimsMatch ? `${dimsMatch[1]}x${dimsMatch[2]}` : null;
          
          // Extract color: last non-numeric token after removing dimensions
          const withoutDims = normalized.replace(/\d+x\d+/g, '').replace(/[-_×\s]+/g, ' ').trim();
          const tokens = withoutDims.split(/\s+/).filter(t => t && !/^\d+$/.test(t));
          const color = tokens.length > 0 ? tokens[tokens.length - 1].replace(/formatka/gi, '') : null;
          
          return { dims, color: color || null };
        };
        
        // Helper function to check if SKU matches component name
        const skuMatchesComponent = (sku: string, componentName: string): boolean => {
          if (!componentName || !sku) return false;
          
          const normalizedSku = normalizeForMatch(sku);
          const normalizedComp = normalizeForMatch(componentName);
          
          // Direct match after full normalization
          if (normalizedSku === normalizedComp) return true;
          
          // Bidirectional includes check with normalized strings
          if (normalizedSku.includes(normalizedComp)) return true;
          if (normalizedComp.includes(normalizedSku)) return true;
          
          // Extract and compare key parts (dimensions + color)
          const skuParts = extractKeyParts(sku);
          const compParts = extractKeyParts(componentName);
          
          // Must have matching dimensions
          if (skuParts.dims && compParts.dims && skuParts.dims === compParts.dims) {
            // If both have colors, they must match
            if (skuParts.color && compParts.color) {
              return skuParts.color === compParts.color;
            }
            // If only one has color or neither, consider it a match
            return true;
          }
          
          return false;
        };

        // FIRST: Check history map for matching SKUs (includes all statuses)
        let reservationHistory: any[] = [];
        let matchedSku: string | null = null;
        for (const [sku, history] of reservationHistoryMap.entries()) {
          if (skuMatchesComponent(sku, comp.generated_name)) {
            reservationHistory = history;
            matchedSku = sku;
            break;
          }
        }
        
        // SECOND: Check if there's an ACTIVE reservation for this component
        let formatkiReservation = null;
        if (matchedSku && activeReservationsMap.has(matchedSku)) {
          formatkiReservation = activeReservationsMap.get(matchedSku);
        } else {
          // Fallback: check all active reservations
          for (const [sku, res] of activeReservationsMap.entries()) {
            if (skuMatchesComponent(sku, comp.generated_name)) {
              formatkiReservation = res;
              // If we didn't find history via matched SKU, get it now
              if (reservationHistory.length === 0) {
                reservationHistory = reservationHistoryMap.get(sku) || [];
              }
              break;
            }
          }
        }

        // Determine production status:
        // - If packed product is reserved → formatka is PRODUCED (part of finished product)
        // - If formatka is reserved from formatki warehouse → formatka is RESERVED (ready for production)
        // - Otherwise → formatka needs TO BE PRODUCED
        let status: 'produced' | 'reserved' | 'to_production';
        if (isPackedProductReserved) {
          status = 'produced';
          producedCount++;
        } else if (formatkiReservation) {
          status = 'reserved';
          producedCount++; // Reserved formatki are also "ready", not needing production
        } else {
          status = 'to_production';
        }

        // Determine routing variant (simplified)
        let routingVariant = 'SUROWA';
        if (color && !['SUROWA', 'SUROWY', 'RAW'].includes(color.toUpperCase())) {
          routingVariant = 'OKLEINOWANA';
        }

        // Get color hex from dictionary
        const colorHex = colorDict.get(color) || null;

        formatki.push({
          componentId: comp.component_id,
          generatedName: comp.generated_name,
          componentType: comp.component_type || nameParts[0] || 'PANEL',
          length: parseFloat(comp.calculated_length),
          width: parseFloat(comp.calculated_width),
          thickness: comp.thickness ? parseFloat(comp.thickness) : 18,
          color,
          colorHex,
          quantity: componentQty,
          edgingPattern: comp.edging_pattern,
          routingVariant,
          status,
          formatkiReservation: formatkiReservation ? {
            id: formatkiReservation.reservation_id,
            sku: formatkiReservation.product_sku,
            quantityReserved: parseFloat(formatkiReservation.quantity_reserved),
            materialId: formatkiReservation.material_id,
            materialName: formatkiReservation.material_name,
            reservedAt: formatkiReservation.reserved_at
          } : null,
          reservationHistory: reservationHistory.length > 0 ? reservationHistory : null
        });
      }

      // Collect all reservations history for this line (not per-component, but for the whole line)
      const allLineReservations: any[] = [];
      for (const [sku, history] of reservationHistoryMap.entries()) {
        for (const item of history) {
          allLineReservations.push({
            ...item,
            sku
          });
        }
      }
      // Sort by reservedAt descending
      allLineReservations.sort((a, b) => 
        new Date(b.reservedAt).getTime() - new Date(a.reservedAt).getTime()
      );

      res.json({
        lineId,
        productSku: line.product_sku,
        productTitle: line.product_title,
        bomId: line.bom_id,
        isPackedProductReserved,
        formatki,
        lineReservationHistory: allLineReservations.length > 0 ? allLineReservations : null,
        summary: {
          total: totalComponents,
          produced: producedCount,
          toProduction: totalComponents - producedCount
        }
      });
    } catch (error) {
      console.error("Error fetching BOM formatki:", error);
      res.status(500).json({ message: "Failed to fetch BOM formatki" });
    }
  });

  app.delete("/api/production/planning/lines/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }


      
      const success = await planningService.deletePlanLine(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting plan line:", error);
      res.status(500).json({ message: "Failed to delete plan line" });
    }
  });

  // POST /api/production/planning/lines/:id/transfer - Transfer plan line to another plan (keeps reservations)
  app.post("/api/production/planning/lines/:id/transfer", requirePermission('manage_production'), async (req, res) => {
    try {
      const lineId = parseInt(req.params.id);
      if (isNaN(lineId)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const { targetPlanId } = req.body;
      if (!targetPlanId || isNaN(parseInt(targetPlanId))) {
        return res.status(400).json({ message: "Target plan ID is required" });
      }

      const transferredLine = await planningService.transferPlanLine(pool, lineId, parseInt(targetPlanId));
      
      res.json({
        message: "Linia została przeniesiona do nowego planu",
        line: transferredLine
      });
    } catch (error: any) {
      console.error("Error transferring plan line:", error);
      
      if (error.message?.includes('nie istnieje') || error.message?.includes('not found')) {
        return res.status(404).json({ message: error.message });
      }
      if (error.message?.includes('już istnieje') || error.message?.includes('already')) {
        return res.status(409).json({ message: error.message });
      }
      if (error.message?.includes('już w tym planie')) {
        return res.status(400).json({ message: error.message });
      }
      
      res.status(500).json({ message: error.message || "Failed to transfer plan line" });
    }
  });

  // POST /api/production/planning/lines/:id/generate-batches - Generate component batches for plan line
  app.post("/api/production/planning/lines/:id/generate-batches", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const batches = await batchEngine.generateBatchesForPlanLine(pool, id);
      
      res.status(201).json({
        message: `Generated ${batches.length} batches`,
        batches: batches.map(b => b.batch),
      });
    } catch (error) {
      console.error("Error generating batches:", error);
      
      if (error instanceof Error) {
        if (error.message.includes('not found')) {
          return res.status(404).json({ message: error.message });
        }
        if (error.message.includes('No active BOM')) {
          return res.status(400).json({ message: error.message });
        }
        if (error.message.includes('No components found')) {
          return res.status(400).json({ message: error.message });
        }
      }
      
      res.status(500).json({ message: "Failed to generate batches" });
    }
  });

  // GET /api/production/planning/lines/:id/batches - Get all batches for a plan line
  app.get("/api/production/planning/lines/:id/batches", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const batches = await batchEngine.getBatchesForPlanLine(pool, id);
      res.json(batches);
    } catch (error) {
      console.error("Error fetching batches:", error);
      res.status(500).json({ message: "Failed to fetch batches" });
    }
  });

  // GET /api/production/planning/plans/:id/available-orders - Get available marketplace orders for production planning
  app.get("/api/production/planning/plans/:id/available-orders", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Coerce empty strings to undefined before parsing (URL params issue)
      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      // Parse and validate query parameters
      const filters = availableOrdersFiltersSchema.parse(normalizedQuery);

      // Parse search tokens into structured filters
      const parsedTokens = parseSearchTokens(filters.search);

      // Whitelist map for safe sorting (prevents SQL injection)
      const sortColumnMap: Record<string, string> = {
        'order_date': 'o.order_date',
        'order_number': 'o.order_number',
        'buyer_name': 'LOWER(o.buyer_last_name || \' \' || o.buyer_first_name)',
        'total_amount': 'o.total_to_pay_amount',
        'product_sku': 'MIN(cp.sku)',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase(); // 'ASC' or 'DESC'

      // Build WHERE clauses for filtering - use validated filters
      const conditions: string[] = ['1=1'];
      const params: any[] = [];
      let paramIndex = 1; // Start at 1 (no planId in params for count query)

      // Marketplace filter from parsed tokens (supports multiple)
      if (parsedTokens.marketplaces.length > 0) {
        const placeholders = parsedTokens.marketplaces.map((_, i) => `$${paramIndex + i}`).join(', ');
        conditions.push(`o.source IN (${placeholders})`);
        params.push(...parsedTokens.marketplaces);
        paramIndex += parsedTokens.marketplaces.length;
      }

      // Payment status filter from parsed tokens (supports multiple)
      if (parsedTokens.paymentStatuses.length > 0) {
        const placeholders = parsedTokens.paymentStatuses.map((_, i) => `$${paramIndex + i}`).join(', ');
        conditions.push(`o.payment_status IN (${placeholders})`);
        params.push(...parsedTokens.paymentStatuses);
        paramIndex += parsedTokens.paymentStatuses.length;
      }

      // Search terms filter (each term AND-ed together)
      if (parsedTokens.searchTerms.length > 0) {
        parsedTokens.searchTerms.forEach(term => {
          conditions.push(`(
            o.order_number ILIKE $${paramIndex} OR
            o.buyer_first_name ILIKE $${paramIndex} OR
            o.buyer_last_name ILIKE $${paramIndex} OR
            o.buyer_email ILIKE $${paramIndex} OR
            COALESCE(mp.sku, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.sku, ps.sku, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.product_type, sm.product_group, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.title, ps.title, '')::text ILIKE $${paramIndex} OR
            COALESCE(oi.name, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.color, ps.color, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.doors, sm.doors, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.legs, sm.legs, '')::text ILIKE $${paramIndex} OR
            EXISTS (
              SELECT 1 FROM unnest(COALESCE(cp.color_options, ps.color_options, ARRAY[]::text[])) AS co 
              WHERE co ILIKE $${paramIndex}
            )
          )`);
          params.push(`%${term}%`);
          paramIndex++;
        });
      }

      // Order number filter
      if (filters.orderNumber) {
        conditions.push(`o.order_number ILIKE $${paramIndex}`);
        params.push(`%${filters.orderNumber}%`);
        paramIndex++;
      }

      // Customer name filter
      if (filters.customerName) {
        conditions.push(`(
          o.buyer_first_name ILIKE $${paramIndex} OR
          o.buyer_last_name ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.customerName}%`);
        paramIndex++;
      }

      // SKU filter
      if (filters.sku) {
        conditions.push(`(
          COALESCE(mp.sku, '')::text ILIKE $${paramIndex} OR
          COALESCE(cp.sku, ps.sku, '')::text ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.sku}%`);
        paramIndex++;
      }

      // Color filter
      if (filters.color) {
        conditions.push(`COALESCE(cp.color, ps.color, '') ILIKE $${paramIndex}`);
        params.push(`%${filters.color}%`);
        paramIndex++;
      }

      // Length filters
      if (filters.minLength) {
        conditions.push(`cp.length >= $${paramIndex}`);
        params.push(filters.minLength);
        paramIndex++;
      }
      if (filters.maxLength) {
        conditions.push(`cp.length <= $${paramIndex}`);
        params.push(filters.maxLength);
        paramIndex++;
      }

      // Width filters
      if (filters.minWidth) {
        conditions.push(`cp.width >= $${paramIndex}`);
        params.push(filters.minWidth);
        paramIndex++;
      }
      if (filters.maxWidth) {
        conditions.push(`cp.width <= $${paramIndex}`);
        params.push(filters.maxWidth);
        paramIndex++;
      }

      // Legacy marketplace filter (kept for backward compatibility if passed explicitly)
      if (filters.marketplace && filters.marketplace !== 'all' && parsedTokens.marketplaces.length === 0) {
        conditions.push(`o.source = $${paramIndex}`);
        params.push(filters.marketplace);
        paramIndex++;
      }

      // Filter: Show only catalog-linked products or sets
      if (filters.showCatalogLinked === 'true') {
        conditions.push(`(cp.id IS NOT NULL OR ps.id IS NOT NULL)`);
      }

      // Filter: Show only products that are sets (linked to catalog.product_sets)
      if (filters.showSetsOnly === 'true') {
        conditions.push(`ps.id IS NOT NULL`);
      }

      // Filter: Show only items that are already in any plan (exclude soft-deleted lines)
      // Match specific order items: must match both order_number AND (product_id OR set_id)
      if (filters.showInPlans === 'true') {
        conditions.push(`(
          EXISTS (
            SELECT 1 FROM production.production_plan_lines pl 
            WHERE pl.deleted_at IS NULL
              AND pl.metadata->>'order_number' = o.order_number
              AND (
                pl.product_id = cp.id
                OR (pl.metadata->>'set_id')::int = ps.id
              )
          )
        )`);
      }

      // Date filter - filter orders by date
      if (filters.dateFilter && filters.dateFilter !== 'all') {
        switch (filters.dateFilter) {
          case 'today':
            conditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw')`);
            break;
          case 'yesterday':
            conditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '1 day')`);
            break;
          case 'day-before':
            conditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '2 days')`);
            break;
          case 'custom-days':
            if (filters.customDays && filters.customDays > 0) {
              conditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '${filters.customDays} days')`);
            }
            break;
        }
      }

      const whereClause = conditions.join(' AND ');

      // Step 1: Get total count for pagination (with same core joins to ensure cp and ps aliases exist)
      const countResult = await pool.query(`
        SELECT COUNT(DISTINCT o.id)::int as total
        FROM commerce.orders o
        LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
        LEFT JOIN commerce.marketplace_products mp ON oi.offer_external_id = mp.offer_external_id AND mp.source = o.source
        LEFT JOIN catalog.product_platform_data ppd ON oi.offer_external_id = ppd.external_id
        LEFT JOIN catalog.products cp ON ppd.product_id = cp.id AND (ppd.link_type = 'product' OR ppd.link_type IS NULL)
        LEFT JOIN product_creator.product_sets ps ON ppd.set_id = ps.id AND ppd.link_type = 'set'
        LEFT JOIN product_creator.set_matrices sm ON ps.set_matrix_id = sm.id
        WHERE ${whereClause}
          AND oi.id IS NOT NULL
      `, params);

      const total = countResult.rows[0]?.total || 0;

      // Step 2: Build WHERE clause for data query with adjusted param indices
      // Data query needs planId as $1, so we rebuild conditions with paramIndex starting from 2
      const dataConditions: string[] = ['1=1'];
      const dataParams: any[] = [planId]; // Start with planId at $1
      let dataParamIndex = 2; // Start filter params at $2

      // Rebuild all filter conditions with adjusted indices
      // Marketplace filter from parsed tokens (supports multiple)
      if (parsedTokens.marketplaces.length > 0) {
        const placeholders = parsedTokens.marketplaces.map((_, i) => `$${dataParamIndex + i}`).join(', ');
        dataConditions.push(`o.source IN (${placeholders})`);
        dataParams.push(...parsedTokens.marketplaces);
        dataParamIndex += parsedTokens.marketplaces.length;
      }

      // Payment status filter from parsed tokens (supports multiple)
      if (parsedTokens.paymentStatuses.length > 0) {
        const placeholders = parsedTokens.paymentStatuses.map((_, i) => `$${dataParamIndex + i}`).join(', ');
        dataConditions.push(`o.payment_status IN (${placeholders})`);
        dataParams.push(...parsedTokens.paymentStatuses);
        dataParamIndex += parsedTokens.paymentStatuses.length;
      }

      // Search terms filter (each term AND-ed together)
      if (parsedTokens.searchTerms.length > 0) {
        parsedTokens.searchTerms.forEach(term => {
          dataConditions.push(`(
            o.order_number ILIKE $${dataParamIndex} OR
            o.buyer_first_name ILIKE $${dataParamIndex} OR
            o.buyer_last_name ILIKE $${dataParamIndex} OR
            o.buyer_email ILIKE $${dataParamIndex} OR
            COALESCE(mp.sku, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.sku, ps.sku, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.product_type, sm.product_group, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.title, ps.title, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(oi.name, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.color, ps.color, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.doors, sm.doors, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.legs, sm.legs, '')::text ILIKE $${dataParamIndex} OR
            EXISTS (
              SELECT 1 FROM unnest(COALESCE(cp.color_options, ps.color_options, ARRAY[]::text[])) AS co 
              WHERE co ILIKE $${dataParamIndex}
            )
          )`);
          dataParams.push(`%${term}%`);
          dataParamIndex++;
        });
      }
      if (filters.orderNumber) {
        dataConditions.push(`o.order_number ILIKE $${dataParamIndex}`);
        dataParams.push(`%${filters.orderNumber}%`);
        dataParamIndex++;
      }
      if (filters.customerName) {
        dataConditions.push(`(
          o.buyer_first_name ILIKE $${dataParamIndex} OR
          o.buyer_last_name ILIKE $${dataParamIndex}
        )`);
        dataParams.push(`%${filters.customerName}%`);
        dataParamIndex++;
      }
      if (filters.sku) {
        dataConditions.push(`(
          COALESCE(mp.sku, '')::text ILIKE $${dataParamIndex} OR
          COALESCE(cp.sku, ps.sku, '')::text ILIKE $${dataParamIndex}
        )`);
        dataParams.push(`%${filters.sku}%`);
        dataParamIndex++;
      }
      if (filters.color) {
        dataConditions.push(`COALESCE(cp.color, ps.color, '') ILIKE $${dataParamIndex}`);
        dataParams.push(`%${filters.color}%`);
        dataParamIndex++;
      }
      if (filters.minLength) {
        dataConditions.push(`cp.length >= $${dataParamIndex}`);
        dataParams.push(filters.minLength);
        dataParamIndex++;
      }
      if (filters.maxLength) {
        dataConditions.push(`cp.length <= $${dataParamIndex}`);
        dataParams.push(filters.maxLength);
        dataParamIndex++;
      }
      if (filters.minWidth) {
        dataConditions.push(`cp.width >= $${dataParamIndex}`);
        dataParams.push(filters.minWidth);
        dataParamIndex++;
      }
      if (filters.maxWidth) {
        dataConditions.push(`cp.width <= $${dataParamIndex}`);
        dataParams.push(filters.maxWidth);
        dataParamIndex++;
      }
      // Legacy marketplace filter (kept for backward compatibility if passed explicitly)
      if (filters.marketplace && filters.marketplace !== 'all' && parsedTokens.marketplaces.length === 0) {
        dataConditions.push(`o.source = $${dataParamIndex}`);
        dataParams.push(filters.marketplace);
        dataParamIndex++;
      }

      // Filter: Show only catalog-linked products or sets
      if (filters.showCatalogLinked === 'true') {
        dataConditions.push(`(cp.id IS NOT NULL OR ps.id IS NOT NULL)`);
      }

      // Filter: Show only products that are sets (linked to catalog.product_sets)
      if (filters.showSetsOnly === 'true') {
        dataConditions.push(`ps.id IS NOT NULL`);
      }

      // Filter: Show only items that are already in any plan (exclude soft-deleted lines)
      // Match specific order items: must match both order_number AND (product_id OR set_id)
      if (filters.showInPlans === 'true') {
        dataConditions.push(`(
          EXISTS (
            SELECT 1 FROM production.production_plan_lines pl 
            WHERE pl.deleted_at IS NULL
              AND pl.metadata->>'order_number' = o.order_number
              AND (
                pl.product_id = cp.id
                OR (pl.metadata->>'set_id')::int = ps.id
              )
          )
        )`);
      }

      // Date filter - filter orders by date
      if (filters.dateFilter && filters.dateFilter !== 'all') {
        switch (filters.dateFilter) {
          case 'today':
            dataConditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw')`);
            break;
          case 'yesterday':
            dataConditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '1 day')`);
            break;
          case 'day-before':
            dataConditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '2 days')`);
            break;
          case 'custom-days':
            if (filters.customDays && filters.customDays > 0) {
              dataConditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '${filters.customDays} days')`);
            }
            break;
        }
      }

      const dataWhereClause = dataConditions.join(' AND ');
      
      // Include MIN(cp.sku) in SELECT/GROUP BY when sorting by product_sku
      const includeMinSku = filters.sortBy === 'product_sku';
      
      const dataResult = await pool.query(`
        SELECT 
          o.id as order_id,
          o.order_number,
          o.source as marketplace,
          o.buyer_first_name,
          o.buyer_last_name,
          o.buyer_email,
          o.order_date,
          o.payment_status,
          o.total_to_pay_amount,
          o.total_to_pay_currency as currency,
          ${includeMinSku ? 'MIN(cp.sku) as min_sku,' : ''}
          json_agg(
            json_build_object(
              'item_id', oi.id,
              'offer_external_id', oi.offer_external_id,
              'name', oi.name,
              'quantity', oi.quantity,
              'unit_price', oi.unit_price,
              'price', oi.price,
              'image_url', oi.image_url,
              'product_length', COALESCE(cp.length::text, sm.length),
              'product_width', COALESCE(cp.width::text, sm.width),
              'product_height', COALESCE(cp.height::text, sm.height),
              'product_color', COALESCE(cp.color, ps.color),
              'product_color_options', COALESCE(cp.color_options, ps.color_options),
              'product_sku', COALESCE(cp.sku, ps.sku),
              'product_type', COALESCE(cp.product_type, sm.product_group),
              'product_doors', COALESCE(cp.doors, sm.doors),
              'product_legs', COALESCE(cp.legs, sm.legs),
              'warehouse_total_qty', wpp.quantity,
              'warehouse_reserved_qty', wpp.reserved_quantity,
              'line_reserved_qty', ppl_any.reserved_quantity,
              'marketplace_product_id', mp.id,
              'link_type', ppd.link_type,
              'catalog_product_id', cp.id,
              'catalog_product_sku', cp.sku,
              'catalog_product_title', cp.title,
              'catalog_set_id', ps.id,
              'catalog_set_sku', ps.sku,
              'catalog_set_title', ps.title,
              'platform_link_id', ppd.id,
              'bom_component_count', COALESCE(pb_comp_count.component_count, 0),
              'product_group_id', ppg.id,
              'product_group_name', ppg.name,
              'product_group_color_hex', ppg.color_hex,
              'is_in_plan', CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END,
              'in_plan_number', pp_any.plan_number,
              'in_plan_id', pp_any.id,
              'in_current_plan', CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END,
              'set_components', set_components.components
            ) ORDER BY oi.id
          ) FILTER (WHERE oi.id IS NOT NULL) as items
        FROM commerce.orders o
        LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
        LEFT JOIN commerce.marketplace_products mp ON oi.offer_external_id = mp.offer_external_id AND mp.source = o.source
        LEFT JOIN LATERAL (
          SELECT DISTINCT ON (external_id, link_type) *
          FROM catalog.product_platform_data
          WHERE external_id = oi.offer_external_id
          ORDER BY external_id, link_type, id DESC
        ) ppd ON true
        LEFT JOIN catalog.products cp ON ppd.product_id = cp.id AND (ppd.link_type = 'product' OR ppd.link_type IS NULL)
        LEFT JOIN product_creator.product_sets ps ON ppd.set_id = ps.id AND ppd.link_type = 'set'
        LEFT JOIN product_creator.set_matrices sm ON ps.set_matrix_id = sm.id
        LEFT JOIN bom.product_boms pb ON pb.product_id = cp.id AND pb.is_active = true
        LEFT JOIN LATERAL (
          SELECT COUNT(*)::int as component_count
          FROM bom.product_components pc
          WHERE pc.product_bom_id = pb.id
        ) pb_comp_count ON true
        LEFT JOIN warehouse.packed_products wpp ON wpp.catalog_product_id = cp.id
        LEFT JOIN production.production_product_group_items ppgi ON ppgi.product_id = cp.id
        LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
        LEFT JOIN production.production_plan_lines ppl_current ON (
          ppl_current.plan_id = $1 
          AND ppl_current.source_type = 'order_demand'
          AND ppl_current.product_id = cp.id
          AND (ppd.link_type != 'set' OR ppd.link_type IS NULL)
          AND (ppl_current.metadata->>'order_number') = o.order_number
          AND ppl_current.deleted_at IS NULL
        )
        LEFT JOIN production.production_plan_lines ppl_any ON (
          ppl_any.source_type = 'order_demand'
          AND ppl_any.product_id = cp.id
          AND (ppd.link_type != 'set' OR ppd.link_type IS NULL)
          AND (ppl_any.metadata->>'order_number') = o.order_number
          AND ppl_any.deleted_at IS NULL
        )
        LEFT JOIN production.production_plans pp_any ON pp_any.id = ppl_any.plan_id
        LEFT JOIN LATERAL (
          SELECT json_agg(
            json_build_object(
              'component_id', comp_data.component_id,
              'component_sku', comp_data.component_sku,
              'component_title', comp_data.component_title,
              'component_color', comp_data.component_color,
              'component_length', comp_data.component_length,
              'component_width', comp_data.component_width,
              'component_height', comp_data.component_height,
              'component_product_type', comp_data.component_product_type,
              'component_doors', comp_data.component_doors,
              'component_legs', comp_data.component_legs,
              'quantity', comp_data.quantity,
              'primary_image_url', comp_data.primary_image_url,
              'parent_set_image_url', comp_data.parent_set_image_url,
              'is_in_current_plan', comp_data.is_in_current_plan,
              'is_in_any_plan', comp_data.is_in_any_plan,
              'in_plan_number', comp_data.in_plan_number,
              'in_plan_id', comp_data.in_plan_id
            ) ORDER BY comp_data.spl_id
          ) as components
          FROM (
            SELECT DISTINCT ON (comp.id)
              comp.id as component_id,
              comp.sku as component_sku,
              comp.title as component_title,
              comp.color as component_color,
              comp.length as component_length,
              comp.width as component_width,
              comp.height as component_height,
              comp.product_type as component_product_type,
              comp.doors as component_doors,
              comp.legs as component_legs,
              spl.quantity,
              COALESCE(comp.image_url, comp_img.url, oi.image_url) as primary_image_url,
              oi.image_url as parent_set_image_url,
              CASE WHEN ppl_comp_current.id IS NOT NULL THEN true ELSE false END as is_in_current_plan,
              CASE WHEN ppl_comp_any.id IS NOT NULL THEN true ELSE false END as is_in_any_plan,
              pp_comp_any.plan_number as in_plan_number,
              pp_comp_any.id as in_plan_id,
              spl.id as spl_id
            FROM product_creator.set_product_links spl
            JOIN catalog.products comp ON comp.id = spl.product_id
            LEFT JOIN catalog.product_images comp_img ON comp_img.product_id = comp.id AND comp_img.is_primary = true
            LEFT JOIN production.production_plan_lines ppl_comp_current ON (
              ppl_comp_current.plan_id = $1
              AND ppl_comp_current.source_type = 'order_demand'
              AND (ppl_comp_current.metadata->>'set_id')::int = ps.id
              AND (ppl_comp_current.metadata->>'component_id')::int = comp.id
              AND (ppl_comp_current.metadata->>'order_number') = o.order_number
              AND ppl_comp_current.deleted_at IS NULL
            )
            LEFT JOIN production.production_plan_lines ppl_comp_any ON (
              ppl_comp_any.source_type = 'order_demand'
              AND (ppl_comp_any.metadata->>'set_id')::int = ps.id
              AND (ppl_comp_any.metadata->>'component_id')::int = comp.id
              AND (ppl_comp_any.metadata->>'order_number') = o.order_number
              AND ppl_comp_any.deleted_at IS NULL
            )
            LEFT JOIN production.production_plans pp_comp_any ON pp_comp_any.id = ppl_comp_any.plan_id
            WHERE spl.set_id = ps.id
            ORDER BY comp.id, ppl_comp_any.id DESC NULLS LAST
          ) comp_data
        ) set_components ON ps.id IS NOT NULL
        WHERE ${dataWhereClause}
        GROUP BY o.id, o.order_number, o.source, o.buyer_first_name, o.buyer_last_name, 
                 o.buyer_email, o.order_date, o.payment_status, o.total_to_pay_amount, o.total_to_pay_currency
        HAVING COUNT(oi.id) > 0
        ORDER BY ${includeMinSku ? 'min_sku' : orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      res.json({
        orders: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching available orders:", error);
      res.status(500).json({ message: "Failed to fetch available orders" });
    }
  });

  // GET /api/production/planning/plans/:id/available-catalog-products - Get catalog products for internal orders
  app.get("/api/production/planning/plans/:id/available-catalog-products", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      const filters = z.object({
        search: z.string().optional(),
        color: z.string().optional(),
        sku: z.string().optional(),
        minLength: z.coerce.number().optional(),
        maxLength: z.coerce.number().optional(),
        minWidth: z.coerce.number().optional(),
        maxWidth: z.coerce.number().optional(),
        limit: z.coerce.number().min(1).max(500).optional().default(100),
        offset: z.coerce.number().min(0).optional().default(0),
        sortBy: z.enum(['title', 'sku', 'color', 'created_at']).optional().default('created_at'),
        sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
      }).parse(normalizedQuery);

      const sortColumnMap: Record<string, string> = {
        'title': 'cp.title',
        'sku': 'cp.sku',
        'color': 'cp.color',
        'created_at': 'cp.created_at',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase();

      const conditions: string[] = ['cp.is_active = true'];
      const params: any[] = [];
      let paramIndex = 1;

      if (filters.search) {
        conditions.push(`(
          cp.title ILIKE $${paramIndex} OR
          cp.sku ILIKE $${paramIndex} OR
          cp.color ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.search}%`);
        paramIndex++;
      }

      if (filters.color) {
        conditions.push(`cp.color ILIKE $${paramIndex}`);
        params.push(`%${filters.color}%`);
        paramIndex++;
      }

      if (filters.sku) {
        conditions.push(`cp.sku ILIKE $${paramIndex}`);
        params.push(`%${filters.sku}%`);
        paramIndex++;
      }

      if (filters.minLength) {
        conditions.push(`cp.length::numeric >= $${paramIndex}`);
        params.push(filters.minLength);
        paramIndex++;
      }

      if (filters.maxLength) {
        conditions.push(`cp.length::numeric <= $${paramIndex}`);
        params.push(filters.maxLength);
        paramIndex++;
      }

      if (filters.minWidth) {
        conditions.push(`cp.width::numeric >= $${paramIndex}`);
        params.push(filters.minWidth);
        paramIndex++;
      }

      if (filters.maxWidth) {
        conditions.push(`cp.width::numeric <= $${paramIndex}`);
        params.push(filters.maxWidth);
        paramIndex++;
      }

      const whereClause = conditions.join(' AND ');

      const countResult = await pool.query(`
        SELECT COUNT(*)::int as total
        FROM catalog.products cp
        WHERE ${whereClause}
      `, params);

      const total = countResult.rows[0].total;

      const dataParams = [...params, planId];
      let dataParamIndex = dataParams.length + 1;

      const dataResult = await pool.query(`
        SELECT DISTINCT ON (cp.id)
          cp.id,
          cp.sku,
          cp.title,
          cp.color,
          cp.color_options,
          cp.length,
          cp.width,
          cp.height,
          cp.product_type,
          cp.doors,
          cp.legs,
          cp.created_at,
          (
            SELECT pi.url
            FROM catalog.product_images pi
            WHERE pi.product_id = cp.id
            ORDER BY pi.is_primary DESC, pi.sort_order ASC
            LIMIT 1
          ) as image_url,
          COALESCE(pb_comp_count.component_count, 0) as bom_component_count,
          (
            SELECT ppg.id
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_id,
          (
            SELECT ppg.name
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_name,
          (
            SELECT ppg.color_hex
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_color_hex,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as is_in_plan,
          (
            SELECT pp_any.plan_number
            FROM production.production_plan_lines ppl_any
            LEFT JOIN production.production_plans pp_any ON pp_any.id = ppl_any.plan_id
            WHERE ppl_any.source_type = 'catalog_internal'
              AND ppl_any.product_id = cp.id
            LIMIT 1
          ) as in_plan_number,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as in_current_plan
        FROM catalog.products cp
        LEFT JOIN bom.product_boms pb ON pb.product_id = cp.id AND pb.is_active = true
        LEFT JOIN LATERAL (
          SELECT COUNT(*)::int as component_count
          FROM bom.product_components pc
          WHERE pc.product_bom_id = pb.id
        ) pb_comp_count ON true
        LEFT JOIN production.production_plan_lines ppl_current ON (
          ppl_current.plan_id = $${dataParams.length}
          AND ppl_current.source_type = 'catalog_internal'
          AND ppl_current.product_id = cp.id
        )
        WHERE ${whereClause}
        ORDER BY cp.id, ${orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      res.json({
        products: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching catalog products:", error);
      res.status(500).json({ message: "Failed to fetch catalog products" });
    }
  });

  // GET /api/production/planning/plans/:id/available-cutting-patterns - Get cutting patterns for internal orders
  app.get("/api/production/planning/plans/:id/available-cutting-patterns", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      const filters = z.object({
        search: z.string().optional(),
        status: z.string().optional(),
        limit: z.coerce.number().min(1).max(500).optional().default(100),
        offset: z.coerce.number().min(0).optional().default(0),
        sortBy: z.enum(['name', 'code', 'created_at']).optional().default('created_at'),
        sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
      }).parse(normalizedQuery);

      const sortColumnMap: Record<string, string> = {
        'name': 'cpt.name',
        'code': 'cpt.code',
        'created_at': 'cpt.created_at',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase();

      const conditions: string[] = ['cpt.is_active = true'];
      const params: any[] = [];
      let paramIndex = 1;

      if (filters.search) {
        conditions.push(`(
          cpt.name ILIKE $${paramIndex} OR
          cpt.code ILIKE $${paramIndex} OR
          cpt.description ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.search}%`);
        paramIndex++;
      }

      const whereClause = conditions.join(' AND ');

      const countResult = await pool.query(`
        SELECT COUNT(*)::int as total
        FROM production.cut_pattern_templates cpt
        WHERE ${whereClause}
      `, params);

      const total = countResult.rows[0].total;

      const dataParams = [...params, planId];
      let dataParamIndex = dataParams.length + 1;

      const dataResult = await pool.query(`
        SELECT 
          cpt.id,
          cpt.code,
          cpt.name,
          cpt.description,
          cpt.is_active,
          cpt.board_length,
          cpt.board_width,
          cpt.board_thickness,
          cpt.kerf,
          cpt.created_at,
          COALESCE(SUM(cpti.quantity_default), 0)::int as total_quantity,
          COUNT(cpti.id)::int as items_count,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as is_in_plan,
          pp_any.plan_number as in_plan_number,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as in_current_plan
        FROM production.cut_pattern_templates cpt
        LEFT JOIN production.cut_pattern_template_items cpti ON cpti.template_id = cpt.id
        LEFT JOIN production.production_plan_lines ppl_current ON (
          ppl_current.plan_id = $${dataParams.length}
          AND ppl_current.source_type = 'cutting_pattern'
          AND (ppl_current.metadata->>'cutting_pattern_id')::int = cpt.id
        )
        LEFT JOIN production.production_plan_lines ppl_any ON (
          ppl_any.source_type = 'cutting_pattern'
          AND (ppl_any.metadata->>'cutting_pattern_id')::int = cpt.id
        )
        LEFT JOIN production.production_plans pp_any ON pp_any.id = ppl_any.plan_id
        WHERE ${whereClause}
        GROUP BY cpt.id, cpt.code, cpt.name, cpt.description, cpt.is_active, 
                 cpt.board_length, cpt.board_width, cpt.board_thickness, cpt.kerf, 
                 cpt.created_at, ppl_current.id, pp_any.plan_number
        ORDER BY ${orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      res.json({
        patterns: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching cutting patterns:", error);
      res.status(500).json({ message: "Failed to fetch cutting patterns" });
    }
  });
}
