import { Router } from "express";
import { pool } from "../../postgres";
import {
  getBufferStock,
  getBufferStockBySku,
  getBufferStockByLocation,
} from "../../services/production/buffer-stock";
import {
  getBufferMovements,
  createBufferMovement,
} from "../../services/production/buffer-movements";
import {
  getReservationsByZlp,
  getReservationsBySku,
  createReservation,
  consumeReservation,
  cancelReservation,
} from "../../services/production/buffer-reservations";
import { insertProductionBufferMovementSchema, insertProductionBufferReservationSchema } from "@shared/schema";

export const bufferRouter = Router();

// ===== Buffer Stock Endpoints =====

// GET /api/production/buffer/stock - Get all buffer stock
bufferRouter.get("/stock", async (req, res) => {
  try {
    const stock = await getBufferStock(pool);
    res.json(stock);
  } catch (error: any) {
    console.error("Error fetching buffer stock:", error);
    res.status(500).json({ error: "Failed to fetch buffer stock" });
  }
});

// GET /api/production/buffer/stock/sku/:sku - Get buffer stock by SKU
bufferRouter.get("/stock/sku/:sku", async (req, res) => {
  try {
    const { sku } = req.params;
    const stock = await getBufferStockBySku(pool, sku);
    res.json(stock);
  } catch (error: any) {
    console.error("Error fetching buffer stock by SKU:", error);
    res.status(500).json({ error: "Failed to fetch buffer stock" });
  }
});

// GET /api/production/buffer/stock/location/:locationId - Get buffer stock by location
bufferRouter.get("/stock/location/:locationId", async (req, res) => {
  try {
    const locationId = parseInt(req.params.locationId);
    const stock = await getBufferStockByLocation(pool, locationId);
    res.json(stock);
  } catch (error: any) {
    console.error("Error fetching buffer stock by location:", error);
    res.status(500).json({ error: "Failed to fetch buffer stock" });
  }
});

// ===== Buffer Movements Endpoints =====

// GET /api/production/buffer/movements - Get buffer movements with filters
bufferRouter.get("/movements", async (req, res) => {
  try {
    const filters = {
      productSku: req.query.productSku as string | undefined,
      locationId: req.query.locationId ? parseInt(req.query.locationId as string) : undefined,
      zlpId: req.query.zlpId ? parseInt(req.query.zlpId as string) : undefined,
      movementType: req.query.movementType as 'IN' | 'OUT' | 'RESERVE' | 'RELEASE' | undefined,
      limit: req.query.limit ? parseInt(req.query.limit as string) : 100,
    };
    
    const movements = await getBufferMovements(pool, filters);
    res.json(movements);
  } catch (error: any) {
    console.error("Error fetching buffer movements:", error);
    res.status(500).json({ error: "Failed to fetch buffer movements" });
  }
});

// POST /api/production/buffer/movements - Create buffer movement
bufferRouter.post("/movements", async (req, res) => {
  try {
    const parsed = insertProductionBufferMovementSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    const movement = await createBufferMovement(pool, parsed.data);
    res.status(201).json(movement);
  } catch (error: any) {
    console.error("Error creating buffer movement:", error);
    res.status(500).json({ error: "Failed to create buffer movement" });
  }
});

// ===== Buffer Reservations Endpoints =====

// GET /api/production/buffer/reservations/zlp/:zlpId - Get reservations by ZLP
bufferRouter.get("/reservations/zlp/:zlpId", async (req, res) => {
  try {
    const zlpId = parseInt(req.params.zlpId);
    const reservations = await getReservationsByZlp(pool, zlpId);
    res.json(reservations);
  } catch (error: any) {
    console.error("Error fetching ZLP reservations:", error);
    res.status(500).json({ error: "Failed to fetch reservations" });
  }
});

// GET /api/production/buffer/reservations/sku/:sku - Get active reservations by SKU
bufferRouter.get("/reservations/sku/:sku", async (req, res) => {
  try {
    const { sku } = req.params;
    const reservations = await getReservationsBySku(pool, sku);
    res.json(reservations);
  } catch (error: any) {
    console.error("Error fetching SKU reservations:", error);
    res.status(500).json({ error: "Failed to fetch reservations" });
  }
});

// POST /api/production/buffer/reservations - Create reservation
bufferRouter.post("/reservations", async (req, res) => {
  try {
    const parsed = insertProductionBufferReservationSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    const reservation = await createReservation(pool, parsed.data);
    res.status(201).json(reservation);
  } catch (error: any) {
    console.error("Error creating reservation:", error);
    res.status(500).json({ error: "Failed to create reservation" });
  }
});

// POST /api/production/buffer/reservations/:id/consume - Consume reservation
bufferRouter.post("/reservations/:id/consume", async (req, res) => {
  try {
    const reservationId = parseInt(req.params.id);
    const { quantityConsumed, userId } = req.body;
    
    if (!quantityConsumed || quantityConsumed <= 0) {
      return res.status(400).json({ error: "Invalid quantity" });
    }
    
    const reservation = await consumeReservation(pool, reservationId, quantityConsumed, userId);
    res.json(reservation);
  } catch (error: any) {
    console.error("Error consuming reservation:", error);
    res.status(500).json({ error: error.message || "Failed to consume reservation" });
  }
});

// POST /api/production/buffer/reservations/:id/cancel - Cancel reservation
bufferRouter.post("/reservations/:id/cancel", async (req, res) => {
  try {
    const reservationId = parseInt(req.params.id);
    const { userId } = req.body;
    
    const reservation = await cancelReservation(pool, reservationId, userId);
    res.json(reservation);
  } catch (error: any) {
    console.error("Error cancelling reservation:", error);
    res.status(500).json({ error: error.message || "Failed to cancel reservation" });
  }
});
