import { pool } from './postgres';
import { processImage, generateImageVersionFilenames } from './image-processor';
import { getFileStorageAdapter } from './file-storage-adapter';
import https from 'https';
import http from 'http';

/**
 * Migration script to generate thumbnails for existing matrix images
 * 
 * This script:
 * 1. Finds all product_matrices and set_matrices with old-format colorImages (string[])
 * 2. Downloads each original image
 * 3. Generates 3 optimized WebP versions (thumbnail 80x80, medium 400x400, original)
 * 4. Uploads to SFTP in organized folders
 * 5. Updates colorImages to new format with {url, thumbnailUrl, mediumUrl}
 */

interface ImageEntry {
  url: string;
  thumbnailUrl?: string;
  mediumUrl?: string;
}

async function downloadImage(url: string): Promise<Buffer> {
  return new Promise((resolve, reject) => {
    const protocol = url.startsWith('https') ? https : http;
    
    protocol.get(url, (response) => {
      if (response.statusCode !== 200) {
        reject(new Error(`Failed to download image: ${response.statusCode}`));
        return;
      }
      
      const chunks: Buffer[] = [];
      response.on('data', (chunk) => chunks.push(chunk));
      response.on('end', () => resolve(Buffer.concat(chunks)));
      response.on('error', reject);
    }).on('error', reject);
  });
}

async function processColorImages(
  colorImages: Record<string, (string | ImageEntry)[]>,
  matrixType: 'product' | 'set',
  matrixId: number
): Promise<Record<string, ImageEntry[]>> {
  const adapter = await getFileStorageAdapter();
  const updatedColorImages: Record<string, ImageEntry[]> = {};
  
  let totalImages = 0;
  let processedImages = 0;
  let skippedImages = 0;
  let errorImages = 0;
  
  // Count total images
  for (const colorKey in colorImages) {
    totalImages += colorImages[colorKey].length;
  }
  
  console.log(`\n📊 Matrix ${matrixId} (${matrixType}): ${totalImages} zdjęć do sprawdzenia`);
  
  for (const colorKey in colorImages) {
    const images = colorImages[colorKey];
    updatedColorImages[colorKey] = [];
    
    for (let i = 0; i < images.length; i++) {
      const imageEntry = images[i];
      
      // Check if already in new format
      if (typeof imageEntry === 'object' && imageEntry.thumbnailUrl && imageEntry.mediumUrl) {
        console.log(`  ⏭️  Pomijam [${colorKey}][${i}] - już ma miniatury`);
        updatedColorImages[colorKey].push(imageEntry);
        skippedImages++;
        continue;
      }
      
      // Old format - process it
      const originalUrl = typeof imageEntry === 'string' ? imageEntry : imageEntry.url;
      
      try {
        console.log(`  🔄 Przetwarzam [${colorKey}][${i}]: ${originalUrl}`);
        
        // Download original image
        const imageBuffer = await downloadImage(originalUrl);
        console.log(`    ✅ Pobrano ${(imageBuffer.length / 1024).toFixed(1)}KB`);
        
        // Extract original filename
        const originalFilename = originalUrl.split('/').pop() || `image-${Date.now()}.jpg`;
        const baseFilename = originalFilename.replace(/\.(jpg|jpeg|png|webp)$/i, '');
        
        // Generate filenames for all versions
        const filenames = generateImageVersionFilenames(baseFilename, 'webp');
        
        // Process image to generate 3 versions
        const processed = await processImage(imageBuffer, {
          thumbnailSize: 80,
          mediumSize: 400,
          quality: 90,
          format: 'webp',
        });
        
        console.log(`    📦 Wygenerowano: thumbnail(${(processed.buffers.thumbnail.length / 1024).toFixed(1)}KB), medium(${(processed.buffers.medium.length / 1024).toFixed(1)}KB), original(${(processed.buffers.original.length / 1024).toFixed(1)}KB)`);
        
        // Upload all versions to SFTP
        const folderPrefix = matrixType === 'product' ? 'product-matrix-colors' : 'set-matrix-colors';
        
        // Upload thumbnail
        await adapter.upload({
          filename: filenames.thumbnail,
          buffer: processed.buffers.thumbnail,
          mimetype: 'image/webp',
          subfolder: `${folderPrefix}/thumbnails`,
        });
        
        // Upload medium
        await adapter.upload({
          filename: filenames.medium,
          buffer: processed.buffers.medium,
          mimetype: 'image/webp',
          subfolder: `${folderPrefix}/medium`,
        });
        
        // Upload original
        await adapter.upload({
          filename: filenames.original,
          buffer: processed.buffers.original,
          mimetype: 'image/webp',
          subfolder: folderPrefix,
        });
        
        console.log(`    ☁️  Uploadowano na SFTP`);
        
        // Build URLs
        const newEntry: ImageEntry = {
          url: adapter.getUrl(`${folderPrefix}/${filenames.original}`),
          thumbnailUrl: adapter.getUrl(`${folderPrefix}/thumbnails/${filenames.thumbnail}`),
          mediumUrl: adapter.getUrl(`${folderPrefix}/medium/${filenames.medium}`),
        };
        
        updatedColorImages[colorKey].push(newEntry);
        processedImages++;
        
        console.log(`    ✅ [${processedImages}/${totalImages}] Gotowe!`);
        
      } catch (error: any) {
        console.error(`    ❌ Błąd podczas przetwarzania [${colorKey}][${i}]:`, error.message);
        
        // Use original URL as fallback
        updatedColorImages[colorKey].push({
          url: originalUrl,
          thumbnailUrl: originalUrl,
          mediumUrl: originalUrl,
        });
        errorImages++;
      }
    }
  }
  
  console.log(`\n📈 Podsumowanie Matrix ${matrixId}:`);
  console.log(`   ✅ Przetworzone: ${processedImages}`);
  console.log(`   ⏭️  Pominięte (już miały miniatury): ${skippedImages}`);
  console.log(`   ❌ Błędy (użyto fallback): ${errorImages}`);
  
  return updatedColorImages;
}

async function migrateProductMatrices() {
  console.log('\n🔵 === MIGRACJA PRODUCT MATRICES ===\n');
  
  // Get all product matrices with colorImages
  const result = await pool.query(`
    SELECT id, name, color_images
    FROM product_creator.product_matrices
    WHERE color_images IS NOT NULL
    ORDER BY id ASC
  `);
  
  console.log(`📊 Znaleziono ${result.rows.length} product matrices do sprawdzenia`);
  
  for (const row of result.rows) {
    const { id, name, color_images } = row;
    
    if (!color_images || Object.keys(color_images).length === 0) {
      console.log(`⏭️  Matrix #${id} "${name}" - brak zdjęć, pomijam`);
      continue;
    }
    
    console.log(`\n🔹 Przetwarzam Matrix #${id}: "${name}"`);
    
    const updatedColorImages = await processColorImages(color_images, 'product', id);
    
    // Update database
    await pool.query(`
      UPDATE product_creator.product_matrices
      SET color_images = $1
      WHERE id = $2
    `, [JSON.stringify(updatedColorImages), id]);
    
    console.log(`💾 Zaktualizowano bazę danych dla Matrix #${id}`);
  }
}

async function migrateSetMatrices() {
  console.log('\n🟢 === MIGRACJA SET MATRICES ===\n');
  
  // Get all set matrices with colorImages
  const result = await pool.query(`
    SELECT id, name, color_images
    FROM product_creator.set_matrices
    WHERE color_images IS NOT NULL
    ORDER BY id ASC
  `);
  
  console.log(`📊 Znaleziono ${result.rows.length} set matrices do sprawdzenia`);
  
  for (const row of result.rows) {
    const { id, name, color_images } = row;
    
    if (!color_images || Object.keys(color_images).length === 0) {
      console.log(`⏭️  Set Matrix #${id} "${name}" - brak zdjęć, pomijam`);
      continue;
    }
    
    console.log(`\n🔸 Przetwarzam Set Matrix #${id}: "${name}"`);
    
    const updatedColorImages = await processColorImages(color_images, 'set', id);
    
    // Update database
    await pool.query(`
      UPDATE product_creator.set_matrices
      SET color_images = $1
      WHERE id = $2
    `, [JSON.stringify(updatedColorImages), id]);
    
    console.log(`💾 Zaktualizowano bazę danych dla Set Matrix #${id}`);
  }
}

async function migrateCatalogImages() {
  console.log('\n🟣 === MIGRACJA CATALOG.PRODUCT_IMAGES ===\n');
  
  const adapter = await getFileStorageAdapter();
  
  // Count total images without thumbnails
  const countResult = await pool.query(`
    SELECT COUNT(*) as total
    FROM catalog.product_images
    WHERE thumbnail_url IS NULL OR medium_url IS NULL
  `);
  
  const totalImages = parseInt(countResult.rows[0].total);
  console.log(`📊 Znaleziono ${totalImages} zdjęć bez miniatur\n`);
  
  if (totalImages === 0) {
    console.log('✅ Wszystkie zdjęcia katalogowe mają już miniatury!');
    return;
  }
  
  const CHUNK_SIZE = 20;
  const DELAY_MS = 2000; // 2 second pause between chunks
  let processedCount = 0;
  let errorCount = 0;
  
  // Process in chunks to avoid timeout
  for (let offset = 0; offset < totalImages; offset += CHUNK_SIZE) {
    console.log(`\n📦 Chunk ${Math.floor(offset / CHUNK_SIZE) + 1}/${Math.ceil(totalImages / CHUNK_SIZE)} (offset: ${offset})`);
    
    // Get next chunk
    const result = await pool.query(`
      SELECT id, image_url
      FROM catalog.product_images
      WHERE (thumbnail_url IS NULL OR medium_url IS NULL)
        AND image_url IS NOT NULL
      ORDER BY id ASC
      LIMIT $1 OFFSET $2
    `, [CHUNK_SIZE, offset]);
    
    console.log(`   Przetwarzam ${result.rows.length} zdjęć...`);
    
    for (const row of result.rows) {
      const { id, image_url } = row;
      
      try {
        console.log(`   🔄 [${id}] ${image_url.substring(0, 60)}...`);
        
        // Download original
        const imageBuffer = await downloadImage(image_url);
        
        // Extract filename
        const originalFilename = image_url.split('/').pop() || `image-${id}.jpg`;
        const baseFilename = originalFilename.replace(/\.(jpg|jpeg|png|webp)$/i, '');
        const filenames = generateImageVersionFilenames(baseFilename, 'webp');
        
        // Process image
        const processed = await processImage(imageBuffer, {
          thumbnailSize: 80,
          mediumSize: 400,
          quality: 90,
          format: 'webp',
        });
        
        // Upload versions
        const folderPrefix = 'catalog-products';
        
        await adapter.upload({
          filename: filenames.thumbnail,
          buffer: processed.buffers.thumbnail,
          mimetype: 'image/webp',
          subfolder: `${folderPrefix}/thumbnails`,
        });
        
        await adapter.upload({
          filename: filenames.medium,
          buffer: processed.buffers.medium,
          mimetype: 'image/webp',
          subfolder: `${folderPrefix}/medium`,
        });
        
        // Build URLs
        const thumbnailUrl = adapter.getUrl(`${folderPrefix}/thumbnails/${filenames.thumbnail}`);
        const mediumUrl = adapter.getUrl(`${folderPrefix}/medium/${filenames.medium}`);
        
        // Update database
        await pool.query(`
          UPDATE catalog.product_images
          SET thumbnail_url = $1, medium_url = $2
          WHERE id = $3
        `, [thumbnailUrl, mediumUrl, id]);
        
        processedCount++;
        console.log(`   ✅ [${id}] Gotowe! (${processedCount}/${totalImages})`);
        
      } catch (error: any) {
        console.error(`   ❌ [${id}] Błąd:`, error.message);
        errorCount++;
      }
    }
    
    // Progress summary
    console.log(`\n   📊 Postęp: ${processedCount}/${totalImages} OK, ${errorCount} błędów`);
    
    // Pause between chunks to avoid CPU timeout
    if (offset + CHUNK_SIZE < totalImages) {
      console.log(`   ⏸️  Pauza ${DELAY_MS}ms przed następnym chunkiem...`);
      await new Promise(resolve => setTimeout(resolve, DELAY_MS));
    }
  }
  
  console.log(`\n✅ Migracja catalog.product_images zakończona: ${processedCount} OK, ${errorCount} błędów`);
}

async function regenerateAllProducts() {
  console.log('\n🔄 === REGENERACJA PRODUKTÓW ===\n');
  console.log('ℹ️  Produkty zostaną automatycznie zregenerowane przy następnym użyciu macierzy');
  console.log('ℹ️  Możesz ręcznie zregenerować przez UI: product-matrices/[id]/edit → "Generuj produkty"');
  
  // For now, we skip automatic regeneration to avoid complexity
  // User can manually regenerate from UI if needed
  console.log('\n⏭️  Pomijam automatyczną regenerację - użyj UI jeśli potrzebne');
}

async function regenerateAllSets() {
  console.log('\n🔄 === REGENERACJA ZESTAWÓW ===\n');
  console.log('ℹ️  Zestawy zostaną automatycznie zregenerowane przy następnym użyciu macierzy');
  console.log('ℹ️  Możesz ręcznie zregenerować przez UI: set-matrices/[id]/edit → "Generuj zestawy"');
  
  // For now, we skip automatic regeneration to avoid complexity
  // User can manually regenerate from UI if needed
  console.log('\n⏭️  Pomijam automatyczną regenerację - użyj UI jeśli potrzebne');
}

async function main() {
  console.log('🚀 === MIGRATION SCRIPT: Generowanie miniatur dla istniejących zdjęć ===\n');
  
  try {
    // Step 1: Migrate product matrices
    await migrateProductMatrices();
    
    // Step 2: Migrate set matrices
    await migrateSetMatrices();
    
    // Step 3: Migrate catalog.product_images directly
    await migrateCatalogImages();
    
    // Step 4: Regenerate all products (informational only)
    await regenerateAllProducts();
    
    // Step 5: Regenerate all sets (informational only)
    await regenerateAllSets();
    
    console.log('\n✅ === MIGRACJA ZAKOŃCZONA POMYŚLNIE ===\n');
    console.log('Wszystkie zdjęcia w macierzach i katalogu mają teraz miniatury!');
    
  } catch (error: any) {
    console.error('\n❌ === BŁĄD PODCZAS MIGRACJI ===');
    console.error(error);
    process.exit(1);
  } finally {
    await pool.end();
  }
}

// Run migration
main();
