import { useState, useMemo, useEffect, useCallback } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm, useWatch } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Combobox } from "@/components/ui/combobox";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { ArrowLeft, Save, Plus, Lock, AlertTriangle, Upload, X, Loader2, Star, Image as ImageIcon } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { DictionaryComboboxWithAdd } from "@/components/dictionary-combobox-with-add";
import { useDropzone } from "react-dropzone";

interface Material {
  id: number;
  groupId: number | null;
  name: string;
  internalCode: string;
  supplierCode?: string;
  description?: string;
  unitOfMeasure?: string;
  price?: number;
  quantity?: string;
  isActive: boolean;
  // Galeria zdjęć
  gallery?: string[];
  primaryImageUrl?: string;
  // Formatki-specific fields
  cz1?: string;
  cz2?: string;
  length?: number;
  width?: number;
  thickness?: number;
  boardCode?: string;
  edgingCode?: string;
  colorCode?: string;
  isDrilled?: boolean;
  // Koszt materiału (wyliczany)
  materialCost?: number;
  boardPrice?: number;
  calculatedMaterialCost?: number;
}

interface MaterialGroup {
  id: number;
  categoryId: number;
  name: string;
  description: string | null;
}

interface Category {
  id: number;
  code: string;
  name: string;
}

const baseMaterialSchema = z.object({
  groupId: z.string().optional(),
  name: z.string().min(1, "Nazwa jest wymagana"),
  internalCode: z.string().min(1, "Kod wewnętrzny jest wymagany"),
  supplierCode: z.string().optional(),
  description: z.string().optional(),
  unitOfMeasure: z.string().optional(),
  price: z.string().optional(),
  quantity: z.string().optional(),
  isActive: z.boolean(),
});

const formatkaMaterialSchema = baseMaterialSchema.extend({
  cz1: z.string().optional(),
  cz2: z.string().optional(),
  length: z.string().optional(),
  width: z.string().optional(),
  thickness: z.string().optional(),
  boardCode: z.string().optional(),
  edgingCode: z.string().optional(),
  colorCode: z.string().optional(),
  isDrilled: z.boolean().optional(),
});

type MaterialFormData = z.infer<typeof baseMaterialSchema>;
type FormatkaMaterialFormData = z.infer<typeof formatkaMaterialSchema>;

function GeneratedFormatkaName({ control }: { control: any }) {
  const cz1 = useWatch({ control, name: "cz1" });
  const cz2 = useWatch({ control, name: "cz2" });
  const length = useWatch({ control, name: "length" });
  const width = useWatch({ control, name: "width" });
  const colorCode = useWatch({ control, name: "colorCode" });

  const generatedName = useMemo(() => {
    let name = '';
    if (cz1) {
      name += cz1;
    }
    if (cz2) {
      name += (name ? '-' : '') + cz2;
    }
    if (length && width) {
      name += (name ? '-' : '') + `${length}×${width}`;
    }
    if (colorCode) {
      name += '-' + colorCode;
    }
    return name || '(brak - uzupełnij parametry)';
  }, [cz1, cz2, length, width, colorCode]);

  return <span data-testid="text-generated-formatka-name">{generatedName}</span>;
}

export default function WarehouseMaterialEdit() {
  const params = useParams<{ category: string; id: string }>();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const materialId = parseInt(params.id || "0");
  const category = params.category || "";

  // Dialog state
  const [isGroupDialogOpen, setIsGroupDialogOpen] = useState(false);
  const [newGroupName, setNewGroupName] = useState("");
  const [newGroupDescription, setNewGroupDescription] = useState("");

  // Gallery state
  const [gallery, setGallery] = useState<string[]>([]);
  const [primaryImageUrl, setPrimaryImageUrl] = useState<string | null>(null);
  const [isUploading, setIsUploading] = useState(false);

  // Fetch category data
  const { data: categoryData, isLoading: categoryLoading } = useQuery<Category>({
    queryKey: [`/api/warehouse/categories/by-code/${category}`],
  });

  // Fetch groups for this category
  const { data: groups = [], isLoading: groupsLoading } = useQuery<MaterialGroup[]>({
    queryKey: [`/api/warehouse/categories/${categoryData?.id}/groups`],
    enabled: !!categoryData?.id,
  });

  // Fetch units from dictionary
  const { data: units = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=unit"],
  });

  // Load dictionaries for formatki category
  const { data: cz1Dictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=component_cz1"],
    enabled: category === 'formatki',
  });

  const { data: cz2Dictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=component_cz2"],
    enabled: category === 'formatki',
  });

  const { data: colorDictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=color"],
    enabled: category === 'formatki',
  });

  // Słowniki wymiarów dla formatek
  const { data: lengthDictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=dimension_length"],
    enabled: category === 'formatki',
  });

  const { data: widthDictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=dimension_width"],
    enabled: category === 'formatki',
  });

  // Sprawdź czy formatka jest używana w BOM lub produkcji
  const { data: formatkaUsage, isLoading: usageLoading } = useQuery<{
    isUsed: boolean;
    bomCount: number;
    productionCount: number;
  }>({
    queryKey: [`/api/warehouse/stock-panels/${materialId}/usage-check`],
    enabled: category === 'formatki' && !!materialId,
  });

  const isFormatkaUsed = formatkaUsage?.isUsed ?? false;
  const isFormatkaEditable = !isFormatkaUsed;
  const isFormatkaCategory = category === 'formatki';

  // Pobierz płyty z magazynu dla formatek
  const { data: plytyMaterials = [] } = useQuery<any[]>({
    queryKey: ["/api/warehouse/materials/search", { category: "plyty", limit: 100 }],
    enabled: category === 'formatki',
    select: (data: any) => data?.materials || [],
  });

  // Pobierz obrzeża z magazynu dla formatek
  const { data: obrzezaMaterials = [] } = useQuery<any[]>({
    queryKey: ["/api/warehouse/materials/search", { category: "obrzeza", limit: 100 }],
    enabled: category === 'formatki',
    select: (data: any) => data?.materials || [],
  });

  // Dla formatek pobieramy dane z stock-panels, dla innych z materials
  const { data: material, isLoading } = useQuery<Material>({
    queryKey: category === 'formatki' 
      ? [`/api/warehouse/stock-panels/${materialId}`]
      : [`/api/warehouse/materials/${materialId}`, { category }],
    enabled: !!materialId,
    select: (data: any) => {
      // Normalizuj dane z stock-panels do formatu Material
      if (category === 'formatki' && data) {
        return {
          id: data.id,
          groupId: data.groupId,
          name: data.generatedName || data.name,
          internalCode: data.generatedName || data.internalCode,
          description: data.notes,
          unitOfMeasure: data.unitOfMeasure,
          price: data.price,
          isActive: data.isActive,
          cz1: data.cz1,
          cz2: data.cz2,
          length: data.length,
          width: data.width,
          thickness: data.thickness,
          boardCode: data.boardCode,
          edgingCode: data.edgingCode,
          colorCode: data.colorCode,
          isDrilled: data.isDrilled,
          materialCost: data.materialCost,
          boardPrice: data.boardPrice,
          calculatedMaterialCost: data.calculatedMaterialCost,
        } as Material;
      }
      return data;
    },
  });

  // Inicjalizacja galerii gdy materiał się załaduje
  useEffect(() => {
    if (material) {
      setGallery(material.gallery || []);
      setPrimaryImageUrl(material.primaryImageUrl || null);
    }
  }, [material]);

  // Upload zdjęcia
  const uploadImageMutation = useMutation({
    mutationFn: async (file: File) => {
      const formData = new FormData();
      formData.append("image", file);
      
      const response = await fetch(`/api/warehouse/materials/${materialId}/images`, {
        method: "POST",
        body: formData,
        credentials: "include",
      });
      
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "Błąd uploadu");
      }
      
      return response.json();
    },
    onSuccess: (data) => {
      setGallery(prev => [...prev, data.url]);
      if (!primaryImageUrl) {
        setPrimaryImageUrl(data.url);
      }
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
      toast({ title: "Zdjęcie dodane" });
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd uploadu",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Usuń zdjęcie
  const deleteImageMutation = useMutation({
    mutationFn: async (imageUrl: string) => {
      return apiRequest("DELETE", `/api/warehouse/materials/${materialId}/images`, { imageUrl });
    },
    onSuccess: (_, imageUrl) => {
      setGallery(prev => prev.filter(url => url !== imageUrl));
      if (primaryImageUrl === imageUrl) {
        const remaining = gallery.filter(url => url !== imageUrl);
        setPrimaryImageUrl(remaining.length > 0 ? remaining[0] : null);
      }
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
      toast({ title: "Zdjęcie usunięte" });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć zdjęcia",
        variant: "destructive",
      });
    },
  });

  // Ustaw główne zdjęcie
  const setPrimaryImageMutation = useMutation({
    mutationFn: async (imageUrl: string) => {
      return apiRequest("POST", `/api/warehouse/materials/${materialId}/primary-image`, { imageUrl });
    },
    onSuccess: (_, imageUrl) => {
      setPrimaryImageUrl(imageUrl);
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
      toast({ title: "Główne zdjęcie ustawione" });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się ustawić głównego zdjęcia",
        variant: "destructive",
      });
    },
  });

  // Dropzone config
  const onDrop = useCallback(async (acceptedFiles: File[]) => {
    setIsUploading(true);
    for (const file of acceptedFiles) {
      await uploadImageMutation.mutateAsync(file);
    }
    setIsUploading(false);
  }, [uploadImageMutation]);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      'image/*': ['.jpeg', '.jpg', '.png', '.webp', '.gif'],
    },
    multiple: true,
    disabled: isUploading,
  });

  const form = useForm<FormatkaMaterialFormData>({
    resolver: zodResolver(category === 'formatki' ? formatkaMaterialSchema : baseMaterialSchema),
    defaultValues: {
      groupId: "",
      name: "",
      internalCode: "",
      supplierCode: "",
      description: "",
      unitOfMeasure: "",
      price: "",
      quantity: "",
      isActive: true,
      cz1: "",
      cz2: "",
      length: "",
      width: "",
      thickness: "",
      boardCode: "",
      edgingCode: "",
      colorCode: "",
      isDrilled: false,
    },
    values: material
      ? {
          groupId: material.groupId?.toString() || "",
          name: material.name || "",
          internalCode: material.internalCode || "",
          supplierCode: material.supplierCode || "",
          description: material.description || "",
          unitOfMeasure: material.unitOfMeasure || "",
          price: material.price?.toString() || "",
          quantity: material.quantity?.toString() || "",
          isActive: material.isActive ?? true,
          cz1: material.cz1 || "",
          cz2: material.cz2 || "",
          length: material.length ? Math.floor(parseFloat(String(material.length))).toString() : "",
          width: material.width ? Math.floor(parseFloat(String(material.width))).toString() : "",
          thickness: material.thickness ? Math.floor(parseFloat(String(material.thickness))).toString() : "",
          boardCode: material.boardCode || "",
          edgingCode: material.edgingCode || "",
          colorCode: material.colorCode || "",
          isDrilled: material.isDrilled ?? false,
        }
      : undefined,
  });

  const updateMutation = useMutation({
    mutationFn: async (data: FormatkaMaterialFormData) => {
      // Dla formatek używamy innego endpointu (stock-panels)
      if (category === 'formatki') {
        const length = data.length ? parseInt(data.length) : null;
        const width = data.width ? parseInt(data.width) : null;
        
        // Generuj nazwę w formacie: CZ1-CZ2-długość×szerokość-KOLOR
        let generatedName = '';
        if (data.cz1) {
          generatedName += data.cz1;
        }
        if (data.cz2) {
          generatedName += (generatedName ? '-' : '') + data.cz2;
        }
        if (length && width) {
          generatedName += (generatedName ? '-' : '') + `${length}×${width}`;
        }
        if (data.colorCode) {
          generatedName += '-' + data.colorCode;
        }
        
        return apiRequest("PATCH", `/api/warehouse/stock-panels/${materialId}`, {
          generatedName,
          cz1: data.cz1 || null,
          cz2: data.cz2 || null,
          length,
          width,
          thickness: data.thickness ? parseInt(data.thickness) : null,
          boardCode: data.boardCode || null,
          edgingCode: data.edgingCode || null,
          colorCode: data.colorCode || null,
          groupId: data.groupId ? parseInt(data.groupId) : null,
          isActive: data.isActive,
          unit: data.unitOfMeasure || null,
          price: data.price ? parseFloat(data.price) : null,
          notes: data.description || null,
          isDrilled: data.isDrilled ?? false,
        });
      }
      
      // Dla innych kategorii używamy standardowego endpointu materials
      const basePayload = {
        groupId: data.groupId ? parseInt(data.groupId) : null,
        name: data.name,
        internalCode: data.internalCode,
        supplierCode: data.supplierCode || null,
        description: data.description || null,
        unitOfMeasure: data.unitOfMeasure || null,
        price: data.price ? parseFloat(data.price) : null,
        quantity: data.quantity ? parseFloat(data.quantity) : null,
        isActive: data.isActive,
      };

      return apiRequest("PATCH", `/api/warehouse/materials/${materialId}?category=${category}`, basePayload);
    },
    onSuccess: () => {
      if (category === 'formatki') {
        queryClient.invalidateQueries({ queryKey: [`/api/warehouse/stock-panels/${materialId}`] });
        queryClient.invalidateQueries({ 
          predicate: (query) => 
            Array.isArray(query.queryKey) && 
            typeof query.queryKey[0] === 'string' && 
            query.queryKey[0].includes('/api/warehouse/materials/search')
        });
      } else {
        queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
        queryClient.invalidateQueries({ 
          predicate: (query) => 
            Array.isArray(query.queryKey) && 
            typeof query.queryKey[0] === 'string' && 
            query.queryKey[0].startsWith('/api/warehouse/materials')
        });
      }
      toast({ title: category === 'formatki' ? "Formatka zaktualizowana pomyślnie" : "Materiał zaktualizowany pomyślnie" });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się zaktualizować materiału",
        variant: "destructive",
      });
    },
  });

  const createGroupMutation = useMutation({
    mutationFn: async (data: { name: string; description: string }): Promise<MaterialGroup> => {
      const response = await apiRequest("POST", `/api/warehouse/categories/${categoryData?.id}/groups`, data);
      return response.json();
    },
    onSuccess: (newGroup: MaterialGroup) => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/categories/${categoryData?.id}/groups`] });
      toast({ title: "Grupa utworzona pomyślnie" });
      // Automatically select the new group
      form.setValue("groupId", newGroup.id.toString());
      setIsGroupDialogOpen(false);
      setNewGroupName("");
      setNewGroupDescription("");
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się utworzyć grupy",
        variant: "destructive",
      });
    },
  });

  const handleCreateGroup = () => {
    if (!categoryData?.id) {
      toast({
        title: "Błąd",
        description: "Kategoria nie jest załadowana",
        variant: "destructive",
      });
      return;
    }
    if (!newGroupName.trim()) {
      toast({
        title: "Błąd",
        description: "Nazwa grupy jest wymagana",
        variant: "destructive",
      });
      return;
    }
    createGroupMutation.mutate({
      name: newGroupName,
      description: newGroupDescription,
    });
  };

  const onSubmit = (data: MaterialFormData) => {
    updateMutation.mutate(data);
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6 max-w-4xl">
        <div className="text-center">Ładowanie...</div>
      </div>
    );
  }

  if (!material) {
    return (
      <div className="container mx-auto p-6 max-w-4xl">
        <Card>
          <CardContent className="pt-6">
            <p className="text-center text-muted-foreground">Materiał nie znaleziony</p>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 max-w-4xl">
      <div className="mb-6">
        <Button
          variant="ghost"
          onClick={() => navigate(`/warehouse/${category}/${materialId}`)}
          className="mb-4"
          data-testid="button-back"
        >
          <ArrowLeft className="w-4 h-4 mr-2" />
          Powrót
        </Button>

        <h1 className="text-3xl font-bold" data-testid="text-page-title">
          Edytuj materiał
        </h1>
        <p className="text-muted-foreground">Edycja: {material.name}</p>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane materiału</CardTitle>
        </CardHeader>
        <CardContent>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
              <FormField
                control={form.control}
                name="groupId"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Grupa</FormLabel>
                    <div className="flex gap-2">
                      <FormControl>
                        <Select value={field.value || "null"} onValueChange={(value) => field.onChange(value === "null" ? "" : value)}>
                          <SelectTrigger data-testid="select-group">
                            <SelectValue placeholder="Wybierz grupę" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="null">Brak grupy</SelectItem>
                            {groups.map((group) => (
                              <SelectItem key={group.id} value={group.id.toString()}>
                                {group.name}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </FormControl>
                      <Button
                        type="button"
                        variant="outline"
                        size="icon"
                        onClick={() => setIsGroupDialogOpen(true)}
                        disabled={categoryLoading || groupsLoading}
                        data-testid="button-add-group"
                        title={categoryLoading || groupsLoading ? "Ładowanie..." : "Dodaj nową grupę"}
                      >
                        <Plus className="h-4 w-4" />
                      </Button>
                    </div>
                    <FormMessage />
                  </FormItem>
                )}
              />

              {/* Ukryj pola nazwy i kodu dla formatek - są generowane automatycznie */}
              {category !== 'formatki' && (
                <>
                  <FormField
                    control={form.control}
                    name="name"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Nazwa *</FormLabel>
                        <FormControl>
                          <Input {...field} data-testid="input-name" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <div className={`grid gap-4 ${category === 'opakowania' || category === 'produkty-spakowane' ? 'grid-cols-1' : 'grid-cols-2'}`}>
                    <FormField
                      control={form.control}
                      name="internalCode"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Kod wewnętrzny *</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-internal-code" />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    {category !== 'opakowania' && category !== 'produkty-spakowane' && (
                      <FormField
                        control={form.control}
                        name="supplierCode"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Kod dostawcy</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-supplier-code" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    )}
                  </div>
                </>
              )}

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} data-testid="input-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              {/* Formatki-specific fields */}
              {category === 'formatki' && (
                <>
                  {/* Alert jeśli formatka jest używana */}
                  {isFormatkaUsed && (
                    <Alert variant="destructive" className="mb-4">
                      <Lock className="h-4 w-4" />
                      <AlertTitle>Formatka jest używana</AlertTitle>
                      <AlertDescription>
                        Ta formatka jest używana w {formatkaUsage?.bomCount || 0} BOM(ach) i {formatkaUsage?.productionCount || 0} zleceniu(ach) produkcji. 
                        Edycja parametrów nie jest możliwa.
                      </AlertDescription>
                    </Alert>
                  )}

                  {/* Podgląd wygenerowanej nazwy */}
                  <div className="p-4 bg-muted rounded-lg">
                    <FormLabel className="text-sm font-medium">Nazwa formatki (generowana automatycznie)</FormLabel>
                    <div className="text-lg font-mono mt-1 flex items-center gap-2">
                      <GeneratedFormatkaName control={form.control} />
                    </div>
                    <p className="text-xs text-muted-foreground mt-1">
                      Format: CZ1-CZ2-długość×szerokość-KOLOR
                    </p>
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <FormField
                      control={form.control}
                      name="cz1"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>CZ1 (opcjonalnie)</FormLabel>
                          <FormControl>
                            <DictionaryComboboxWithAdd
                              items={cz1Dictionaries}
                              value={field.value || ""}
                              onChange={field.onChange}
                              dictionaryType="component_cz1"
                              placeholder="Wybierz CZ1..."
                              searchPlaceholder="Szukaj CZ1..."
                              emptyText="Nie znaleziono CZ1."
                              testId="combobox-formatka-cz1"
                              valueField="code"
                              displayField="readableName"
                              disabled={!isFormatkaEditable}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={form.control}
                      name="cz2"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>CZ2 (opcjonalnie)</FormLabel>
                          <FormControl>
                            <DictionaryComboboxWithAdd
                              items={cz2Dictionaries}
                              value={field.value || ""}
                              onChange={field.onChange}
                              dictionaryType="component_cz2"
                              placeholder="Wybierz CZ2..."
                              searchPlaceholder="Szukaj CZ2..."
                              emptyText="Nie znaleziono CZ2."
                              testId="combobox-formatka-cz2"
                              valueField="code"
                              displayField="readableName"
                              disabled={!isFormatkaEditable}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                  <div className="grid grid-cols-3 gap-4">
                    <FormField
                      control={form.control}
                      name="length"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Długość (mm) *</FormLabel>
                          <FormControl>
                            <DictionaryComboboxWithAdd
                              items={lengthDictionaries}
                              value={field.value || ""}
                              onChange={field.onChange}
                              dictionaryType="dimension_length"
                              placeholder="Wybierz długość..."
                              searchPlaceholder="Szukaj długości..."
                              emptyText="Nie znaleziono długości."
                              testId="combobox-formatka-length"
                              valueField="code"
                              displayField="readableName"
                              disabled={!isFormatkaEditable}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={form.control}
                      name="width"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Szerokość (mm) *</FormLabel>
                          <FormControl>
                            <DictionaryComboboxWithAdd
                              items={widthDictionaries}
                              value={field.value || ""}
                              onChange={field.onChange}
                              dictionaryType="dimension_width"
                              placeholder="Wybierz szerokość..."
                              searchPlaceholder="Szukaj szerokości..."
                              emptyText="Nie znaleziono szerokości."
                              testId="combobox-formatka-width"
                              valueField="code"
                              displayField="readableName"
                              disabled={!isFormatkaEditable}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={form.control}
                      name="thickness"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Grubość (mm)</FormLabel>
                          <FormControl>
                            <Input
                              {...field}
                              type="number"
                              step="1"
                              min="1"
                              placeholder="18"
                              data-testid="input-formatka-thickness"
                              disabled={!isFormatkaEditable}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                  <div className="grid grid-cols-3 gap-4">
                    <FormField
                      control={form.control}
                      name="boardCode"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Płyta (opcjonalnie)</FormLabel>
                          <FormControl>
                            <Combobox
                              options={[
                                { value: "", label: "Brak płyty" },
                                ...plytyMaterials.map((plyta: any) => ({
                                  value: plyta.internalCode || plyta.name,
                                  label: plyta.name + (plyta.internalCode && plyta.internalCode !== plyta.name ? ` (${plyta.internalCode})` : '')
                                }))
                              ]}
                              value={field.value || ""}
                              onChange={field.onChange}
                              placeholder="Wybierz płytę..."
                              searchPlaceholder="Szukaj płyty..."
                              emptyText="Nie znaleziono płyty"
                              className="w-full"
                              disabled={!isFormatkaEditable}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={form.control}
                      name="edgingCode"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Obrzeże (opcjonalnie)</FormLabel>
                          <FormControl>
                            <Combobox
                              options={[
                                { value: "", label: "Brak obrzeża" },
                                ...obrzezaMaterials.map((obrzeze: any) => ({
                                  value: obrzeze.internalCode || obrzeze.name,
                                  label: obrzeze.name + (obrzeze.internalCode && obrzeze.internalCode !== obrzeze.name ? ` (${obrzeze.internalCode})` : '')
                                }))
                              ]}
                              value={field.value || ""}
                              onChange={field.onChange}
                              placeholder="Wybierz obrzeże..."
                              searchPlaceholder="Szukaj obrzeża..."
                              emptyText="Nie znaleziono obrzeża"
                              className="w-full"
                              disabled={!isFormatkaEditable}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={form.control}
                      name="colorCode"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Kod koloru (opcjonalnie)</FormLabel>
                          <FormControl>
                            <DictionaryComboboxWithAdd
                              items={colorDictionaries}
                              value={field.value || ""}
                              onChange={field.onChange}
                              dictionaryType="color"
                              placeholder="Wybierz kolor..."
                              searchPlaceholder="Szukaj koloru..."
                              emptyText="Nie znaleziono koloru."
                              testId="combobox-formatka-color-code"
                              valueField="code"
                              displayField="name"
                              disabled={!isFormatkaEditable}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                  
                  <FormField
                    control={form.control}
                    name="isDrilled"
                    render={({ field }) => (
                      <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3">
                        <div className="space-y-0.5">
                          <FormLabel>Nawiercona</FormLabel>
                          <p className="text-sm text-muted-foreground">
                            Formatka posiada nawierty
                          </p>
                        </div>
                        <FormControl>
                          <Switch
                            checked={field.value}
                            onCheckedChange={field.onChange}
                            data-testid="switch-is-drilled"
                          />
                        </FormControl>
                      </FormItem>
                    )}
                  />
                  
                  {/* Koszt materiału - wyliczany automatycznie */}
                  {material && (material.calculatedMaterialCost || material.boardPrice) && (
                    <div className="rounded-lg border bg-muted/50 p-4 space-y-3">
                      <h4 className="font-medium text-sm">Koszt materiału (wyliczony)</h4>
                      <div className="grid grid-cols-2 gap-4 text-sm">
                        <div>
                          <p className="text-muted-foreground">Cena płyty (za m²)</p>
                          <p className="font-medium">
                            {material.boardPrice ? `${Number(material.boardPrice).toFixed(2)} zł/m²` : '-'}
                          </p>
                        </div>
                        <div>
                          <p className="text-muted-foreground">Powierzchnia formatki</p>
                          <p className="font-medium">
                            {material.length && material.width 
                              ? `${((Number(material.length) * Number(material.width)) / 1000000).toFixed(4)} m²`
                              : '-'
                            }
                          </p>
                        </div>
                        <div className="col-span-2 pt-2 border-t">
                          <p className="text-muted-foreground">Wyliczony koszt materiału</p>
                          <p className="font-medium text-lg">
                            {material.calculatedMaterialCost 
                              ? `${Number(material.calculatedMaterialCost).toFixed(4)} zł`
                              : '-'
                            }
                          </p>
                        </div>
                      </div>
                    </div>
                  )}
                </>
              )}

              <div className="grid grid-cols-3 gap-4">
                <FormField
                  control={form.control}
                  name="unitOfMeasure"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Jednostka miary</FormLabel>
                      <FormControl>
                        <Combobox
                          options={units.map((unit: any) => ({
                            value: unit.code,
                            label: unit.name
                          }))}
                          value={field.value || ""}
                          onChange={field.onChange}
                          placeholder="Wybierz jednostkę..."
                          searchPlaceholder="Szukaj jednostki..."
                          emptyText="Nie znaleziono jednostki"
                          className="w-full"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="price"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="flex items-center gap-2">
                        Cena (zł)
                        {isFormatkaCategory && <Lock className="h-3 w-3 text-muted-foreground" />}
                      </FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          type="number"
                          step="0.01"
                          placeholder="0.00"
                          data-testid="input-price"
                          disabled={isFormatkaCategory}
                          value={isFormatkaCategory && material?.calculatedMaterialCost 
                            ? Number(material.calculatedMaterialCost).toFixed(2) 
                            : field.value}
                          className={isFormatkaCategory ? "bg-muted cursor-not-allowed" : ""}
                        />
                      </FormControl>
                      {isFormatkaCategory && (
                        <p className="text-xs text-muted-foreground">
                          Cena wyliczana automatycznie ze składników
                        </p>
                      )}
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormItem>
                  <FormLabel className="flex items-center gap-2">
                    Ilość
                    <Lock className="h-3 w-3 text-muted-foreground" />
                  </FormLabel>
                  <FormControl>
                    <Input 
                      value={material?.quantity || "0"} 
                      type="text"
                      disabled
                      className="bg-muted cursor-not-allowed"
                      data-testid="input-quantity-readonly" 
                    />
                  </FormControl>
                  <p className="text-xs text-muted-foreground">
                    Ilość zmienia się tylko przez inwentaryzację lub ruchy produkcyjne
                  </p>
                </FormItem>
              </div>

              <FormField
                control={form.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex items-center gap-4">
                    <FormLabel className="mt-2">Aktywny</FormLabel>
                    <FormControl>
                      <Switch
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        data-testid="switch-is-active"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              {/* Galeria zdjęć - tylko dla materiałów (nie formatek) */}
              {category !== 'formatki' && (
                <div className="space-y-4">
                  <Label className="flex items-center gap-2">
                    <ImageIcon className="h-4 w-4" />
                    Zdjęcia materiału
                  </Label>

                  {/* Dropzone */}
                  <div
                    {...getRootProps()}
                    className={`border-2 border-dashed rounded-lg p-6 text-center cursor-pointer transition-colors ${
                      isDragActive 
                        ? 'border-primary bg-primary/5' 
                        : 'border-muted-foreground/25 hover:border-primary/50'
                    } ${isUploading ? 'opacity-50 pointer-events-none' : ''}`}
                    data-testid="dropzone-images"
                  >
                    <input {...getInputProps()} data-testid="input-images" />
                    {isUploading ? (
                      <div className="flex flex-col items-center gap-2">
                        <Loader2 className="h-8 w-8 animate-spin text-primary" />
                        <p className="text-sm text-muted-foreground">Wysyłanie zdjęć...</p>
                      </div>
                    ) : isDragActive ? (
                      <div className="flex flex-col items-center gap-2">
                        <Upload className="h-8 w-8 text-primary" />
                        <p className="text-sm text-primary">Upuść zdjęcia tutaj...</p>
                      </div>
                    ) : (
                      <div className="flex flex-col items-center gap-2">
                        <Upload className="h-8 w-8 text-muted-foreground" />
                        <p className="text-sm text-muted-foreground">
                          Przeciągnij i upuść zdjęcia lub kliknij aby wybrać
                        </p>
                        <p className="text-xs text-muted-foreground">
                          JPG, PNG, WebP, GIF
                        </p>
                      </div>
                    )}
                  </div>

                  {/* Galeria zdjęć */}
                  {gallery.length > 0 && (
                    <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-4">
                      {gallery.map((imageUrl, index) => (
                        <div
                          key={index}
                          className="relative group rounded-lg overflow-hidden border bg-muted aspect-square"
                          data-testid={`gallery-image-${index}`}
                        >
                          <img
                            src={imageUrl}
                            alt={`Zdjęcie ${index + 1}`}
                            className="w-full h-full object-cover"
                          />
                          
                          {/* Primary indicator */}
                          {primaryImageUrl === imageUrl && (
                            <div className="absolute top-2 left-2">
                              <div className="bg-yellow-500 text-white px-2 py-0.5 rounded text-xs flex items-center gap-1">
                                <Star className="h-3 w-3 fill-current" />
                                Główne
                              </div>
                            </div>
                          )}

                          {/* Actions overlay */}
                          <div className="absolute inset-0 bg-black/60 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center gap-2">
                            {primaryImageUrl !== imageUrl && (
                              <Button
                                type="button"
                                size="sm"
                                variant="secondary"
                                onClick={() => setPrimaryImageMutation.mutate(imageUrl)}
                                disabled={setPrimaryImageMutation.isPending}
                                data-testid={`button-set-primary-${index}`}
                              >
                                <Star className="h-4 w-4" />
                              </Button>
                            )}
                            <Button
                              type="button"
                              size="sm"
                              variant="destructive"
                              onClick={() => deleteImageMutation.mutate(imageUrl)}
                              disabled={deleteImageMutation.isPending}
                              data-testid={`button-delete-image-${index}`}
                            >
                              <X className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}

              <div className="flex gap-4">
                <Button
                  type="submit"
                  disabled={updateMutation.isPending || (category === 'formatki' && !isFormatkaEditable)}
                  data-testid="button-submit"
                >
                  <Save className="w-4 h-4 mr-2" />
                  {category === 'formatki' && !isFormatkaEditable ? 'Edycja zablokowana' : 'Zapisz zmiany'}
                </Button>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => navigate(`/warehouse/${category}/${materialId}`)}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
              </div>
            </form>
          </Form>
        </CardContent>
      </Card>

      {/* Dialog dodawania nowej grupy */}
      <Dialog open={isGroupDialogOpen} onOpenChange={setIsGroupDialogOpen}>
        <DialogContent data-testid="dialog-add-group">
          <DialogHeader>
            <DialogTitle>Dodaj nową grupę</DialogTitle>
            <DialogDescription>
              Utwórz nową grupę dla kategorii {categoryData?.name || "tej kategorii"}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="group-name">Nazwa grupy *</Label>
              <Input
                id="group-name"
                value={newGroupName}
                onChange={(e) => setNewGroupName(e.target.value)}
                placeholder="np. Zawiasy VB"
                data-testid="input-new-group-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="group-description">Opis</Label>
              <Textarea
                id="group-description"
                value={newGroupDescription}
                onChange={(e) => setNewGroupDescription(e.target.value)}
                placeholder="Opcjonalny opis grupy"
                data-testid="input-new-group-description"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setIsGroupDialogOpen(false);
                setNewGroupName("");
                setNewGroupDescription("");
              }}
              data-testid="button-cancel-add-group"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleCreateGroup}
              disabled={createGroupMutation.isPending}
              data-testid="button-confirm-add-group"
            >
              <Plus className="w-4 h-4 mr-2" />
              Dodaj grupę
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
