import { useParams, useLocation, Link } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { safeGoBack } from "@/lib/navigation";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { ArrowLeft, Save, CheckCircle, Copy, Download, ExternalLink, X, Plus, Minus } from "lucide-react";
import { useState } from "react";

interface InventoryCountItem {
  id: number;
  inventoryCountId: number;
  materialId: number;
  systemQuantity: string;
  countedQuantity: string | null;
  difference: string | null;
  notes: string | null;
  materialName: string;
  materialCode: string;
  unitOfMeasure: string;
  groupName: string | null;
  updatedAt: string;
  itemType: string;
  itemId: number;
  itemName: string;
  itemCode: string | null;
  itemCategory: string | null;
  dimensions: string | null;
  colorCode: string | null;
}

interface InventoryCount {
  id: number;
  name: string;
  status: string;
  notes: string | null;
  createdBy: string;
  createdAt: string;
  finalizedBy: string | null;
  finalizedAt: string | null;
}

export default function WarehouseInventoryCountDetailPage() {
  const params = useParams<{ id: string }>();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const countId = parseInt(params.id || "0");

  const [editedQuantities, setEditedQuantities] = useState<Record<number, string>>({});

  // Format quantity based on unit of measure
  const formatQuantity = (quantity: string | null, unit: string): string => {
    if (quantity === null || quantity === '') return '-';
    const num = parseFloat(quantity);
    if (isNaN(num)) return '-';
    // For pieces (szt), show as integer
    if (unit === 'szt') {
      return Math.floor(num).toString();
    }
    // For other units, show up to 3 decimal places, removing trailing zeros
    return parseFloat(num.toFixed(3)).toString();
  };

  // Generate link to warehouse item detail page
  const getItemLink = (item: InventoryCountItem): string | null => {
    if (!item.itemId) return null;
    
    if (item.itemType === 'material') {
      // Materials: plyty, okucia, tkaniny, sruby, pianki, obrzeza
      if (!item.itemCategory) return null;
      return `/warehouse/${item.itemCategory}/${item.itemId}`;
    } else if (item.itemType === 'stock_panel') {
      // Stock panels (formatki) - always use formatki route
      return `/warehouse/formatki/${item.itemId}`;
    } else if (item.itemType === 'packaging_material') {
      // Packaging materials (opakowania) - always use opakowania route
      return `/warehouse/opakowania/${item.itemId}`;
    } else if (item.itemType === 'packed_product') {
      // Packed products (produkty-spakowane) - always use produkty-spakowane route
      return `/warehouse/produkty-spakowane/${item.itemId}`;
    }
    
    return null;
  };

  const { data, isLoading } = useQuery<{ inventoryCount: InventoryCount; items: InventoryCountItem[] }>({
    queryKey: [`/api/warehouse/inventory-counts/${countId}`],
    enabled: !!countId,
  });

  const updateItemMutation = useMutation({
    mutationFn: async ({ itemId, countedQuantity }: { itemId: number; countedQuantity: string }) => {
      // Validate before sending to backend
      const parsed = parseFloat(countedQuantity);
      if (isNaN(parsed) || parsed < 0) {
        throw new Error("Wartość musi być poprawną liczbą nieujemną");
      }
      
      return apiRequest("PATCH", `/api/warehouse/inventory-counts/${countId}/items/${itemId}`, {
        countedQuantity: parsed,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/inventory-counts/${countId}`] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować ilości",
        variant: "destructive",
      });
    },
  });

  const deleteItemMutation = useMutation({
    mutationFn: async (itemId: number) => {
      return apiRequest("DELETE", `/api/warehouse/inventory-counts/${countId}/items/${itemId}`, {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/inventory-counts/${countId}`] });
      toast({ title: "Sukces", description: "Pozycja usunięta ze spisu" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć pozycji",
        variant: "destructive",
      });
    },
  });

  const finalizeMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/warehouse/inventory-counts/${countId}/finalize`, {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/inventory-counts/${countId}`] });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny zamknięty i magazyn przeliczony",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zamknąć spisu",
        variant: "destructive",
      });
    },
  });

  const duplicateMutation = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("POST", `/api/warehouse/inventory-counts/${countId}/duplicate`, {
        name: `${data?.inventoryCount.name} (kopia)`,
      });
      return await response.json();
    },
    onSuccess: (result: any) => {
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny zduplikowany",
      });
      setLocation(`/warehouse/inventory-counts/${result.inventoryCount.id}`);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zduplikować spisu",
        variant: "destructive",
      });
    },
  });

  const handleSaveItem = (itemId: number, isUnitPieces: boolean) => {
    const quantity = editedQuantities[itemId];
    if (quantity !== undefined && quantity.trim() !== "") {
      // Validate number before saving
      const parsed = parseFloat(quantity);
      if (isNaN(parsed) || parsed < 0) {
        toast({
          title: "Błąd",
          description: "Wartość musi być poprawną liczbą nieujemną",
          variant: "destructive",
        });
        return;
      }
      
      let finalQuantity = quantity;
      // For pieces (szt), ensure integer only
      if (isUnitPieces) {
        finalQuantity = Math.floor(parsed).toString();
      }
      
      updateItemMutation.mutate(
        { itemId, countedQuantity: finalQuantity },
        {
          onError: () => {
            // Revert on error
            setEditedQuantities(prev => {
              const next = { ...prev };
              delete next[itemId];
              return next;
            });
          },
          onSettled: async () => {
            // Wait for query refetch to complete before clearing edited state
            await queryClient.refetchQueries({ queryKey: [`/api/warehouse/inventory-counts/${countId}`] });
            
            setEditedQuantities(prev => {
              const next = { ...prev };
              delete next[itemId];
              return next;
            });
          }
        }
      );
    }
  };

  const handleQuantityChange = (itemId: number, value: string) => {
    setEditedQuantities(prev => ({ ...prev, [itemId]: value }));
  };

  const handleQuickChange = (itemId: number, delta: number, isUnitPieces: boolean, item: InventoryCountItem) => {
    const currentValue = editedQuantities[itemId] !== undefined ? editedQuantities[itemId] : (item.countedQuantity || "0");
    const current = parseFloat(currentValue) || 0;
    let newValue = current + delta;
    
    // Ensure non-negative
    if (newValue < 0) newValue = 0;
    
    // For pieces, round to integer
    if (isUnitPieces) {
      newValue = Math.floor(newValue);
    }
    
    // Validate before update
    if (isNaN(newValue) || !isFinite(newValue) || newValue < 0) {
      toast({
        title: "Błąd",
        description: "Nieprawidłowa wartość",
        variant: "destructive",
      });
      return;
    }
    
    const newValueStr = newValue.toString();
    
    // Update local state only - user must click Save button to persist
    setEditedQuantities(prev => ({ ...prev, [itemId]: newValueStr }));
  };

  const handleSetZero = (itemId: number) => {
    // Update local state only - user must click Save button to persist
    setEditedQuantities(prev => ({ ...prev, [itemId]: "0" }));
  };

  if (isLoading || !data) {
    return (
      <WarehouseLayout category="inventory-counts">
        <div className="p-6">Ładowanie...</div>
      </WarehouseLayout>
    );
  }

  const { inventoryCount, items } = data;
  const isDraft = inventoryCount.status === 'draft';

  // Check if item has valid counted quantity (not null, not empty string, and is a valid finite number >= 0)
  const hasValidCountedQuantity = (item: InventoryCountItem): boolean => {
    if (item.countedQuantity === null || item.countedQuantity === '') return false;
    const num = parseFloat(item.countedQuantity);
    return Number.isFinite(num) && num >= 0;
  };
  
  const countedItems = items.filter(hasValidCountedQuantity).length;
  const uncountedItems = items.filter(i => !hasValidCountedQuantity(i)).length;
  const hasUnsavedEdits = Object.keys(editedQuantities).length > 0;
  const canFinalize = isDraft && uncountedItems === 0 && items.length > 0 && !hasUnsavedEdits;

  return (
    <WarehouseLayout category="inventory-counts">
      <div className="space-y-6 p-6">
        <div className="flex items-center gap-4">
          <Button variant="ghost" onClick={() => safeGoBack("/warehouse/inventory-counts", setLocation)} data-testid="button-back">
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div className="flex-1">
            <h1 className="text-3xl font-bold">{inventoryCount.name}</h1>
            <p className="text-muted-foreground">
              Utworzony przez {inventoryCount.createdBy} • {new Date(inventoryCount.createdAt).toLocaleString("pl-PL")}
            </p>
          </div>
          <Badge variant={isDraft ? "secondary" : "default"} data-testid="badge-status">
            {isDraft ? "Roboczy" : "Zamknięty"}
          </Badge>
        </div>

        <div className="grid gap-4 md:grid-cols-3">
          <Card>
            <CardHeader>
              <CardTitle>Pozycje</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold">{items.length}</p>
              <p className="text-sm text-muted-foreground">wszystkich</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Spisane</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-green-600">{countedItems}</p>
              <p className="text-sm text-muted-foreground">pozycji spisanych</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Do spisania</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-orange-600">{uncountedItems}</p>
              <p className="text-sm text-muted-foreground">pozycji pozostało</p>
            </CardContent>
          </Card>
        </div>

        <div className="flex gap-2">
          {!isDraft && (
            <>
              <Button
                onClick={() => duplicateMutation.mutate()}
                disabled={duplicateMutation.isPending}
                data-testid="button-duplicate"
              >
                <Copy className="h-4 w-4 mr-2" />
                Duplikuj spis
              </Button>
              <Button
                onClick={() => {
                  window.location.href = `/api/warehouse/inventory-counts/${countId}/export/nexo`;
                }}
                variant="outline"
                data-testid="button-export-nexo"
              >
                <Download className="h-4 w-4 mr-2" />
                Eksport do Nexo (XML)
              </Button>
            </>
          )}
        </div>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 flex-wrap">
            <CardTitle>Pozycje spisu</CardTitle>
            {isDraft && items.length > 0 && (
              <Button
                onClick={() => finalizeMutation.mutate()}
                disabled={!canFinalize || finalizeMutation.isPending || deleteItemMutation.isPending || updateItemMutation.isPending}
                data-testid="button-finalize-top"
                className="flex-shrink-0"
                title={
                  hasUnsavedEdits ? "Zapisz wszystkie zmiany przed przeliczeniem" :
                  uncountedItems > 0 ? `Pozostało ${uncountedItems} pozycji do spisania` :
                  "Przelicz magazyn"
                }
              >
                <CheckCircle className="h-4 w-4 mr-2" />
                {finalizeMutation.isPending ? "Zamykanie..." : "Przelicz magazyn"}
              </Button>
            )}
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              {items.map((item, index) => {
                const hasEdited = editedQuantities[item.id] !== undefined;
                // Format the initial value from database to remove trailing zeros
                const formattedInitialValue = item.countedQuantity ? formatQuantity(item.countedQuantity, item.unitOfMeasure) : "";
                const displayedCountedQty = hasEdited ? editedQuantities[item.id] : formattedInitialValue;
                const itemLink = getItemLink(item);
                const isUnitPieces = item.unitOfMeasure === 'szt';
                
                // Calculate current difference based on edited or saved value
                const currentCountedValue = hasEdited ? editedQuantities[item.id] : item.countedQuantity;
                const currentCountedNum = currentCountedValue ? parseFloat(currentCountedValue) : null;
                const systemNum = parseFloat(item.systemQuantity);
                const currentDifference = (currentCountedNum !== null && !isNaN(currentCountedNum) && !isNaN(systemNum)) 
                  ? (currentCountedNum - systemNum).toString() 
                  : null;
                
                return (
                  <div
                    key={item.id}
                    className="p-3 border rounded-md"
                    data-testid={`item-${item.id}`}
                  >
                    <div className="flex items-center gap-3 flex-wrap">
                      <div className="w-12 flex-shrink-0 text-center">
                        <p className="text-sm font-semibold text-muted-foreground" data-testid={`row-number-${item.id}`}>
                          {index + 1}
                        </p>
                      </div>
                      <div className="flex-shrink-0 min-w-[200px]">
                        {itemLink ? (
                          <Link href={itemLink} data-testid={`link-item-detail-${item.id}`} className="inline-flex items-center gap-1 font-semibold hover:text-primary">
                            {item.itemName}
                            <ExternalLink className="h-3 w-3" data-testid={`icon-external-link-${item.id}`} />
                          </Link>
                        ) : (
                          <p className="font-semibold">{item.itemName}</p>
                        )}
                      </div>
                      <div className="flex items-center gap-2 flex-shrink-0 flex-wrap">
                        <span className="text-sm text-muted-foreground">Spisano:</span>
                        {isDraft ? (
                          <>
                            <div className="flex items-center gap-1">
                              <Button
                                variant="outline"
                                size="icon"
                                onClick={() => handleQuickChange(item.id, -1, isUnitPieces, item)}
                                data-testid={`button-decrement-${item.id}`}
                                className="h-9 w-9"
                              >
                                <Minus className="h-5 w-5" />
                              </Button>
                              <Input
                                type="text"
                                inputMode={isUnitPieces ? "numeric" : "decimal"}
                                placeholder={isUnitPieces ? "0 (tylko całkowite)" : "0.00"}
                                value={displayedCountedQty}
                                onChange={(e) => {
                                  let value = e.target.value.replace(',', '.');
                                  // For pieces, only allow integers (no decimal point)
                                  if (isUnitPieces) {
                                    value = value.replace(/[^0-9]/g, '');
                                  } else {
                                    // Allow digits and one decimal point
                                    const parts = value.split('.');
                                    if (parts.length > 2) {
                                      value = parts[0] + '.' + parts.slice(1).join('');
                                    }
                                    value = value.replace(/[^0-9.]/g, '');
                                  }
                                  handleQuantityChange(item.id, value);
                                }}
                                onKeyDown={(e) => {
                                  if (isUnitPieces && (e.key === '.' || e.key === ',')) {
                                    e.preventDefault();
                                  }
                                }}
                                onBlur={(e) => {
                                  // Validate on blur - ensure it's a valid number
                                  const value = e.target.value;
                                  const num = parseFloat(value);
                                  if (value !== '' && (isNaN(num) || num < 0)) {
                                    // Reset to 0 if invalid
                                    handleQuantityChange(item.id, '0');
                                  } else if (value !== '' && isUnitPieces) {
                                    // For pieces, ensure integer only
                                    const intValue = Math.floor(num).toString();
                                    if (value !== intValue) {
                                      handleQuantityChange(item.id, intValue);
                                    }
                                  }
                                }}
                                disabled={!isDraft}
                                data-testid={`input-counted-${item.id}`}
                                className={`w-24 h-9 text-center ${hasEdited ? 'bg-yellow-100 dark:bg-yellow-900/30 border-yellow-500' : ''}`}
                                title={isUnitPieces ? "Tylko liczby całkowite dla sztuk" : `Wartość w ${item.unitOfMeasure}`}
                              />
                              <Button
                                variant="outline"
                                size="icon"
                                onClick={() => handleQuickChange(item.id, 1, isUnitPieces, item)}
                                data-testid={`button-increment-${item.id}`}
                                className="h-9 w-9"
                              >
                                <Plus className="h-5 w-5" />
                              </Button>
                              <Button
                                variant="outline"
                                size="icon"
                                onClick={() => handleSetZero(item.id)}
                                data-testid={`button-zero-${item.id}`}
                                className="h-9 w-9 font-semibold"
                              >
                                0
                              </Button>
                            </div>
                            <div className="w-9 h-9 flex items-center justify-center flex-shrink-0">
                              <Button
                                size="icon"
                                variant="default"
                                onClick={() => handleSaveItem(item.id, isUnitPieces)}
                                disabled={updateItemMutation.isPending || !hasEdited}
                                data-testid={`button-save-${item.id}`}
                                className={`h-9 w-9 transition-opacity ${!hasEdited ? 'opacity-0 pointer-events-none' : 'opacity-100'}`}
                              >
                                <Save className="h-4 w-4" />
                              </Button>
                            </div>
                          </>
                        ) : (
                          <span className="font-semibold">{formatQuantity(item.countedQuantity, item.unitOfMeasure)} {item.unitOfMeasure}</span>
                        )}
                      </div>
                      <div className="flex-shrink-0">
                        <p className="text-sm text-muted-foreground">Stan systemowy</p>
                        <p className="font-semibold">{formatQuantity(item.systemQuantity, item.unitOfMeasure)} {item.unitOfMeasure}</p>
                      </div>
                      {currentDifference !== null && (
                        <div className="flex-shrink-0">
                          <p className="text-sm text-muted-foreground">Różnica</p>
                          <p className={`font-semibold ${parseFloat(currentDifference) > 0 ? "text-green-600" : parseFloat(currentDifference) < 0 ? "text-red-600" : ""}`}>
                            {parseFloat(currentDifference) > 0 ? "+" : ""}{formatQuantity(currentDifference, item.unitOfMeasure)}
                          </p>
                        </div>
                      )}
                      {isDraft && (
                        <div className="ml-auto">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => {
                              if (confirm("Czy na pewno chcesz usunąć tę pozycję ze spisu?")) {
                                deleteItemMutation.mutate(item.id);
                              }
                            }}
                            disabled={deleteItemMutation.isPending}
                            data-testid={`button-delete-${item.id}`}
                            className="h-8 w-8 text-destructive hover:text-destructive"
                          >
                            <X className="h-4 w-4" />
                          </Button>
                        </div>
                      )}
                    </div>
                    <div className="flex items-center gap-2 text-sm text-muted-foreground flex-wrap mt-2 ml-16">
                      {item.itemCode && <span>Kod: {item.itemCode}</span>}
                      {item.groupName && (
                        <>
                          {item.itemCode && <span>•</span>}
                          <span>Grupa: {item.groupName}</span>
                        </>
                      )}
                      {item.dimensions && (
                        <>
                          <span>•</span>
                          <span>{item.dimensions}</span>
                        </>
                      )}
                      {item.colorCode && (
                        <>
                          <span>•</span>
                          <span>Kolor: {item.colorCode}</span>
                        </>
                      )}
                      <span>•</span>
                      <span className="text-xs">
                        Aktualizowano: {new Date(item.updatedAt).toLocaleString("pl-PL")}
                      </span>
                    </div>
                  </div>
                );
              })}
            </div>
          </CardContent>
        </Card>

        {isDraft && items.length > 0 && (
          <div className="flex justify-end">
            <Button
              onClick={() => finalizeMutation.mutate()}
              disabled={!canFinalize || finalizeMutation.isPending || deleteItemMutation.isPending || updateItemMutation.isPending}
              data-testid="button-finalize-bottom"
              size="lg"
              title={
                hasUnsavedEdits ? "Zapisz wszystkie zmiany przed przeliczeniem" :
                uncountedItems > 0 ? `Pozostało ${uncountedItems} pozycji do spisania` :
                "Przelicz magazyn"
              }
            >
              <CheckCircle className="h-5 w-5 mr-2" />
              {finalizeMutation.isPending ? "Zamykanie..." : "Przelicz magazyn"}
            </Button>
          </div>
        )}
      </div>
    </WarehouseLayout>
  );
}
