import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { AlertCircle, CheckCircle, AlertTriangle, RefreshCw, Activity } from "lucide-react";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

type SyncStatus = 'healthy' | 'warning' | 'critical';

interface PlatformHealth {
  platform: string;
  status: SyncStatus;
  last_sync_at: string | null;
  total_orders: number;
  failed_orders: number;
  last_error: string | null;
  updated_at: string;
}

interface SyncHealthResponse {
  overallStatus: SyncStatus;
  platforms: PlatformHealth[];
  lastUpdate: string | null;
}

export default function SyncHealthPage() {
  const { toast } = useToast();

  const { data: health, isLoading, refetch } = useQuery<SyncHealthResponse>({
    queryKey: ['/api/sync/health'],
  });

  const getStatusIcon = (status: SyncStatus) => {
    switch (status) {
      case 'healthy':
        return <CheckCircle className="h-5 w-5 text-green-500" data-testid="icon-healthy" />;
      case 'warning':
        return <AlertTriangle className="h-5 w-5 text-yellow-500" data-testid="icon-warning" />;
      case 'critical':
        return <AlertCircle className="h-5 w-5 text-red-500" data-testid="icon-critical" />;
    }
  };

  const getStatusBadge = (status: SyncStatus) => {
    switch (status) {
      case 'healthy':
        return <Badge variant="default" className="bg-green-500" data-testid="badge-healthy">Zdrowy</Badge>;
      case 'warning':
        return <Badge variant="default" className="bg-yellow-500" data-testid="badge-warning">Ostrzeżenie</Badge>;
      case 'critical':
        return <Badge variant="destructive" data-testid="badge-critical">Krytyczny</Badge>;
    }
  };

  const getPlatformName = (platform: string) => {
    switch (platform) {
      case 'allegro':
        return 'Allegro';
      case 'shoper':
        return 'Shoper';
      case 'odoo':
        return 'Odoo';
      default:
        return platform;
    }
  };

  const handleRunAutoHealing = async () => {
    try {
      toast({
        title: "Uruchamianie Auto-Healing...",
        description: "System naprawia automatycznie błędy synchronizacji",
      });

      const response = await fetch('/api/sync/auto-heal', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error('Failed to run auto-healing');
      }

      toast({
        title: "✅ Auto-Healing ukończony",
        description: "System automatycznie naprawił problemy synchronizacji",
      });

      // Refresh health data
      refetch();
    } catch (error: any) {
      toast({
        title: "❌ Błąd Auto-Healing",
        description: error.message || "Nie udało się uruchomić Auto-Healing",
        variant: "destructive",
      });
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <RefreshCw className="h-8 w-8 animate-spin mx-auto mb-4" />
          <p className="text-muted-foreground">Ładowanie stanu synchronizacji...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold flex items-center gap-2" data-testid="heading-sync-health">
            <Activity className="h-8 w-8" />
            Monitor Zdrowia Synchronizacji
          </h1>
          <p className="text-muted-foreground mt-2">
            Przegląd stanu synchronizacji ze wszystkich platform
          </p>
        </div>
        <Button onClick={handleRunAutoHealing} data-testid="button-run-auto-healing">
          <RefreshCw className="h-4 w-4 mr-2" />
          Uruchom Auto-Healing
        </Button>
      </div>

      {/* Overall Status Card */}
      <Card data-testid="card-overall-status">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            {health?.overallStatus && getStatusIcon(health.overallStatus)}
            Status Ogólny
          </CardTitle>
          <CardDescription>
            Ostatnia aktualizacja:{" "}
            {health?.lastUpdate
              ? format(new Date(health.lastUpdate), "d MMMM yyyy, HH:mm", { locale: pl })
              : "Brak danych"}
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex items-center gap-2">
            {health?.overallStatus && getStatusBadge(health.overallStatus)}
            {health?.overallStatus === 'healthy' && (
              <span className="text-sm text-muted-foreground">
                Wszystkie platformy działają poprawnie
              </span>
            )}
            {health?.overallStatus === 'warning' && (
              <span className="text-sm text-muted-foreground">
                Wykryto drobne problemy synchronizacji
              </span>
            )}
            {health?.overallStatus === 'critical' && (
              <span className="text-sm text-muted-foreground">
                Wykryto poważne problemy wymagające uwagi
              </span>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Platform Health Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        {health?.platforms.map((platform) => (
          <Card key={platform.platform} data-testid={`card-platform-${platform.platform}`}>
            <CardHeader>
              <CardTitle className="flex items-center justify-between">
                <span>{getPlatformName(platform.platform)}</span>
                {getStatusIcon(platform.status)}
              </CardTitle>
              <CardDescription>
                {platform.last_sync_at
                  ? `Ostatnia sync: ${format(new Date(platform.last_sync_at), "HH:mm, d MMM", { locale: pl })}`
                  : "Brak synchronizacji"}
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <span className="text-sm text-muted-foreground">Status:</span>
                {getStatusBadge(platform.status)}
              </div>

              <div className="space-y-2">
                <div className="flex items-center justify-between">
                  <span className="text-sm text-muted-foreground">Łącznie zamówień:</span>
                  <span className="font-medium" data-testid={`text-total-orders-${platform.platform}`}>
                    {platform.total_orders}
                  </span>
                </div>

                <div className="flex items-center justify-between">
                  <span className="text-sm text-muted-foreground">Nieudane:</span>
                  <span
                    className={`font-medium ${
                      platform.failed_orders > 0 ? "text-red-500" : "text-green-500"
                    }`}
                    data-testid={`text-failed-orders-${platform.platform}`}
                  >
                    {platform.failed_orders}
                  </span>
                </div>
              </div>

              {platform.last_error && (
                <div className="mt-4 p-3 bg-red-50 dark:bg-red-950 rounded-md">
                  <p className="text-xs font-medium text-red-600 dark:text-red-400 mb-1">
                    Ostatni błąd:
                  </p>
                  <p className="text-xs text-red-700 dark:text-red-300" data-testid={`text-last-error-${platform.platform}`}>
                    {platform.last_error}
                  </p>
                </div>
              )}
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Auto-Healing Info */}
      <Card data-testid="card-auto-healing-info">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <RefreshCw className="h-5 w-5" />
            Auto-Healing System
          </CardTitle>
          <CardDescription>
            System automatycznie naprawia problemy synchronizacji
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-3">
          <div className="flex items-start gap-2">
            <CheckCircle className="h-5 w-5 text-green-500 mt-0.5" />
            <div>
              <p className="font-medium">Automatyczne pobieranie brakujących zamówień</p>
              <p className="text-sm text-muted-foreground">
                System skanuje Allegro i Shoper w poszukiwaniu zamówień których nie ma w OMS i automatycznie je pobiera
              </p>
            </div>
          </div>

          <div className="flex items-start gap-2">
            <CheckCircle className="h-5 w-5 text-green-500 mt-0.5" />
            <div>
              <p className="font-medium">Wykrywanie brakujących zamówień w Odoo</p>
              <p className="text-sm text-muted-foreground">
                System automatycznie wykrywa zamówienia z OMS które nie są w Odoo i dodaje je do kolejki
              </p>
            </div>
          </div>

          <div className="flex items-start gap-2">
            <CheckCircle className="h-5 w-5 text-green-500 mt-0.5" />
            <div>
              <p className="font-medium">Automatyczne resetowanie nieudanych zamówień</p>
              <p className="text-sm text-muted-foreground">
                System resetuje nieudane synchronizacje z exponential backoff (1, 5, 15, 30, 60 min)
              </p>
            </div>
          </div>

          <div className="flex items-start gap-2">
            <CheckCircle className="h-5 w-5 text-green-500 mt-0.5" />
            <div>
              <p className="font-medium">Cykliczne uruchamianie</p>
              <p className="text-sm text-muted-foreground">
                Auto-Healing uruchamia się automatycznie co 15 minut
              </p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
