import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { ArrowLeft, Plus, Trash2, Loader2, ArrowUp, ArrowDown, Save } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import type { ProductionRouting, ProductionRoutingOperation, ProductionWorkCenter, ProductCreatorDictionary, ProductionLocation } from "@shared/schema";

export default function ProductionRoutingDetail() {
  const { id } = useParams();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [newOp, setNewOp] = useState<{ code: string; name: string; estimatedTime: string; workCenterId: string; createsBuffer: boolean; bufferLocationId: string } | null>(null);
  const [editingId, setEditingId] = useState<number | null>(null);
  const [editForm, setEditForm] = useState<{ estimatedTime: string; workCenterId: string; createsBuffer: boolean; bufferLocationId: string }>({ estimatedTime: "", workCenterId: "", createsBuffer: false, bufferLocationId: "" });

  const { data: routing, isLoading: isLoadingRouting } = useQuery<ProductionRouting>({
    queryKey: ["routing-detail", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/routings/${id}`);
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
  });

  const { data: operations = [], refetch: refetchOps } = useQuery<ProductionRoutingOperation[]>({
    queryKey: ["routing-ops", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/routings/${id}/operations`);
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
    enabled: !!id,
  });

  const { data: workCenters = [] } = useQuery<ProductionWorkCenter[]>({
    queryKey: ["wcs"],
    queryFn: async () => {
      const res = await fetch("/api/production/work-centers");
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
  });

  const { data: dictOps = [] } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["dict-ops"],
    queryFn: async () => {
      const res = await fetch("/api/dictionaries?type=production_operation");
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
  });

  const { data: locations = [] } = useQuery<ProductionLocation[]>({
    queryKey: ["/api/production/locations"],
    queryFn: async () => {
      const res = await fetch("/api/production/locations");
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => apiRequest("POST", `/api/production/routings/${id}/operations`, data),
    onSuccess: () => {
      refetchOps();
      setNewOp(null);
      toast({ title: "Dodano operację" });
    },
    onError: () => toast({ title: "Błąd", variant: "destructive" }),
  });

  const deleteMutation = useMutation({
    mutationFn: (opId: number) => apiRequest("DELETE", `/api/production/routings/${id}/operations/${opId}`),
    onSuccess: () => {
      refetchOps();
      toast({ title: "Usunięto operację" });
    },
    onError: () => toast({ title: "Błąd usuwania", variant: "destructive" }),
  });

  const updateMutation = useMutation({
    mutationFn: ({ opId, data }: { opId: number; data: any }) => 
      apiRequest("PATCH", `/api/production/routings/${id}/operations/${opId}`, data),
    onSuccess: () => {
      refetchOps();
      setEditingId(null);
      toast({ title: "Zaktualizowano operację" });
    },
    onError: () => toast({ title: "Błąd aktualizacji", variant: "destructive" }),
  });

  const reseqMutation = useMutation({
    mutationFn: (orderedIds: number[]) => apiRequest("POST", `/api/production/routings/${id}/operations/resequence`, { orderedIds }),
    onSuccess: () => {
      refetchOps();
      toast({ title: "Zmieniono kolejność" });
    },
    onError: () => toast({ title: "Błąd kolejności", variant: "destructive" }),
  });

  const handleMove = (opId: number, direction: 'up' | 'down') => {
    const idx = operations.findIndex(op => op.id === opId);
    if (idx === -1) return;
    if (direction === 'up' && idx === 0) return;
    if (direction === 'down' && idx === operations.length - 1) return;

    const newArr = [...operations];
    const targetIdx = direction === 'up' ? idx - 1 : idx + 1;
    [newArr[idx], newArr[targetIdx]] = [newArr[targetIdx], newArr[idx]];
    
    reseqMutation.mutate(newArr.map(op => op.id));
  };

  const handleSaveNew = () => {
    if (!newOp?.code || !newOp?.name) {
      toast({ title: "Wybierz operację ze słownika", variant: "destructive" });
      return;
    }

    createMutation.mutate({
      routingId: Number(id),
      code: newOp.code,
      name: newOp.name,
      estimatedTime: newOp.estimatedTime || "0",
      setupTime: "0",
      workCenterId: newOp.workCenterId && newOp.workCenterId !== "" ? Number(newOp.workCenterId) : null,
      createsBuffer: newOp.createsBuffer,
      bufferLocationId: newOp.bufferLocationId && newOp.bufferLocationId !== "" ? Number(newOp.bufferLocationId) : null,
      isActive: true,
    });
  };

  const handleSelectDict = (dictId: string) => {
    const dict = dictOps.find(d => d.id === Number(dictId));
    if (dict) {
      setNewOp(prev => ({
        ...prev!,
        code: dict.code,
        name: dict.name,
      }));
    }
  };

  if (isLoadingRouting) {
    return <div className="flex items-center justify-center h-96"><Loader2 className="h-8 w-8 animate-spin" /></div>;
  }

  if (!routing) {
    return <div className="container mx-auto p-6"><div className="text-center py-8">Nie znaleziono</div></div>;
  }

  return (
    <div className="container mx-auto p-6 space-y-4">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <Button variant="ghost" size="icon" onClick={() => setLocation("/production/routings")}>
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div>
            <h1 className="text-2xl font-bold">{routing.name}</h1>
            <p className="text-sm text-muted-foreground font-mono">{routing.code}</p>
          </div>
        </div>
        <Badge variant={routing.isActive ? "default" : "secondary"}>
          {routing.isActive ? "Aktywna" : "Nieaktywna"}
        </Badge>
      </div>

      <Card>
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between">
            <CardTitle className="text-lg">Operacje</CardTitle>
            {!newOp && (
              <Button size="sm" onClick={() => setNewOp({ code: "", name: "", estimatedTime: "0", workCenterId: "", createsBuffer: false, bufferLocationId: "" })}>
                <Plus className="h-4 w-4 mr-1" />
                Dodaj
              </Button>
            )}
          </div>
        </CardHeader>
        <CardContent className="p-0">
          <div className="border-t">
            <table className="w-full">
              <thead>
                <tr className="border-b bg-muted/50 text-xs">
                  <th className="text-left py-2 px-3 font-medium w-14">Kol.</th>
                  <th className="text-left py-2 px-3 font-medium">Operacja</th>
                  <th className="text-left py-2 px-3 font-medium w-32">Kod</th>
                  <th className="text-left py-2 px-3 font-medium">Gniazdo</th>
                  <th className="text-left py-2 px-3 font-medium w-24">Czas</th>
                  <th className="text-center py-2 px-3 font-medium w-20">Bufor</th>
                  <th className="text-left py-2 px-3 font-medium">Lokalizacja</th>
                  <th className="text-right py-2 px-3 font-medium w-32">Akcje</th>
                </tr>
              </thead>
              <tbody>
                {newOp && (
                  <tr className="border-b bg-accent/20 h-[30px]">
                    <td className="py-1 px-3 text-sm text-muted-foreground">Nowa</td>
                    <td className="py-1 px-3">
                      <Select onValueChange={handleSelectDict}>
                        <SelectTrigger className="h-7 text-xs">
                          <SelectValue placeholder="Wybierz operację..." />
                        </SelectTrigger>
                        <SelectContent>
                          {dictOps.map(d => (
                            <SelectItem key={d.id} value={String(d.id)} className="text-xs">
                              {d.name}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </td>
                    <td className="py-1 px-3">
                      <Input
                        value={newOp.code || ""}
                        placeholder="Automatycznie"
                        className="h-7 text-xs bg-muted/50 cursor-not-allowed font-mono"
                        disabled
                      />
                    </td>
                    <td className="py-1 px-3">
                      <Select value={newOp.workCenterId || undefined} onValueChange={(v) => setNewOp({ ...newOp, workCenterId: v === "none" ? "" : v })}>
                        <SelectTrigger className="h-7 text-xs">
                          <SelectValue placeholder="Brak" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="none" className="text-xs">Brak</SelectItem>
                          {workCenters.map(wc => (
                            <SelectItem key={wc.id} value={String(wc.id)} className="text-xs">
                              {wc.name}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </td>
                    <td className="py-1 px-3">
                      <Input
                        type="number"
                        value={newOp.estimatedTime || ""}
                        onChange={(e) => setNewOp({ ...newOp, estimatedTime: e.target.value })}
                        placeholder="min"
                        className="h-7 text-xs w-20"
                      />
                    </td>
                    <td className="py-1 px-3 text-center">
                      <Checkbox
                        checked={newOp.createsBuffer}
                        onCheckedChange={(checked) => setNewOp({ ...newOp, createsBuffer: checked === true, bufferLocationId: checked ? newOp.bufferLocationId : "" })}
                        data-testid="checkbox-creates-buffer"
                      />
                    </td>
                    <td className="py-1 px-3">
                      {newOp.createsBuffer ? (
                        <Select value={newOp.bufferLocationId || undefined} onValueChange={(v) => setNewOp({ ...newOp, bufferLocationId: v === "none" ? "" : v })}>
                          <SelectTrigger className="h-7 text-xs">
                            <SelectValue placeholder="Wybierz..." />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="none" className="text-xs">Brak</SelectItem>
                            {locations.map(loc => (
                              <SelectItem key={loc.id} value={String(loc.id)} className="text-xs">
                                {loc.name}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      ) : (
                        <span className="text-xs text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-1 px-3 text-right">
                      <div className="flex gap-1 justify-end">
                        <Button size="sm" className="h-7 text-xs" onClick={handleSaveNew} disabled={createMutation.isPending || !newOp.code}>
                          <Save className="h-3 w-3 mr-1" />
                          Zapisz
                        </Button>
                        <Button size="sm" variant="ghost" className="h-7 text-xs" onClick={() => setNewOp(null)}>
                          Anuluj
                        </Button>
                      </div>
                    </td>
                  </tr>
                )}

                {operations.length === 0 && !newOp ? (
                  <tr>
                    <td colSpan={8} className="text-center py-8 text-sm text-muted-foreground">
                      Brak operacji. Kliknij "Dodaj" aby rozpocząć.
                    </td>
                  </tr>
                ) : (
                  operations.map((op, idx) => {
                    const isEditing = editingId === op.id;
                    
                    if (isEditing) {
                      return (
                        <tr key={op.id} className="border-b bg-accent/10 h-[30px]">
                          <td className="py-1 px-3">
                            <Badge variant="outline" className="h-5 text-xs">{op.sequence}</Badge>
                          </td>
                          <td className="py-1 px-3 text-sm">{op.name}</td>
                          <td className="py-1 px-3 text-sm font-mono text-muted-foreground">{op.code}</td>
                          <td className="py-1 px-3">
                            <Select 
                              value={editForm.workCenterId || undefined} 
                              onValueChange={(v) => setEditForm({ ...editForm, workCenterId: v === "none" ? "" : v })}
                            >
                              <SelectTrigger className="h-7 text-xs">
                                <SelectValue placeholder="Brak" />
                              </SelectTrigger>
                              <SelectContent>
                                <SelectItem value="none" className="text-xs">Brak</SelectItem>
                                {workCenters.map(wc => (
                                  <SelectItem key={wc.id} value={String(wc.id)} className="text-xs">
                                    {wc.name}
                                  </SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          </td>
                          <td className="py-1 px-3">
                            <Input
                              type="number"
                              value={editForm.estimatedTime}
                              onChange={(e) => setEditForm({ ...editForm, estimatedTime: e.target.value })}
                              className="h-7 text-xs w-20"
                            />
                          </td>
                          <td className="py-1 px-3 text-center">
                            <Checkbox
                              checked={editForm.createsBuffer}
                              onCheckedChange={(checked) => setEditForm({ ...editForm, createsBuffer: checked === true, bufferLocationId: checked ? editForm.bufferLocationId : "" })}
                            />
                          </td>
                          <td className="py-1 px-3">
                            {editForm.createsBuffer ? (
                              <Select value={editForm.bufferLocationId || undefined} onValueChange={(v) => setEditForm({ ...editForm, bufferLocationId: v === "none" ? "" : v })}>
                                <SelectTrigger className="h-7 text-xs">
                                  <SelectValue placeholder="Wybierz..." />
                                </SelectTrigger>
                                <SelectContent>
                                  <SelectItem value="none" className="text-xs">Brak</SelectItem>
                                  {locations.map(loc => (
                                    <SelectItem key={loc.id} value={String(loc.id)} className="text-xs">
                                      {loc.name}
                                    </SelectItem>
                                  ))}
                                </SelectContent>
                              </Select>
                            ) : (
                              <span className="text-xs text-muted-foreground">-</span>
                            )}
                          </td>
                          <td className="py-1 px-3 text-right">
                            <div className="flex gap-1 justify-end">
                              <Button
                                size="sm"
                                className="h-7 text-xs"
                                onClick={() => {
                                  updateMutation.mutate({
                                    opId: op.id,
                                    data: {
                                      estimatedTime: editForm.estimatedTime || "0",
                                      workCenterId: editForm.workCenterId && editForm.workCenterId !== "" ? Number(editForm.workCenterId) : null,
                                      createsBuffer: editForm.createsBuffer,
                                      bufferLocationId: editForm.bufferLocationId && editForm.bufferLocationId !== "" ? Number(editForm.bufferLocationId) : null,
                                    }
                                  });
                                }}
                                disabled={updateMutation.isPending}
                              >
                                <Save className="h-3 w-3 mr-1" />
                                Zapisz
                              </Button>
                              <Button
                                size="sm"
                                variant="ghost"
                                className="h-7 text-xs"
                                onClick={() => setEditingId(null)}
                              >
                                Anuluj
                              </Button>
                            </div>
                          </td>
                        </tr>
                      );
                    }
                    
                    return (
                      <tr 
                        key={op.id} 
                        className="border-b hover:bg-muted/30 h-[30px] cursor-pointer"
                        onClick={() => {
                          setEditingId(op.id);
                          setEditForm({
                            estimatedTime: op.estimatedTime || "0",
                            workCenterId: op.workCenterId ? String(op.workCenterId) : "",
                            createsBuffer: op.createsBuffer || false,
                            bufferLocationId: op.bufferLocationId ? String(op.bufferLocationId) : "",
                          });
                        }}
                      >
                        <td className="py-1 px-3">
                          <Badge variant="outline" className="h-5 text-xs">{op.sequence}</Badge>
                        </td>
                        <td className="py-1 px-3 text-sm">{op.name}</td>
                        <td className="py-1 px-3 text-sm font-mono text-muted-foreground">{op.code}</td>
                        <td className="py-1 px-3 text-xs text-muted-foreground">
                          {op.workCenterId ? workCenters.find(wc => wc.id === op.workCenterId)?.name || "-" : "-"}
                        </td>
                        <td className="py-1 px-3 text-sm text-muted-foreground">
                          {op.estimatedTime ? `${op.estimatedTime} min` : "-"}
                        </td>
                        <td className="py-1 px-3 text-center">
                          {op.createsBuffer ? <Badge variant="secondary" className="h-5 text-xs">Tak</Badge> : <span className="text-xs text-muted-foreground">-</span>}
                        </td>
                        <td className="py-1 px-3 text-xs text-muted-foreground">
                          {op.bufferLocationId ? locations.find(loc => loc.id === op.bufferLocationId)?.name || "-" : "-"}
                        </td>
                        <td className="py-1 px-3 text-right" onClick={(e) => e.stopPropagation()}>
                          <div className="flex gap-1 justify-end">
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => handleMove(op.id, 'up')}
                              disabled={idx === 0 || reseqMutation.isPending}
                            >
                              <ArrowUp className="h-3.5 w-3.5" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => handleMove(op.id, 'down')}
                              disabled={idx === operations.length - 1 || reseqMutation.isPending}
                            >
                              <ArrowDown className="h-3.5 w-3.5" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => {
                                if (confirm("Czy na pewno chcesz usunąć?")) {
                                  deleteMutation.mutate(op.id);
                                }
                              }}
                            >
                              <Trash2 className="h-3.5 w-3.5" />
                            </Button>
                          </div>
                        </td>
                      </tr>
                    );
                  })
                )}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
