import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Plus, Edit, Trash2, Loader2, Wrench } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { z } from "zod";
import type { ProductCreatorDictionary } from "@shared/schema";

const operationFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().optional(),
  sortOrder: z.coerce.number().optional(),
});

type OperationFormData = z.infer<typeof operationFormSchema>;

// Auto-generate code from name
function generateOperationCode(name: string): string {
  if (!name) return "";
  
  return name
    .toLowerCase()
    .normalize("NFD")
    .replace(/[\u0300-\u036f]/g, "") // Remove diacritics
    .replace(/ą/g, "a")
    .replace(/ć/g, "c")
    .replace(/ę/g, "e")
    .replace(/ł/g, "l")
    .replace(/ń/g, "n")
    .replace(/ó/g, "o")
    .replace(/ś/g, "s")
    .replace(/ź/g, "z")
    .replace(/ż/g, "z")
    .replace(/[^a-z0-9]+/g, "_") // Replace non-alphanumeric with underscore
    .replace(/^_+|_+$/g, "")     // Remove leading/trailing underscores
    .replace(/_+/g, "_");         // Replace multiple underscores with single
}

export default function ProductionOperations() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingOperation, setEditingOperation] = useState<ProductCreatorDictionary | null>(null);
  const { toast } = useToast();

  const form = useForm<OperationFormData>({
    resolver: zodResolver(operationFormSchema),
    defaultValues: {
      name: "",
      description: "",
      sortOrder: 0,
    },
  });

  const { data: operations = [], isLoading } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries", { type: "production_operation" }],
    queryFn: async () => {
      const response = await fetch("/api/dictionaries?type=production_operation");
      if (!response.ok) throw new Error("Failed to fetch operations");
      return response.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: (data: any) =>
      apiRequest("POST", "/api/dictionaries", {
        ...data,
        dictionaryType: "production_operation",
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      setIsDialogOpen(false);
      form.reset();
      toast({ title: "Operacja została utworzona" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia operacji",
        description: error.message || "Nieznany błąd",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) =>
      apiRequest("PATCH", `/api/dictionaries/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      setIsDialogOpen(false);
      setEditingOperation(null);
      form.reset();
      toast({ title: "Operacja została zaktualizowana" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/dictionaries/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      toast({ title: "Operacja została usunięta" });
    },
  });

  useEffect(() => {
    if (editingOperation) {
      form.reset({
        name: editingOperation.name,
        description: editingOperation.description ?? "",
        sortOrder: editingOperation.sortOrder ?? 0,
      });
    } else {
      form.reset({
        name: "",
        description: "",
        sortOrder: 0,
      });
    }
  }, [editingOperation, form]);

  const onSubmit = (data: OperationFormData) => {
    const generatedCode = editingOperation ? editingOperation.code : generateOperationCode(data.name);
    
    const normalizedData = {
      code: generatedCode,
      ...data,
      description: data.description || undefined,
      sortOrder: data.sortOrder || 0,
    };

    if (editingOperation) {
      updateMutation.mutate({ id: editingOperation.id, data: normalizedData });
    } else {
      createMutation.mutate(normalizedData);
    }
  };

  const handleEdit = (operation: ProductCreatorDictionary) => {
    setEditingOperation(operation);
    setIsDialogOpen(true);
  };

  const handleDelete = (id: number) => {
    if (confirm("Czy na pewno chcesz usunąć tę operację?")) {
      deleteMutation.mutate(id);
    }
  };

  const handleAddNew = () => {
    setEditingOperation(null);
    setIsDialogOpen(true);
  };

  const sortedOperations = [...operations].sort((a, b) => {
    const orderA = a.sortOrder ?? 0;
    const orderB = b.sortOrder ?? 0;
    return orderA - orderB;
  });

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Operacje Produkcyjne</h1>
          <p className="text-muted-foreground">Zarządzaj słownikiem operacji produkcyjnych</p>
        </div>
        <Button onClick={handleAddNew} data-testid="button-add-operation">
          <Plus className="w-4 h-4 mr-2" />
          Dodaj Operację
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Wrench className="w-5 h-5" />
            Lista Operacji ({operations.length})
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex items-center justify-center py-8">
              <Loader2 className="w-6 h-6 animate-spin" />
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-[100px]">Kolejność</TableHead>
                  <TableHead className="w-[150px]">Kod</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Opis</TableHead>
                  <TableHead className="w-[120px]">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {sortedOperations.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={5} className="text-center text-muted-foreground py-8">
                      Brak operacji. Dodaj pierwszą operację.
                    </TableCell>
                  </TableRow>
                ) : (
                  sortedOperations.map((operation) => (
                    <TableRow key={operation.id} data-testid={`row-operation-${operation.id}`}>
                      <TableCell>
                        <Badge variant="outline">{operation.sortOrder ?? 0}</Badge>
                      </TableCell>
                      <TableCell>
                        <code className="text-sm font-mono bg-muted px-2 py-1 rounded">
                          {operation.code}
                        </code>
                      </TableCell>
                      <TableCell className="font-medium">{operation.name}</TableCell>
                      <TableCell className="text-muted-foreground">
                        {operation.description || "-"}
                      </TableCell>
                      <TableCell>
                        <div className="flex gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleEdit(operation)}
                            data-testid={`button-edit-operation-${operation.id}`}
                          >
                            <Edit className="w-4 h-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleDelete(operation.id)}
                            data-testid={`button-delete-operation-${operation.id}`}
                          >
                            <Trash2 className="w-4 h-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>
              {editingOperation ? "Edytuj Operację" : "Dodaj Operację"}
            </DialogTitle>
            <DialogDescription>
              {editingOperation
                ? "Zaktualizuj informacje o operacji produkcyjnej"
                : "Dodaj nową operację do słownika"}
            </DialogDescription>
          </DialogHeader>

          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa *</FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        placeholder="np. Cięcie płyt"
                        data-testid="input-operation-name"
                      />
                    </FormControl>
                    {!editingOperation && field.value && (
                      <p className="text-xs text-muted-foreground">
                        Kod: <span className="font-mono text-primary">{generateOperationCode(field.value)}</span>
                      </p>
                    )}
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="sortOrder"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kolejność</FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        type="number"
                        placeholder="0"
                        data-testid="input-operation-sort"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea
                        {...field}
                        placeholder="Szczegółowy opis operacji..."
                        rows={3}
                        data-testid="input-operation-description"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <DialogFooter>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setIsDialogOpen(false)}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={createMutation.isPending || updateMutation.isPending}
                  data-testid="button-submit-operation"
                >
                  {(createMutation.isPending || updateMutation.isPending) && (
                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  )}
                  {editingOperation ? "Zaktualizuj" : "Utwórz"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
