import { useState, useEffect, useRef, RefObject } from 'react';

interface UseScrollCollapseOptions {
  threshold?: number; // Pixels scrolled before collapsing
  scrollRef?: RefObject<HTMLElement>;
  onChange?: (isCollapsed: boolean) => void; // Callback when collapse state changes
  enabled?: boolean; // Whether this panel is currently active
}

export function useScrollCollapse({ threshold = 64, scrollRef, onChange, enabled = true }: UseScrollCollapseOptions = {}) {
  const [isCollapsed, setIsCollapsed] = useState(false);
  const lastScrollY = useRef(0);
  const ticking = useRef(false);

  // Reset state when panel becomes inactive, check scroll position when becoming active
  useEffect(() => {
    if (!enabled) {
      setIsCollapsed(false);
      lastScrollY.current = 0;
      onChange?.(false); // Notify parent of reset
    } else {
      // Panel just became active - check if it's already scrolled
      const element = scrollRef?.current;
      if (element && element.scrollTop > threshold) {
        setIsCollapsed(true);
        onChange?.(true);
      } else if (element && element.scrollTop <= threshold) {
        // Panel is at top, ensure parent knows it's expanded
        setIsCollapsed(false);
        onChange?.(false);
      }
    }
  }, [enabled, scrollRef, threshold, onChange]);

  useEffect(() => {
    const element = scrollRef?.current;
    if (!element || !enabled) return;

    const handleScroll = () => {
      if (!ticking.current) {
        window.requestAnimationFrame(() => {
          const currentScrollY = element.scrollTop;
          const scrollHeight = element.scrollHeight;
          const clientHeight = element.clientHeight;
          const scrollBottom = scrollHeight - (currentScrollY + clientHeight);
          
          // Detect if we're at the bottom of the scroll container (with 50px buffer)
          const isAtBottom = scrollBottom < 50;
          
          // If at bottom, don't change collapse state to prevent flickering
          if (isAtBottom) {
            lastScrollY.current = currentScrollY;
            ticking.current = false;
            return;
          }
          
          const scrollingDown = currentScrollY > lastScrollY.current;
          
          // Collapse when scrolling down past threshold
          // Expand when scrolling up or near top
          const newCollapsedState = scrollingDown && currentScrollY > threshold;
          const newExpandedState = !scrollingDown || currentScrollY < threshold / 2;
          
          if (newCollapsedState && !isCollapsed) {
            setIsCollapsed(true);
            onChange?.(true);
          } else if (newExpandedState && isCollapsed) {
            setIsCollapsed(false);
            onChange?.(false);
          }
          
          lastScrollY.current = currentScrollY;
          ticking.current = false;
        });
        
        ticking.current = true;
      }
    };

    element.addEventListener('scroll', handleScroll, { passive: true });
    return () => element.removeEventListener('scroll', handleScroll);
  }, [scrollRef, threshold, onChange, isCollapsed, enabled]);

  return { isCollapsed };
}
