import { useState, useMemo, useEffect, useRef } from "react";
import { ShoppingCart, Search, X, Plus, LinkIcon, Loader2, Package, List, Layers, ClipboardList, Check, Database, Box, FileStack } from "lucide-react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { ToggleGroup, ToggleGroupItem } from "@/components/ui/toggle-group";
import { useToast } from "@/hooks/use-toast";
import { OrdersTabsBar } from "./OrdersTabsBar";
import { CompactOrdersList } from "./CompactOrdersList";
import { GroupedOrdersList } from "./GroupedOrdersList";
import { OrdersControlBar } from "./OrdersControlBar";
import { useScrollCollapse } from "./use-scroll-collapse";

interface SetComponent {
  component_id: number;
  component_sku: string;
  component_title: string;
  component_color: string | null;
  component_length: string | null;
  component_width: string | null;
  component_height: string | null;
  component_product_type: string | null;
  component_doors: string | null;
  component_legs: string | null;
  quantity: number;
  primary_image_url: string | null;
  parent_set_image_url: string | null;
  is_in_current_plan: boolean;
  is_in_any_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
}

interface MarketplaceOrderItem {
  item_id: number;
  offer_external_id: string;
  name: string;
  quantity: number;
  unit_price: string;
  price: string;
  image_url: string | null;
  product_length: string | null;
  product_width: string | null;
  product_height: string | null;
  product_color: string | null;
  product_color_options: string[] | null;
  product_sku: string | null;
  product_type: string | null;
  product_doors: string | null;
  product_legs: string | null;
  marketplace_product_id: number | null;
  link_type: string | null;
  catalog_product_id: number | null;
  catalog_product_sku: string | null;
  catalog_product_title: string | null;
  catalog_set_id: number | null;
  catalog_set_sku: string | null;
  catalog_set_title: string | null;
  platform_link_id: number | null;
  bom_component_count: number | null;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
  in_current_plan: boolean;
  set_components: SetComponent[] | null;
  warehouse_total_qty: number | null;
  warehouse_reserved_qty: number | null;
  line_reserved_qty: number | null;
}

interface MarketplaceOrder {
  order_id: number;
  order_number: string;
  marketplace: string;
  buyer_first_name: string;
  buyer_last_name: string;
  buyer_email: string;
  order_date: string;
  payment_status: string;
  total_to_pay_amount: string;
  currency: string;
  items: MarketplaceOrderItem[];
}

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

interface OrdersPanelProps {
  planId: string;
  orders: MarketplaceOrder[];
  isLoading: boolean;
  statistics: {
    ordersCount: number;
    productsCount: number;
  };
  filters: {
    search?: string;
    color?: string;
    sku?: string;
    limit?: number;
    offset?: number;
    sortBy?: 'order_date' | 'order_number' | 'buyer_name' | 'total_amount' | 'product_sku';
    sortOrder?: 'asc' | 'desc';
    [key: string]: any;
  };
  searchFilter: string;
  onSearchChange: (value: string) => void;
  onFilterChange: (key: string, value: string) => void;
  onClearFilter: (key: string) => void;
  onResetFilters: () => void;
  onAddItem: (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string, setId?: number, componentId?: number) => void;
  onLinkItem: (item: MarketplaceOrderItem, marketplace: string) => void;
  activeFiltersCount: number;
  isAddingItem?: boolean;
  addingItems?: Array<{
    id: string;
    productSku: string;
    productName: string;
    quantity: number;
    orderNumber?: string;
    status: 'pending' | 'processing' | 'completed';
    reservationInfo?: {
      packedProductReserved: boolean;
      packedProductSku?: string;
      packedQtyReserved?: number;
      formatkaSearched: boolean;
      formatkiFound: string[];
      formatkiReserved: string[];
    };
  }>;
  recentlyAddedItems?: Set<number>;
  colors?: DictionaryItem[];
  productTypes?: DictionaryItem[];
  onCollapseChange?: (isCollapsed: boolean) => void;
  enabled?: boolean;
  onHighlightInPlan?: (productId: number, orderNumber: string) => void;
}

export function OrdersPanel({
  planId,
  orders,
  isLoading,
  statistics,
  filters,
  searchFilter,
  onSearchChange,
  onFilterChange,
  onClearFilter,
  onResetFilters,
  onAddItem,
  onLinkItem,
  activeFiltersCount,
  isAddingItem = false,
  addingItems = [],
  recentlyAddedItems = new Set(),
  colors,
  productTypes = [],
  onCollapseChange,
  enabled = true,
  onHighlightInPlan,
}: OrdersPanelProps) {
  const { toast } = useToast();
  
  // View mode with localStorage persistence
  const [viewMode, setViewMode] = useState<"flat" | "grouped">(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-view-mode`);
      return (saved === "flat" || saved === "grouped") ? saved : "flat";
    } catch {
      return "flat";
    }
  });
  
  // Save view mode to localStorage
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-view-mode`, viewMode);
  }, [planId, viewMode]);
  
  const [expandedOrders, setExpandedOrders] = useState<Set<number>>(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-expanded-orders`);
      return saved ? new Set(JSON.parse(saved)) : new Set();
    } catch {
      return new Set();
    }
  });
  const [selectedItemIds, setSelectedItemIds] = useState<Set<string>>(new Set());
  
  // Save expanded orders to localStorage
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-expanded-orders`, JSON.stringify(Array.from(expandedOrders)));
  }, [planId, expandedOrders]);
  
  // Track if initial expansion has been done (to prevent re-seeding on data changes)
  const [hasInitializedExpansion, setHasInitializedExpansion] = useState(() => {
    // If localStorage has saved state, consider it already initialized
    try {
      const savedColors = localStorage.getItem(`plan-${planId}-expanded-colors`);
      const savedTypes = localStorage.getItem(`plan-${planId}-expanded-types`);
      return !!(savedColors || savedTypes);
    } catch {
      return false;
    }
  });
  
  // Grouped view expansion state - persist in localStorage
  const [expandedColorGroups, setExpandedColorGroups] = useState<Set<string>>(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-expanded-colors`);
      return saved ? new Set(JSON.parse(saved)) : new Set();
    } catch {
      return new Set();
    }
  });
  
  const [expandedTypeGroups, setExpandedTypeGroups] = useState<Set<string>>(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-expanded-types`);
      return saved ? new Set(JSON.parse(saved)) : new Set();
    } catch {
      return new Set();
    }
  });
  
  const [expandedSets, setExpandedSets] = useState<Set<number>>(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-expanded-sets`);
      if (saved) {
        return new Set(JSON.parse(saved));
      }
      // Initialize with all sets expanded by default
      const allSetItemIds = new Set<number>();
      orders.forEach(order => {
        order.items.forEach(item => {
          if (item.set_components && item.set_components.length > 0) {
            allSetItemIds.add(item.item_id);
          }
        });
      });
      return allSetItemIds;
    } catch {
      return new Set();
    }
  });
  
  // Save expansion state to localStorage
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-expanded-colors`, JSON.stringify(Array.from(expandedColorGroups)));
  }, [planId, expandedColorGroups]);
  
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-expanded-types`, JSON.stringify(Array.from(expandedTypeGroups)));
  }, [planId, expandedTypeGroups]);
  
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-expanded-sets`, JSON.stringify(Array.from(expandedSets)));
  }, [planId, expandedSets]);
  
  // Calculate all possible set item IDs from orders
  const allSetItemIds = useMemo(() => {
    const setIds = new Set<number>();
    orders.forEach(order => {
      order.items.forEach(item => {
        if (item.set_components && item.set_components.length > 0) {
          setIds.add(item.item_id);
        }
      });
    });
    return setIds;
  }, [orders]);

  // Calculate all color and type group keys for grouped view
  const { allColorKeys, allTypeKeys } = useMemo(() => {
    const colorKeys = new Set<string>();
    const typeKeys = new Set<string>();
    
    orders.forEach(order => {
      order.items.forEach(item => {
        const colorKey = item.product_color || 'Bez koloru';
        const typeKey = item.product_type || 'Bez typu';
        colorKeys.add(colorKey);
        typeKeys.add(`${colorKey}-${typeKey}`);
      });
    });
    
    return { allColorKeys: colorKeys, allTypeKeys: typeKeys };
  }, [orders]);

  // Initialize expansion state on first load (when localStorage is empty)
  useEffect(() => {
    if (!hasInitializedExpansion && orders.length > 0 && allColorKeys.size > 0) {
      // Seed initial state with all groups expanded
      setExpandedColorGroups(new Set(allColorKeys));
      setExpandedTypeGroups(new Set(allTypeKeys));
      setExpandedSets(new Set(allSetItemIds));
      setHasInitializedExpansion(true);
    }
  }, [hasInitializedExpansion, orders.length, allColorKeys, allTypeKeys, allSetItemIds]);

  const allExpanded = useMemo(() => {
    if (orders.length === 0) return false;
    
    if (viewMode === 'grouped') {
      // In grouped mode, check if all color groups and type groups are expanded
      if (allColorKeys.size === 0) return true;
      const allColorsExpanded = Array.from(allColorKeys).every(key => expandedColorGroups.has(key));
      const allTypesExpanded = Array.from(allTypeKeys).every(key => expandedTypeGroups.has(key));
      return allColorsExpanded && allTypesExpanded;
    } else {
      // In flat mode, check if all sets are expanded
      if (allSetItemIds.size === 0) return true; // No sets = consider all expanded
      return Array.from(allSetItemIds).every(id => expandedSets.has(id));
    }
  }, [viewMode, orders, expandedColorGroups, expandedTypeGroups, expandedSets, allSetItemIds, allColorKeys, allTypeKeys]);

  const toggleOrder = (orderId: number) => {
    setExpandedOrders(prev => {
      const newSet = new Set(prev);
      if (newSet.has(orderId)) {
        newSet.delete(orderId);
      } else {
        newSet.add(orderId);
      }
      return newSet;
    });
  };

  const handleExpandAll = () => {
    if (viewMode === 'grouped') {
      // In grouped mode, expand all color groups, type groups, and sets
      setExpandedColorGroups(new Set(allColorKeys));
      setExpandedTypeGroups(new Set(allTypeKeys));
      setExpandedSets(new Set(allSetItemIds));
    } else {
      // In flat mode, expand all sets
      setExpandedSets(new Set(allSetItemIds));
    }
  };

  const handleCollapseAll = () => {
    if (viewMode === 'grouped') {
      // In grouped mode, collapse all color groups, type groups, and sets
      setExpandedColorGroups(new Set());
      setExpandedTypeGroups(new Set());
      setExpandedSets(new Set());
    } else {
      // In flat mode, collapse all sets
      setExpandedSets(new Set());
    }
  };

  const handlePageSizeChange = (size: number) => {
    onFilterChange('limit', size.toString());
    onFilterChange('offset', '0');
  };

  const handleSortByChange = (sortBy: 'order_date' | 'order_number' | 'buyer_name' | 'total_amount' | 'product_sku') => {
    onFilterChange('sortBy', sortBy);
  };

  const handleSortOrderToggle = () => {
    const currentOrder = filters.sortOrder || 'desc';
    onFilterChange('sortOrder', currentOrder === 'asc' ? 'desc' : 'asc');
  };
  
  // Batch selection handlers
  const toggleItemSelection = (itemId: string) => {
    setSelectedItemIds(prev => {
      const newSet = new Set(prev);
      if (newSet.has(itemId)) {
        newSet.delete(itemId);
      } else {
        newSet.add(itemId);
      }
      return newSet;
    });
  };
  
  const clearSelection = () => {
    setSelectedItemIds(new Set());
  };
  
  const handleBulkAdd = () => {
    if (selectedItemIds.size === 0) return;
    
    let addedCount = 0;
    
    // Process each order and its items/components
    orders.forEach(order => {
      const orderNumber = order.order_number;
      const buyerName = `${order.buyer_first_name} ${order.buyer_last_name}`;
      const marketplace = order.marketplace;
      
      order.items.forEach(item => {
        const hasSetComponents = item.set_components && item.set_components.length > 0;
        
        // Check if main product/set is selected
        if (selectedItemIds.has(`item-${item.item_id}`)) {
          if (hasSetComponents) {
            // For sets: add all components that aren't already in any plan
            const componentsToAdd = item.set_components?.filter(c => !c.is_in_any_plan) || [];
            componentsToAdd.forEach(component => {
              const componentItem: MarketplaceOrderItem = {
                ...item,
                item_id: component.component_id,
                name: component.component_title,
                product_sku: component.component_sku,
                product_length: component.component_length,
                product_width: component.component_width,
                product_height: component.component_height,
                product_color: component.component_color,
                product_type: component.component_product_type,
                product_doors: component.component_doors,
                product_legs: component.component_legs,
                catalog_product_id: component.component_id,
                quantity: component.quantity,
                image_url: component.primary_image_url,
                is_in_plan: component.is_in_any_plan,
                in_plan_number: component.in_plan_number,
                in_plan_id: component.in_plan_id,
                in_current_plan: component.is_in_current_plan,
                set_components: null,
              };
              onAddItem(componentItem, orderNumber, buyerName, marketplace, item.catalog_set_id ?? undefined, component.component_id);
              addedCount++;
            });
          } else {
            // For single products: add directly
            onAddItem(item, orderNumber, buyerName, marketplace);
            addedCount++;
          }
        }
        
        // Check if any individual set components are selected (not the whole set)
        if (hasSetComponents) {
          item.set_components!.forEach(component => {
            const componentKey = `component-${orderNumber}-${component.component_id}`;
            // Only add if component is individually selected AND not already in a plan
            if (selectedItemIds.has(componentKey) && !component.is_in_any_plan) {
              // Skip if the whole set was already selected (components already added above)
              if (!selectedItemIds.has(`item-${item.item_id}`)) {
                const componentItem: MarketplaceOrderItem = {
                  ...item,
                  item_id: component.component_id,
                  name: component.component_title,
                  product_sku: component.component_sku,
                  product_length: component.component_length,
                  product_width: component.component_width,
                  product_height: component.component_height,
                  product_color: component.component_color,
                  product_type: component.component_product_type,
                  product_doors: component.component_doors,
                  product_legs: component.component_legs,
                  catalog_product_id: component.component_id,
                  quantity: component.quantity,
                  image_url: component.primary_image_url,
                  is_in_plan: component.is_in_any_plan,
                  in_plan_number: component.in_plan_number,
                  in_plan_id: component.in_plan_id,
                  in_current_plan: component.is_in_current_plan,
                  set_components: null,
                };
                onAddItem(componentItem, orderNumber, buyerName, marketplace, item.catalog_set_id ?? undefined, component.component_id);
                addedCount++;
              }
            }
          });
        }
      });
    });
    
    clearSelection();
    
    if (addedCount > 0) {
      toast({
        title: "Dodano do planu",
        description: `Dodano ${addedCount} ${addedCount === 1 ? 'produkt' : addedCount > 4 ? 'produktów' : 'produkty'} do planu produkcji`,
      });
    }
  };

  // Scroll collapse setup
  const scrollContainerRef = useRef<HTMLDivElement>(null);
  const { isCollapsed } = useScrollCollapse({ 
    scrollRef: scrollContainerRef, 
    threshold: 80,
    onChange: onCollapseChange,
    enabled
  });
  
  // Note: Filtering for Sets Only and Catalog Linked is done server-side
  // The filters.showSetsOnly and filters.showCatalogLinked are sent to the backend API
  
  // Save scroll position before data changes
  const savedScrollPositionRef = useRef<number>(0);
  const previousOrdersLengthRef = useRef<number>(orders.length);
  
  // Save scroll position when adding items starts
  useEffect(() => {
    if (isAddingItem && scrollContainerRef.current) {
      savedScrollPositionRef.current = scrollContainerRef.current.scrollTop;
    }
  }, [isAddingItem]);
  
  // Restore scroll position after data changes (refetch)
  useEffect(() => {
    // Only restore if orders count changed (indicating refetch happened)
    if (orders.length !== previousOrdersLengthRef.current && savedScrollPositionRef.current > 0) {
      // Use setTimeout to ensure DOM has updated
      setTimeout(() => {
        if (scrollContainerRef.current) {
          scrollContainerRef.current.scrollTop = savedScrollPositionRef.current;
          // Reset saved position after restoration
          savedScrollPositionRef.current = 0;
        }
      }, 0);
    }
    previousOrdersLengthRef.current = orders.length;
  }, [orders]);

  // Progress steps state for animated loader
  const [currentStep, setCurrentStep] = useState(0);
  const [stepStatuses, setStepStatuses] = useState<Record<number, 'pending' | 'active' | 'success' | 'warning' | 'info'>>({});
  const [stepMessages, setStepMessages] = useState<Record<number, string>>({});
  const [formatkiFoundList, setFormatkiFoundList] = useState<string[]>([]);
  const [formatkiReservedList, setFormatkiReservedList] = useState<string[]>([]);
  const [resultType, setResultType] = useState<'packed' | 'formatki' | 'none'>('none');
  const foundPackedRef = useRef(false);
  
  // Define processing steps with dynamic messages
  const processingSteps = [
    { id: 1, label: "Pobieranie danych produktu", icon: Database },
    { id: 2, label: "Szukanie produktów spakowanych", icon: Package },
    { id: 3, label: "Sprawdzanie dostępności magazynowej", icon: Box },
    { id: 4, label: "Szukanie formatek (BOM)", icon: FileStack },
    { id: 5, label: "Rezerwowanie komponentów", icon: ClipboardList },
    { id: 6, label: "Wynik rezerwacji", icon: Check },
  ];
  
  // No longer using step animation - simplified to show list of items with their status
  // Reset state when not adding
  useEffect(() => {
    if (!isAddingItem) {
      setCurrentStep(0);
      setStepStatuses({});
      setStepMessages({});
      setFormatkiFoundList([]);
      setFormatkiReservedList([]);
      setResultType('none');
      foundPackedRef.current = false;
    }
  }, [isAddingItem]);

  // Compute completed/processing counts for header display
  const completedCount = addingItems.filter(item => item.status === 'completed').length;
  const processingCount = addingItems.filter(item => item.status === 'processing').length;
  const totalCount = addingItems.length;

  return (
    <>
      {/* Bulk Progress Window - shows list of products being added */}
      {isAddingItem && addingItems.length > 0 && (
        <div className="fixed top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 z-50 pointer-events-auto">
          <div className="bg-background border-2 border-primary rounded-lg shadow-2xl p-4 w-[380px] max-h-[80vh] overflow-hidden flex flex-col">
            {/* Header with spinner and count */}
            <div className="flex items-center gap-3 pb-3 border-b flex-shrink-0">
              <Loader2 className="h-6 w-6 animate-spin text-primary flex-shrink-0" />
              <div className="min-w-0 flex-1">
                <p className="text-base font-semibold">Dodawanie do planu...</p>
                <p className="text-xs text-muted-foreground">
                  {completedCount} / {totalCount} produktów
                  {processingCount > 0 && ` (${processingCount} w trakcie)`}
                </p>
              </div>
            </div>
            
            {/* Products list - scrollable */}
            <div className="flex-1 overflow-y-auto py-2 space-y-1 min-h-0 max-h-[50vh]">
              {addingItems.map((item) => {
                const reservationInfo = item.reservationInfo;
                const isProcessing = item.status === 'processing';
                const isCompleted = item.status === 'completed';
                
                // Determine result type for completed items
                let resultType: 'packed' | 'formatki' | 'none' = 'none';
                if (reservationInfo) {
                  if (reservationInfo.packedProductReserved) {
                    resultType = 'packed';
                  } else if (reservationInfo.formatkiReserved.length > 0) {
                    resultType = 'formatki';
                  }
                }
                
                return (
                  <div
                    key={item.id}
                    className={`flex items-start gap-2 py-1.5 px-2 rounded-md text-xs transition-all duration-300 ${
                      isProcessing
                        ? "bg-primary/10 text-primary"
                        : isCompleted
                        ? "bg-muted/30 text-foreground"
                        : "text-muted-foreground"
                    }`}
                  >
                    {/* Status icon */}
                    <div className="w-4 h-4 flex items-center justify-center flex-shrink-0 mt-0.5">
                      {isProcessing ? (
                        <Loader2 className="h-3 w-3 animate-spin text-primary" />
                      ) : isCompleted ? (
                        resultType === 'packed' ? (
                          <Package className="h-3 w-3 text-green-500" />
                        ) : resultType === 'formatki' ? (
                          <FileStack className="h-3 w-3 text-amber-500" />
                        ) : (
                          <Check className="h-3 w-3 text-green-500" />
                        )
                      ) : (
                        <div className="h-2 w-2 rounded-full bg-muted-foreground/30" />
                      )}
                    </div>
                    
                    {/* Product info */}
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2">
                        <span className="font-medium truncate">{item.productSku}</span>
                        <span className="text-muted-foreground">×{item.quantity}</span>
                      </div>
                      
                      {/* Reservation details for completed items */}
                      {isCompleted && reservationInfo && (
                        <div className="mt-0.5 text-[10px]">
                          {resultType === 'packed' && (
                            <span className="text-green-600 dark:text-green-400">
                              Zarezerwowano produkt spakowany
                            </span>
                          )}
                          {resultType === 'formatki' && (
                            <div className="space-y-0.5">
                              <span className="text-amber-600 dark:text-amber-400">
                                Zarezerwowano {reservationInfo.formatkiReserved.length} formatek:
                              </span>
                              <div className="pl-2 space-y-0.5 max-h-12 overflow-y-auto">
                                {reservationInfo.formatkiReserved.map((name, i) => (
                                  <div key={i} className="text-[9px] text-muted-foreground truncate" title={name}>
                                    {name}
                                  </div>
                                ))}
                              </div>
                            </div>
                          )}
                          {resultType === 'none' && (
                            <span className="text-muted-foreground">
                              Dodano (brak rezerwacji)
                            </span>
                          )}
                        </div>
                      )}
                      
                      {/* Processing indicator */}
                      {isProcessing && (
                        <span className="text-[10px] text-primary">
                          Przetwarzanie...
                        </span>
                      )}
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        </div>
      )}

      <Card className="h-full flex flex-col rounded-none border-0 overflow-hidden relative">
      
      {/* Collapsible Header + Search + Controls */}
      <div 
        className={`flex-shrink-0 transition-all duration-300 ease-in-out overflow-hidden ${
          isCollapsed ? 'max-h-0 opacity-0' : 'max-h-[500px] opacity-100'
        }`}
        style={{ transitionProperty: 'max-height, opacity' }}
      >
        <CardHeader className="pb-2 px-4 border-b bg-background">
          {/* Compact header: title + inline counters */}
          <div className="flex items-center gap-6">
            <CardTitle className="flex items-center gap-2">
              <ShoppingCart className="h-5 w-5" />
              Dostępne Zamówienia
            </CardTitle>
            <div className="flex items-center gap-4 text-sm font-medium">
              <span data-testid="text-orders-count">Zamówienia: {statistics?.ordersCount ?? 0}</span>
              <Separator orientation="vertical" className="h-4" />
              <span data-testid="text-products-count">Produkty: {statistics?.productsCount ?? 0}</span>
            </div>
          </div>
        </CardHeader>

        {/* View Toggles + Date Filters - in collapsible section */}
        <div className="px-4 pt-4 pb-2 bg-background">
          {/* Responsive row with view tabs and date filters */}
          <div className="flex flex-wrap items-center gap-2 mb-2">
            {/* View Tabs */}
            <div className="flex items-center gap-[2px]">
              <button
                onClick={() => setViewMode("flat")}
                className={`
                  relative px-4 h-[32px] text-sm font-medium
                  transition-all flex items-center gap-2
                  ${viewMode === "flat" 
                    ? "bg-muted/20 text-foreground" 
                    : "bg-transparent text-muted-foreground hover:text-foreground/80"
                  }
                `}
                data-testid="toggle-flat-view"
              >
                <ShoppingCart className="h-4 w-4 text-purple-500" />
                Zamówienia
              </button>
              <button
                onClick={() => setViewMode("grouped")}
                className={`
                  relative px-4 h-[32px] text-sm font-medium
                  transition-all flex items-center gap-2
                  ${viewMode === "grouped" 
                    ? "bg-muted/20 text-foreground" 
                    : "bg-transparent text-muted-foreground hover:text-foreground/80"
                  }
                `}
                data-testid="toggle-grouped-view"
              >
                <Layers className="h-4 w-4 text-purple-500" />
                Grupowany
              </button>
            </div>
            
            {/* Date Filter Buttons */}
            <div className="flex flex-wrap items-center gap-1 ml-auto">
              <Button
                variant={filters.dateFilter === 'all' ? "default" : "outline"}
                size="sm"
                onClick={() => onFilterChange('dateFilter', 'all')}
                className="h-[25px] px-2 text-xs"
                data-testid="filter-date-all"
              >
                Wszystkie
              </Button>
              <Button
                variant={filters.dateFilter === 'today' ? "default" : "outline"}
                size="sm"
                onClick={() => onFilterChange('dateFilter', 'today')}
                className="h-[25px] px-2 text-xs"
                data-testid="filter-date-today"
              >
                Dzisiaj
              </Button>
              <Button
                variant={filters.dateFilter === 'yesterday' ? "default" : "outline"}
                size="sm"
                onClick={() => onFilterChange('dateFilter', 'yesterday')}
                className="h-[25px] px-2 text-xs"
                data-testid="filter-date-yesterday"
              >
                Wczoraj
              </Button>
              <Button
                variant={filters.dateFilter === 'day-before' ? "default" : "outline"}
                size="sm"
                onClick={() => onFilterChange('dateFilter', 'day-before')}
                className="h-[25px] px-2 text-xs"
                data-testid="filter-date-day-before"
              >
                Przedwczoraj
              </Button>
              <div className="flex items-center gap-1">
                <Input
                  type="number"
                  min="1"
                  max="365"
                  placeholder="Dni"
                  value={filters.customDays || ''}
                  onChange={(e) => {
                    onFilterChange('customDays', e.target.value);
                    if (e.target.value) {
                      onFilterChange('dateFilter', 'custom-days');
                    }
                  }}
                  className="h-[25px] w-[60px] text-xs px-2"
                  data-testid="input-custom-days"
                />
                <span className="text-xs text-muted-foreground">dni temu</span>
              </div>
            </div>
          </div>

          {/* Controls Bar + Toggle Filters */}
          <div className="flex flex-wrap items-center justify-between gap-2 py-1 bg-muted/30 px-4">
            {/* Left: Toggle Filters */}
            <div className="flex items-center gap-1">
              <Button
                variant="outline"
                size="sm"
                onClick={() => onFilterChange('showSetsOnly', filters.showSetsOnly === 'true' ? 'false' : 'true')}
                className={`h-[25px] px-2 text-xs gap-1 ${filters.showSetsOnly === 'true' ? 'bg-purple-600 hover:bg-purple-700 text-white border-purple-600' : ''}`}
                data-testid="toggle-sets-only"
                title="Pokaż tylko zestawy"
              >
                <Package className="h-3 w-3" />
                Zestawy
              </Button>
              
              <Button
                variant="outline"
                size="sm"
                onClick={() => onFilterChange('showCatalogLinked', filters.showCatalogLinked === 'true' ? 'false' : 'true')}
                className={`h-[25px] px-2 text-xs gap-1 ${filters.showCatalogLinked === 'true' ? 'bg-purple-600 hover:bg-purple-700 text-white border-purple-600' : ''}`}
                data-testid="toggle-catalog-linked"
                title="Pokaż tylko połączone z katalogiem"
              >
                <LinkIcon className="h-3 w-3" />
                Połączone
              </Button>
              
              <Button
                variant="outline"
                size="sm"
                onClick={() => onFilterChange('showInPlans', filters.showInPlans === 'true' ? 'false' : 'true')}
                className={`h-[25px] px-2 text-xs gap-1 ${filters.showInPlans === 'true' ? 'bg-blue-600 hover:bg-blue-700 text-white border-blue-600' : ''}`}
                data-testid="toggle-in-plans"
                title="Pokaż tylko produkty już dodane do planów"
              >
                <ClipboardList className="h-3 w-3" />
                W planach
              </Button>
            </div>
            
            {/* Right: Sort Controls */}
            <OrdersControlBar
              viewMode={viewMode}
              allExpanded={allExpanded}
              onExpandAll={handleExpandAll}
              onCollapseAll={handleCollapseAll}
              pageSize={filters.limit || 100}
              onPageSizeChange={handlePageSizeChange}
              sortBy={filters.sortBy || 'order_date'}
              sortOrder={filters.sortOrder || 'desc'}
              onSortByChange={handleSortByChange}
              onSortOrderToggle={handleSortOrderToggle}
            />
          </div>
        </div>
      </div>

      {/* Bulk Actions Toolbar - Sticky when items selected */}
      {selectedItemIds.size > 0 && (
        <div className="sticky top-0 z-30 flex-shrink-0 flex items-center justify-between h-[30px] px-4 bg-primary/10 border-b">
          <span className="text-xs font-medium">
            Zaznaczono: {selectedItemIds.size}
          </span>
          <div className="flex items-center gap-2">
            <Button
              size="sm"
              variant="ghost"
              onClick={clearSelection}
              className="h-6 text-xs"
              data-testid="button-clear-selection"
            >
              Wyczyść
            </Button>
            <Button
              size="sm"
              onClick={handleBulkAdd}
              className="h-6 text-xs"
              data-testid="button-bulk-add"
            >
              <Plus className="h-3 w-3 mr-1" />
              Dodaj do planu
            </Button>
          </div>
        </div>
      )}

      {/* OrdersTabsBar - Sticky Filters */}
      <div 
        className={`sticky ${selectedItemIds.size > 0 ? 'top-[30px]' : 'top-0'} z-20 flex-shrink-0 px-4 pt-4 pb-3 bg-background border-b`} 
        data-testid="orders-tabs-section"
      >
        <OrdersTabsBar
          planId={planId}
          searchFilter={searchFilter}
          onSearchChange={onSearchChange}
          showSetsOnly={filters.showSetsOnly === 'true'}
          showCatalogLinked={filters.showCatalogLinked === 'true'}
          onToggleSetsOnly={() => onFilterChange('showSetsOnly', filters.showSetsOnly === 'true' ? 'false' : 'true')}
          onToggleCatalogLinked={() => onFilterChange('showCatalogLinked', filters.showCatalogLinked === 'true' ? 'false' : 'true')}
        />
      </div>
      {/* Scrollable Orders List */}
      <div ref={scrollContainerRef} className="flex-1 overflow-y-auto px-4 pb-4">
        {isLoading ? (
          <div className="flex items-center justify-center p-8">
            <Loader2 className="h-6 w-6 animate-spin" />
          </div>
        ) : viewMode === "flat" ? (
          <CompactOrdersList 
            orders={orders} 
            colors={colors}
            productTypes={productTypes}
            onAddItem={onAddItem}
            isAddingItem={isAddingItem}
            recentlyAddedItems={recentlyAddedItems}
            selectedItemIds={selectedItemIds}
            onToggleItemSelection={toggleItemSelection}
            expandedSets={expandedSets}
            setExpandedSets={setExpandedSets}
            onHighlightInPlan={onHighlightInPlan}
          />
        ) : (
          <GroupedOrdersList
            orders={orders}
            colors={colors}
            productTypes={productTypes}
            onAddItem={onAddItem}
            isAddingItem={isAddingItem}
            recentlyAddedItems={recentlyAddedItems}
            selectedItemIds={selectedItemIds}
            onToggleItemSelection={toggleItemSelection}
            expandedColorGroups={expandedColorGroups}
            setExpandedColorGroups={setExpandedColorGroups}
            expandedTypeGroups={expandedTypeGroups}
            setExpandedTypeGroups={setExpandedTypeGroups}
            expandedSets={expandedSets}
            setExpandedSets={setExpandedSets}
            onHighlightInPlan={onHighlightInPlan}
          />
        )}
      </div>
    </Card>
    </>
  );
}
