import { ShoppingCart, Package, Scissors } from "lucide-react";
import { cn } from "@/lib/utils";

export type ProductSource = 'customer_orders' | 'catalog_products' | 'cutting_patterns';

interface CategoryTileProps {
  icon: React.ElementType;
  title: string;
  description: string;
  isActive: boolean;
  onClick: () => void;
  testId: string;
}

function CategoryTile({ icon: Icon, title, description, isActive, onClick, testId }: CategoryTileProps) {
  return (
    <button
      onClick={onClick}
      className={cn(
        "flex-1 flex flex-col items-center gap-1 p-3 rounded-md transition-all hover-elevate",
        isActive 
          ? "bg-primary/10 border-2 border-primary" 
          : "bg-muted/20 border-2 border-transparent hover:border-muted-foreground/20"
      )}
      data-testid={testId}
    >
      <Icon className={cn(
        "h-6 w-6",
        isActive ? "text-primary" : "text-muted-foreground"
      )} />
      <div className="flex flex-col items-center gap-0.5">
        <span className={cn(
          "text-sm font-medium text-center",
          isActive ? "text-primary" : "text-foreground"
        )}>
          {title}
        </span>
        <span className="text-xs text-muted-foreground text-center">
          {description}
        </span>
      </div>
    </button>
  );
}

interface CategorySelectorProps {
  selectedSource: ProductSource;
  onSourceChange: (source: ProductSource) => void;
  compact?: boolean;
}

export function CategorySelector({ 
  selectedSource, 
  onSourceChange, 
  compact = false
}: CategorySelectorProps) {
  // Compact mode: 3 small buttons in one row (40px height)
  if (compact) {
    return (
      <div className="flex gap-2 mb-2" data-testid="category-selector-compact">
        {/* Customer Orders */}
        <button
          onClick={() => onSourceChange('customer_orders')}
          className={cn(
            "flex-1 flex items-center gap-1.5 px-3 h-10 rounded-md transition-all hover-elevate text-sm font-medium",
            selectedSource === 'customer_orders'
              ? "bg-primary/10 border-2 border-primary text-primary"
              : "bg-muted/20 border-2 border-transparent text-muted-foreground hover:border-muted-foreground/20"
          )}
          data-testid="category-compact-customer-orders"
        >
          <ShoppingCart className="h-4 w-4 flex-shrink-0" />
          <span className="truncate">Zamówienia</span>
        </button>

        {/* Catalog Products */}
        <button
          onClick={() => onSourceChange('catalog_products')}
          className={cn(
            "flex-1 flex items-center gap-1.5 px-3 h-10 rounded-md transition-all hover-elevate text-sm font-medium",
            selectedSource === 'catalog_products'
              ? "bg-primary/10 border-2 border-primary text-primary"
              : "bg-muted/20 border-2 border-transparent text-muted-foreground hover:border-muted-foreground/20"
          )}
          data-testid="category-compact-catalog-products"
        >
          <Package className="h-4 w-4 flex-shrink-0" />
          <span className="truncate">Katalog</span>
        </button>

        {/* Cutting Patterns */}
        <button
          onClick={() => onSourceChange('cutting_patterns')}
          className={cn(
            "flex-1 flex items-center gap-1.5 px-3 h-10 rounded-md transition-all hover-elevate text-sm font-medium",
            selectedSource === 'cutting_patterns'
              ? "bg-primary/10 border-2 border-primary text-primary"
              : "bg-muted/20 border-2 border-transparent text-muted-foreground hover:border-muted-foreground/20"
          )}
          data-testid="category-compact-cutting-patterns"
        >
          <Scissors className="h-4 w-4 flex-shrink-0" />
          <span className="truncate">Rozkroje</span>
        </button>
      </div>
    );
  }

  // Full mode: three tiles with descriptions
  return (
    <div className="flex gap-2 mb-2" data-testid="category-selector">
      <CategoryTile
        icon={ShoppingCart}
        title="Zamówienia klientów"
        description="Produkty z marketplace"
        isActive={selectedSource === 'customer_orders'}
        onClick={() => onSourceChange('customer_orders')}
        testId="category-tile-customer-orders"
      />
      <CategoryTile
        icon={Package}
        title="Produkty katalogowe"
        description="Zamówienia wewnętrzne"
        isActive={selectedSource === 'catalog_products'}
        onClick={() => onSourceChange('catalog_products')}
        testId="category-tile-catalog-products"
      />
      <CategoryTile
        icon={Scissors}
        title="Rozkroje"
        description="Szablony rozkrojów"
        isActive={selectedSource === 'cutting_patterns'}
        onClick={() => onSourceChange('cutting_patterns')}
        testId="category-tile-cutting-patterns"
      />
    </div>
  );
}
