from odoo import _, api, fields, models
from odoo.exceptions import ValidationError

from .allegro_client import AllegroPayment
from .market_mixin import ALLEGRO


class PaymentTerms(models.Model):
    _inherit = 'trilab.market.payment_terms'

    allegro_code = fields.Selection(
        [
            (AllegroPayment.CASH_ON_DELIVERY, 'Cash on Delivery'),
            (AllegroPayment.WIRE_TRANSFER, 'Wire Transfer'),
            (AllegroPayment.ONLINE, 'Online Payment'),
            (AllegroPayment.SPLIT_PAYMENT, 'Split Payment'),
            (AllegroPayment.EXTENDED_TERM, 'Extended Term'),
        ],
        string='Allegro Payment Type',
        index=True,
    )

    def _compute_display_name(self):
        allegro_term_ids = self.filtered(lambda t_id: t_id.account_id.is_allegro())
        for term_id in allegro_term_ids:
            term_id.display_name = term_id.allegro_code

        super(PaymentTerms, self - allegro_term_ids)._compute_display_name()

    @api.constrains('account_id', 'allegro_code')
    def _allegro_constrain_payment_method(self):
        for term_id in self.filtered(lambda t_id: t_id.account_id.is_allegro()):
            if (
                self.search_count(
                    [('account_id', '=', term_id.account_id.id), ('allegro_code', '=', term_id.allegro_code)]
                )
                > 1
            ):
                raise ValidationError(
                    _(
                        'Allegro Payment Type "%s" already exists for this account!',
                        dict(self._fields['allegro_code'].selection).get(self.allegro_code),
                    )
                )

            term_id.payment_method_id = self.env.ref(f'trilab_allegro.payment_method_{term_id.allegro_code.lower()}').id
            term_id.payment_provider_id = self.env['payment.provider'].search(
                [('code', '=', ALLEGRO), ('company_id', '=', term_id.company_id.id)], limit=1
            )
            term_id.is_cod = term_id.allegro_code == AllegroPayment.CASH_ON_DELIVERY
