import { Pool } from "pg";
import type { ProductionPlan, ProductionPlanLine } from "@shared/schema";

interface InsertProductionPlan {
  planNumber?: string;
  name: string;
  description?: string | null;
  plannedStartDate?: Date | null;
  plannedEndDate?: Date | null;
  status?: string;
  priority?: string;
  notes?: string | null;
  metadata?: any;
  createdBy?: number | null;
}

interface InsertProductionPlanLine {
  planId: number;
  productId: number;
  quantity: number;
  sourceType?: string | null;
  sourceId?: number | null;
  sourceReference?: string | null;
  productionOrderId?: number | null;
  routingId?: number | null;
  routingOverride?: any;
  bomId?: number | null;
  plannedStartDate?: Date | null;
  plannedEndDate?: Date | null;
  status?: string;
  sequence?: number | null;
  notes?: string | null;
  metadata?: any;
}

function mapRowToPlan(row: any): ProductionPlan {
  return {
    id: row.id,
    planNumber: row.plan_number,
    name: row.name,
    description: row.description,
    plannedStartDate: row.planned_start_date,
    plannedEndDate: row.planned_end_date,
    actualStartDate: row.actual_start_date,
    actualEndDate: row.actual_end_date,
    status: row.status,
    priority: row.priority,
    notes: row.notes,
    metadata: row.metadata,
    createdBy: row.created_by,
    approvedBy: row.approved_by,
    approvedAt: row.approved_at,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

function mapRowToPlanLine(row: any): ProductionPlanLine {
  return {
    id: row.id,
    planId: row.plan_id,
    productId: row.product_id,
    quantity: row.quantity,
    sourceType: row.source_type,
    sourceId: row.source_id,
    sourceReference: row.source_reference,
    productionOrderId: row.production_order_id,
    routingId: row.routing_id,
    routingOverride: row.routing_override,
    bomId: row.bom_id,
    plannedStartDate: row.planned_start_date,
    plannedEndDate: row.planned_end_date,
    status: row.status,
    sequence: row.sequence,
    notes: row.notes,
    metadata: row.metadata,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

interface PlanFilters {
  status?: string;
  priority?: string;
  createdBy?: number;
  startDate?: Date;
  endDate?: Date;
  search?: string;
  limit?: number;
  offset?: number;
}

export async function getPlans(pool: Pool, filters?: PlanFilters): Promise<ProductionPlan[]> {
  let query = `SELECT * FROM production.production_plans WHERE 1=1`;
  const params: any[] = [];
  let paramIndex = 1;

  if (filters?.status) {
    query += ` AND status = $${paramIndex++}`;
    params.push(filters.status);
  }

  if (filters?.priority) {
    query += ` AND priority = $${paramIndex++}`;
    params.push(filters.priority);
  }

  if (filters?.createdBy) {
    query += ` AND created_by = $${paramIndex++}`;
    params.push(filters.createdBy);
  }

  if (filters?.startDate) {
    query += ` AND planned_start_date >= $${paramIndex++}`;
    params.push(filters.startDate);
  }

  if (filters?.endDate) {
    query += ` AND planned_end_date <= $${paramIndex++}`;
    params.push(filters.endDate);
  }

  if (filters?.search) {
    query += ` AND (plan_number ILIKE $${paramIndex} OR name ILIKE $${paramIndex})`;
    params.push(`%${filters.search}%`);
    paramIndex++;
  }

  query += ` ORDER BY created_at DESC`;

  if (filters?.limit) {
    query += ` LIMIT $${paramIndex++}`;
    params.push(filters.limit);
  }

  if (filters?.offset) {
    query += ` OFFSET $${paramIndex++}`;
    params.push(filters.offset);
  }

  const result = await pool.query(query, params);
  return result.rows.map(mapRowToPlan);
}

export async function getPlanById(pool: Pool, id: number): Promise<ProductionPlan | null> {
  const result = await pool.query(
    'SELECT * FROM production.production_plans WHERE id = $1',
    [id]
  );
  
  if (result.rows.length === 0) {
    return null;
  }
  
  return mapRowToPlan(result.rows[0]);
}

export async function generatePlanNumber(pool: Pool): Promise<string> {
  const result = await pool.query(
    `SELECT plan_number FROM production.production_plans 
     WHERE plan_number ~ '^PLAN-[0-9]{4}$' 
     ORDER BY plan_number DESC LIMIT 1`
  );

  if (result.rows.length === 0) {
    return 'PLAN-0001';
  }

  const lastNumber = parseInt(result.rows[0].plan_number.split('-')[1]);
  return `PLAN-${String(lastNumber + 1).padStart(4, '0')}`;
}

export async function createPlan(pool: Pool, data: InsertProductionPlan): Promise<ProductionPlan> {
  const planNumber = data.planNumber || await generatePlanNumber(pool);

  const result = await pool.query(
    `INSERT INTO production.production_plans 
      (plan_number, name, description, planned_start_date, planned_end_date, 
       status, priority, notes, metadata, created_by, created_at, updated_at)
     VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
     RETURNING *`,
    [
      planNumber,
      data.name,
      data.description,
      data.plannedStartDate,
      data.plannedEndDate,
      data.status || 'draft',
      data.priority || 'normal',
      data.notes,
      data.metadata ? JSON.stringify(data.metadata) : null,
      data.createdBy,
    ]
  );

  return mapRowToPlan(result.rows[0]);
}

export async function updatePlan(
  pool: Pool,
  id: number,
  data: Partial<InsertProductionPlan>
): Promise<ProductionPlan | null> {
  const fields: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.name !== undefined) {
    fields.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }

  if (data.description !== undefined) {
    fields.push(`description = $${paramIndex++}`);
    values.push(data.description);
  }

  if (data.plannedStartDate !== undefined) {
    fields.push(`planned_start_date = $${paramIndex++}`);
    values.push(data.plannedStartDate);
  }

  if (data.plannedEndDate !== undefined) {
    fields.push(`planned_end_date = $${paramIndex++}`);
    values.push(data.plannedEndDate);
  }

  if (data.status !== undefined) {
    fields.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }

  if (data.priority !== undefined) {
    fields.push(`priority = $${paramIndex++}`);
    values.push(data.priority);
  }

  if (data.notes !== undefined) {
    fields.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (data.metadata !== undefined) {
    fields.push(`metadata = $${paramIndex++}`);
    values.push(JSON.stringify(data.metadata));
  }

  if (fields.length === 0) {
    return getPlanById(pool, id);
  }

  fields.push(`updated_at = CURRENT_TIMESTAMP`);
  values.push(id);

  const result = await pool.query(
    `UPDATE production.production_plans 
     SET ${fields.join(', ')} 
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );

  if (result.rows.length === 0) {
    return null;
  }

  return mapRowToPlan(result.rows[0]);
}

export async function deletePlan(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(
    'DELETE FROM production.production_plans WHERE id = $1',
    [id]
  );
  
  return result.rowCount !== null && result.rowCount > 0;
}

export async function getPlanLines(pool: Pool, planId: number): Promise<ProductionPlanLine[]> {
  const result = await pool.query(
    `SELECT * FROM production.production_plan_lines 
     WHERE plan_id = $1 
     ORDER BY sequence ASC, created_at ASC`,
    [planId]
  );
  
  return result.rows.map(mapRowToPlanLine);
}

export async function getPlanLineById(pool: Pool, id: number): Promise<ProductionPlanLine | null> {
  const result = await pool.query(
    'SELECT * FROM production.production_plan_lines WHERE id = $1',
    [id]
  );
  
  if (result.rows.length === 0) {
    return null;
  }
  
  return mapRowToPlanLine(result.rows[0]);
}

export async function createPlanLine(pool: Pool, data: InsertProductionPlanLine): Promise<ProductionPlanLine> {
  const result = await pool.query(
    `INSERT INTO production.production_plan_lines 
      (plan_id, product_id, quantity, source_type, source_id, source_reference,
       production_order_id, routing_id, routing_override, bom_id,
       planned_start_date, planned_end_date, status, sequence, notes, metadata,
       created_at, updated_at)
     VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, 
             CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
     RETURNING *`,
    [
      data.planId,
      data.productId,
      data.quantity,
      data.sourceType,
      data.sourceId,
      data.sourceReference,
      data.productionOrderId,
      data.routingId,
      data.routingOverride ? JSON.stringify(data.routingOverride) : null,
      data.bomId,
      data.plannedStartDate,
      data.plannedEndDate,
      data.status || 'pending',
      data.sequence,
      data.notes,
      data.metadata ? JSON.stringify(data.metadata) : null,
    ]
  );

  return mapRowToPlanLine(result.rows[0]);
}

export async function updatePlanLine(
  pool: Pool,
  id: number,
  data: Partial<InsertProductionPlanLine>
): Promise<ProductionPlanLine | null> {
  const fields: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.productId !== undefined) {
    fields.push(`product_id = $${paramIndex++}`);
    values.push(data.productId);
  }

  if (data.quantity !== undefined) {
    fields.push(`quantity = $${paramIndex++}`);
    values.push(data.quantity);
  }

  if (data.sourceType !== undefined) {
    fields.push(`source_type = $${paramIndex++}`);
    values.push(data.sourceType);
  }

  if (data.sourceId !== undefined) {
    fields.push(`source_id = $${paramIndex++}`);
    values.push(data.sourceId);
  }

  if (data.sourceReference !== undefined) {
    fields.push(`source_reference = $${paramIndex++}`);
    values.push(data.sourceReference);
  }

  if (data.productionOrderId !== undefined) {
    fields.push(`production_order_id = $${paramIndex++}`);
    values.push(data.productionOrderId);
  }

  if (data.routingId !== undefined) {
    fields.push(`routing_id = $${paramIndex++}`);
    values.push(data.routingId);
  }

  if (data.routingOverride !== undefined) {
    fields.push(`routing_override = $${paramIndex++}`);
    values.push(data.routingOverride ? JSON.stringify(data.routingOverride) : null);
  }

  if (data.bomId !== undefined) {
    fields.push(`bom_id = $${paramIndex++}`);
    values.push(data.bomId);
  }

  if (data.plannedStartDate !== undefined) {
    fields.push(`planned_start_date = $${paramIndex++}`);
    values.push(data.plannedStartDate);
  }

  if (data.plannedEndDate !== undefined) {
    fields.push(`planned_end_date = $${paramIndex++}`);
    values.push(data.plannedEndDate);
  }

  if (data.status !== undefined) {
    fields.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }

  if (data.sequence !== undefined) {
    fields.push(`sequence = $${paramIndex++}`);
    values.push(data.sequence);
  }

  if (data.notes !== undefined) {
    fields.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (data.metadata !== undefined) {
    fields.push(`metadata = $${paramIndex++}`);
    values.push(data.metadata ? JSON.stringify(data.metadata) : null);
  }

  if (fields.length === 0) {
    return getPlanLineById(pool, id);
  }

  fields.push(`updated_at = CURRENT_TIMESTAMP`);
  values.push(id);

  const result = await pool.query(
    `UPDATE production.production_plan_lines 
     SET ${fields.join(', ')} 
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );

  if (result.rows.length === 0) {
    return null;
  }

  return mapRowToPlanLine(result.rows[0]);
}

export async function deletePlanLine(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(
    'DELETE FROM production.production_plan_lines WHERE id = $1',
    [id]
  );
  
  return result.rowCount !== null && result.rowCount > 0;
}

export interface DemandFilters {
  startDate?: Date;
  endDate?: Date;
  marketplace?: string;
  orderStatus?: string[];
  paymentStatus?: string[];
}

export interface AggregatedDemand {
  productId: number;
  sku: string;
  title: string;
  totalQuantity: number;
  orderCount: number;
  orderReferences: Array<{
    orderId: number;
    orderNumber: string;
    marketplace: string;
    quantity: number;
  }>;
}

export async function aggregateDemand(
  pool: Pool,
  filters?: DemandFilters
): Promise<AggregatedDemand[]> {
  let query = `
    WITH order_product_mapping AS (
      SELECT 
        oi.id as order_item_id,
        oi.order_id,
        oi.quantity,
        oi.name as product_name,
        oi.raw_data->'offer'->'external'->>'id' as external_sku,
        o.source as marketplace,
        o.order_number,
        o.status as order_status,
        o.payment_status,
        o.order_date
      FROM commerce.order_items oi
      JOIN commerce.orders o ON oi.order_id = o.id
      WHERE oi.raw_data->'offer'->'external'->>'id' IS NOT NULL
    ),
    catalog_matched AS (
      SELECT 
        opm.*,
        cp.id as catalog_product_id,
        cp.sku as catalog_sku,
        cp.title as catalog_title
      FROM order_product_mapping opm
      LEFT JOIN catalog.products cp ON cp.sku = opm.external_sku
      WHERE cp.id IS NOT NULL
    )
    SELECT 
      cm.catalog_product_id,
      cm.catalog_sku,
      cm.catalog_title,
      SUM(cm.quantity) as total_quantity,
      COUNT(DISTINCT cm.order_id) as order_count,
      jsonb_agg(
        jsonb_build_object(
          'orderId', cm.order_id,
          'orderNumber', cm.order_number,
          'marketplace', cm.marketplace,
          'quantity', cm.quantity
        ) ORDER BY cm.order_date DESC
      ) as order_references
    FROM catalog_matched cm
    WHERE 1=1
  `;

  const params: any[] = [];
  let paramIndex = 1;

  if (filters?.startDate) {
    query += ` AND cm.order_date >= $${paramIndex++}`;
    params.push(filters.startDate);
  }

  if (filters?.endDate) {
    query += ` AND cm.order_date <= $${paramIndex++}`;
    params.push(filters.endDate);
  }

  if (filters?.marketplace) {
    query += ` AND cm.marketplace = $${paramIndex++}`;
    params.push(filters.marketplace);
  }

  if (filters?.orderStatus && filters.orderStatus.length > 0) {
    query += ` AND cm.order_status = ANY($${paramIndex++})`;
    params.push(filters.orderStatus);
  }

  if (filters?.paymentStatus && filters.paymentStatus.length > 0) {
    query += ` AND cm.payment_status = ANY($${paramIndex++})`;
    params.push(filters.paymentStatus);
  }

  query += `
    GROUP BY cm.catalog_product_id, cm.catalog_sku, cm.catalog_title
    ORDER BY total_quantity DESC
  `;

  const result = await pool.query(query, params);

  return result.rows.map(row => ({
    productId: row.catalog_product_id,
    sku: row.catalog_sku,
    title: row.catalog_title,
    totalQuantity: parseInt(row.total_quantity),
    orderCount: parseInt(row.order_count),
    orderReferences: row.order_references || []
  }));
}

export async function getPlanLinesWithDetails(pool: Pool, planId: number) {
  const result = await pool.query(`
    SELECT 
      ppl.id,
      ppl.plan_id,
      ppl.product_id,
      ppl.quantity,
      ppl.source_type,
      ppl.source_id,
      ppl.source_reference,
      ppl.production_order_id,
      ppl.routing_id,
      ppl.routing_override,
      ppl.bom_id,
      ppl.planned_start_date,
      ppl.planned_end_date,
      ppl.status,
      ppl.sequence,
      ppl.notes,
      ppl.metadata,
      ppl.created_at,
      ppl.updated_at,
      
      -- Product data from catalog
      cp.sku as product_sku,
      cp.title as product_title,
      cp.color as product_color,
      cp.color_options as product_color_options,
      cp.length as product_length,
      cp.width as product_width,
      cp.height as product_height,
      cp.product_type,
      cp.product_group,
      cp.doors,
      cp.legs,
      cp.material,
      cp.base_price,
      cp.currency,
      COALESCE(
        ppl.metadata->>'image_url',
        (
          SELECT url
          FROM jsonb_array_elements_text(COALESCE(cp.gallery, '[]'::jsonb)) WITH ORDINALITY AS t(url, idx)
          ORDER BY idx
          LIMIT 1
        )
      ) as product_image,
      
      -- Order data from commerce.orders (if source_type = 'order_demand')
      o.order_number,
      o.order_date,
      o.source as marketplace,
      o.buyer_first_name,
      o.buyer_last_name,
      o.buyer_email,
      o.payment_status as order_payment_status,
      o.total_to_pay_amount as order_total_amount,
      
      -- BOM component count
      (SELECT COUNT(*) 
       FROM bom.product_components pc 
       INNER JOIN bom.product_boms pb ON pc.product_bom_id = pb.id 
       WHERE pb.product_id = cp.id) as bom_count,
      
      -- Order item ID from metadata (for duplicate prevention)
      (ppl.metadata->>'orderItemId')::integer as order_item_id
      
    FROM production.production_plan_lines ppl
    INNER JOIN catalog.products cp ON ppl.product_id = cp.id
    LEFT JOIN commerce.orders o ON (
      ppl.source_type = 'order_demand' 
      AND (
        ppl.source_reference = o.order_number::text
        OR ppl.source_id = o.id
        OR (ppl.metadata->>'order_number') = o.order_number::text
      )
    )
    WHERE ppl.plan_id = $1
    ORDER BY ppl.sequence ASC NULLS LAST, ppl.created_at ASC
  `, [planId]);

  return result.rows;
}
