import { useMutation, useQuery } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useLocation } from "wouter";
import { ArrowLeft, Wand2 } from "lucide-react";
import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import { z } from "zod";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";

const planFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().optional(),
  plannedStartDate: z.string().optional(),
  plannedEndDate: z.string().optional(),
  status: z.enum(['draft', 'approved', 'in_progress', 'completed', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  notes: z.string().optional(),
});

type PlanFormValues = z.infer<typeof planFormSchema>;

export default function ProductionPlanNew() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [selectedTemplateId, setSelectedTemplateId] = useState<string>("");

  const form = useForm<PlanFormValues>({
    resolver: zodResolver(planFormSchema),
    defaultValues: {
      name: "",
      description: "",
      status: "draft",
      priority: "normal",
      notes: "",
    },
  });

  const { data: templates, isLoading: templatesLoading } = useQuery<any[]>({
    queryKey: ["/api/production/plan-name-templates"],
  });

  const generateNameMutation = useMutation({
    mutationFn: async (templateId: number) => {
      const response = await fetch("/api/production/plan-name-templates/generate", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({ templateId }),
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to generate name" }));
        throw new Error(error.message);
      }
      return await response.json();
    },
    onSuccess: (data) => {
      form.setValue("name", data.generatedName);
      toast({
        title: "Sukces",
        description: `Wygenerowano nazwę: ${data.generatedName}`,
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się wygenerować nazwy",
      });
    },
  });

  const handleGenerateName = () => {
    if (!selectedTemplateId) {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: "Wybierz szablon nazwy",
      });
      return;
    }
    generateNameMutation.mutate(parseInt(selectedTemplateId, 10));
  };

  const createMutation = useMutation({
    mutationFn: async (data: PlanFormValues) => {
      const response = await fetch("/api/production/planning/plans", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify(data),
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to create plan" }));
        throw new Error(error.message);
      }
      return await response.json();
    },
    onSuccess: (createdPlan) => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/planning/plans"] });
      toast({
        title: "Sukces",
        description: "Plan produkcji został utworzony",
      });
      setLocation(`/production/plans/${createdPlan.id}`);
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć planu",
      });
    },
  });

  const handleSubmit = (data: PlanFormValues) => {
    createMutation.mutate(data);
  };

  const handleCancel = () => {
    setLocation("/production/plans");
  };

  return (
    <div className="container mx-auto p-6 max-w-4xl">
      <div className="mb-6">
        <Button
          variant="ghost"
          onClick={handleCancel}
          className="mb-4"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Powrót do listy planów
        </Button>
        <h1 className="text-3xl font-bold" data-testid="text-page-title">Nowy Plan Produkcji</h1>
        <p className="text-muted-foreground mt-2">Utwórz nowy plan produkcji</p>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane Planu</CardTitle>
          <CardDescription>Wypełnij formularz aby utworzyć nowy plan produkcji</CardDescription>
        </CardHeader>
        <CardContent>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-4">
              <div className="space-y-4 p-4 bg-muted/30 rounded-lg border">
                <div className="flex items-center gap-2">
                  <Wand2 className="h-4 w-4 text-muted-foreground" />
                  <h3 className="text-sm font-medium">Generator nazwy planu</h3>
                </div>
                <div className="flex gap-2">
                  <div className="flex-1">
                    <Select 
                      value={selectedTemplateId} 
                      onValueChange={setSelectedTemplateId}
                      disabled={templatesLoading}
                    >
                      <SelectTrigger data-testid="select-name-template">
                        <SelectValue placeholder="Wybierz szablon nazwy..." />
                      </SelectTrigger>
                      <SelectContent>
                        {templates?.map((template) => (
                          <SelectItem key={template.id} value={String(template.id)} data-testid={`template-${template.id}`}>
                            <div className="flex flex-col">
                              <span className="font-medium">{template.templateName}</span>
                              {template.description && (
                                <span className="text-xs text-muted-foreground">{template.description}</span>
                              )}
                            </div>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <Button
                    type="button"
                    variant="secondary"
                    onClick={handleGenerateName}
                    disabled={!selectedTemplateId || generateNameMutation.isPending}
                    data-testid="button-generate-name"
                  >
                    {generateNameMutation.isPending ? "Generuję..." : "Użyj szablonu"}
                  </Button>
                </div>
                {selectedTemplateId && templates && (
                  <p className="text-xs text-muted-foreground">
                    Wzorzec: {templates.find(t => String(t.id) === selectedTemplateId)?.pattern}
                  </p>
                )}
              </div>

              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa planu *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="Plan produkcji styczeń 2025" data-testid="input-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="Opcjonalny opis planu..." data-testid="input-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="plannedStartDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Data rozpoczęcia</FormLabel>
                      <FormControl>
                        <Input type="date" {...field} data-testid="input-start-date" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="plannedEndDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Data zakończenia</FormLabel>
                      <FormControl>
                        <Input type="date" {...field} data-testid="input-end-date" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="status"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Status</FormLabel>
                      <Select value={field.value} onValueChange={field.onChange}>
                        <FormControl>
                          <SelectTrigger data-testid="select-status">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="draft">Projekt</SelectItem>
                          <SelectItem value="approved">Zatwierdzony</SelectItem>
                          <SelectItem value="in_progress">W realizacji</SelectItem>
                          <SelectItem value="completed">Zakończony</SelectItem>
                          <SelectItem value="cancelled">Anulowany</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="priority"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Priorytet</FormLabel>
                      <Select value={field.value} onValueChange={field.onChange}>
                        <FormControl>
                          <SelectTrigger data-testid="select-priority">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="low">Niski</SelectItem>
                          <SelectItem value="normal">Normalny</SelectItem>
                          <SelectItem value="high">Wysoki</SelectItem>
                          <SelectItem value="urgent">Pilny</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Uwagi</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="Dodatkowe uwagi..." data-testid="input-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2 pt-4">
                <Button
                  type="button"
                  variant="outline"
                  onClick={handleCancel}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={createMutation.isPending}
                  data-testid="button-submit"
                >
                  {createMutation.isPending ? "Tworzenie..." : "Utwórz Plan"}
                </Button>
              </div>
            </form>
          </Form>
        </CardContent>
      </Card>
    </div>
  );
}
