import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Package, Plus, Search, Edit, Trash2, Image, Check, ChevronDown, ChevronUp, X, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, ArrowUpDown, Image as ImageIcon, FileText, FileSpreadsheet, FileDown, Upload, Loader2, Copy, ExternalLink } from "lucide-react";
import { useState, useEffect, useMemo, Fragment } from "react";
import { Skeleton } from "@/components/ui/skeleton";
import { useLocation, Link } from "wouter";
import { Checkbox } from "@/components/ui/checkbox";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { DictionaryCombobox } from "@/components/dictionary-combobox";
import { Switch } from "@/components/ui/switch";
import { ConnectionStatusIcon } from "@/components/connection-status-icon";
import { MarketplaceLinkDialog } from "@/components/marketplace-catalog-link-dialog";
import { SavedFiltersManager } from "@/components/saved-filters-manager";

interface CatalogProduct {
  id: number;
  sku: string;
  skuAirtable: string | null;
  title: string;
  shortDescription: string | null;
  length: number | null;
  width: number | null;
  height: number | null;
  weight: number | null;
  color: string | null;
  colorOptions: string[] | null;
  material: string | null;
  productType: string | null;
  productGroup: string | null;
  doors: string | null;
  legs: string | null;
  basePrice: number | null;
  currency: string;
  isActive: boolean;
  odooProductId: number | null;
  odooExportedAt: string | null;
  odooExportError: string | null;
  imagesCount: number;
  addonsCount: number;
  bomCount: number;
  primaryImageUrl: string | null;
  marketplaceLinksCount: number;
  firstMarketplaceSku: string | null;
  createdAt: string;
  updatedAt: string;
}

interface ProductCreatorDictionary {
  id: number;
  dictionaryType: string;
  code: string;
  name: string;
  readableName: string | null;
  color: string | null;
  isActive: boolean;
}

const STORAGE_KEY = 'catalog-products-filters';

// Helper function to determine text color based on background luminance
function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  // Calculate luminance using standard formula
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}

// Helper function to determine if color needs visible border (very dark or very light colors)
function needsVisibleBorder(hexColor: string | null | undefined): boolean {
  if (!hexColor || !hexColor.startsWith('#')) return false;
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  // Very dark (< 30) or very light (> 230) colors need border for visibility
  return luminance < 30 || luminance > 230;
}

export default function CatalogProductsPage() {
  const { toast } = useToast();
  const [, setLocation] = useLocation();
  
  // Load filters from localStorage
  const loadFilters = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return {};
  };

  const savedFilters = loadFilters();
  
  const [searchQuery, setSearchQuery] = useState(savedFilters.searchQuery || "");
  const [selectedProducts, setSelectedProducts] = useState<number[]>([]);
  const [isBulkEditOpen, setIsBulkEditOpen] = useState(false);
  const [isBulkDeleteOpen, setIsBulkDeleteOpen] = useState(false);
  const [linkDialogOpen, setLinkDialogOpen] = useState(false);
  const [selectedProductForLinking, setSelectedProductForLinking] = useState<CatalogProduct | null>(null);
  const [isGeneratingBom, setIsGeneratingBom] = useState(false);
  const [filtersExpanded, setFiltersExpanded] = useState(false);
  const [exportingProducts, setExportingProducts] = useState<Set<number>>(new Set());
  const [isBulkExporting, setIsBulkExporting] = useState(false);
  const [isBulkBomExporting, setIsBulkBomExporting] = useState(false);

  // Pagination and sorting
  const [page, setPage] = useState(savedFilters.page || 1);
  const [limit, setLimit] = useState(savedFilters.limit || 25);
  const [sortBy, setSortBy] = useState(savedFilters.sortBy || 'createdAt');
  const [sortOrder, setSortOrder] = useState<'ASC' | 'DESC'>(savedFilters.sortOrder || 'DESC');
  const [compactMode, setCompactMode] = useState(savedFilters.compactMode ?? true);
  const [groupBy, setGroupBy] = useState<string | null>(savedFilters.groupBy || null);
  const [expandedGroups, setExpandedGroups] = useState<Set<string>>(
    new Set(savedFilters.expandedGroups || [])
  );

  // Filter states
  const [filterSku, setFilterSku] = useState<string>(savedFilters.filterSku || "");
  const [filterLength, setFilterLength] = useState<string>(savedFilters.filterLength || "");
  const [filterWidth, setFilterWidth] = useState<string>(savedFilters.filterWidth || "");
  const [filterHeight, setFilterHeight] = useState<string>(savedFilters.filterHeight || "");
  const [filterColor, setFilterColor] = useState<string>(savedFilters.filterColor || "");
  const [filterMaterial, setFilterMaterial] = useState<string>(savedFilters.filterMaterial || "");
  const [filterType, setFilterType] = useState<string>(savedFilters.filterType || "");
  const [filterGroup, setFilterGroup] = useState<string>(savedFilters.filterGroup || "");
  const [filterDoors, setFilterDoors] = useState<string>(savedFilters.filterDoors || "");
  const [filterLegs, setFilterLegs] = useState<string>(savedFilters.filterLegs || "");
  const [filterBomStatus, setFilterBomStatus] = useState<string>(savedFilters.filterBomStatus || "all");
  const [filterLinkedStatus, setFilterLinkedStatus] = useState<string>(savedFilters.filterLinkedStatus || "all");

  // Save filters to localStorage whenever they change
  useEffect(() => {
    const filters = {
      searchQuery,
      filterSku,
      filterLength,
      filterWidth,
      filterHeight,
      filterColor,
      filterMaterial,
      filterType,
      filterGroup,
      filterDoors,
      filterLegs,
      filterBomStatus,
      filterLinkedStatus,
      page,
      limit,
      sortBy,
      sortOrder,
      compactMode,
      groupBy,
      expandedGroups: Array.from(expandedGroups),
    };
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(filters));
    } catch (error) {
      console.error('Error saving filters to localStorage:', error);
    }
  }, [searchQuery, filterSku, filterLength, filterWidth, filterHeight, filterColor, filterMaterial, filterType, filterGroup, filterDoors, filterLegs, filterBomStatus, filterLinkedStatus, page, limit, sortBy, sortOrder, compactMode, groupBy, expandedGroups]);

  // Helper functions for SavedFiltersManager
  const getCurrentFilters = () => ({
    searchQuery,
    filterSku,
    filterLength,
    filterWidth,
    filterHeight,
    filterColor,
    filterMaterial,
    filterType,
    filterGroup,
    filterDoors,
    filterLegs,
    filterBomStatus,
    filterLinkedStatus,
    page,
    limit,
    sortBy,
    sortOrder,
    compactMode,
    groupBy,
  });

  const loadFiltersFromSaved = (filters: Record<string, any>) => {
    if (filters.searchQuery !== undefined) setSearchQuery(filters.searchQuery);
    if (filters.filterSku !== undefined) setFilterSku(filters.filterSku);
    if (filters.filterLength !== undefined) setFilterLength(filters.filterLength);
    if (filters.filterWidth !== undefined) setFilterWidth(filters.filterWidth);
    if (filters.filterHeight !== undefined) setFilterHeight(filters.filterHeight);
    if (filters.filterColor !== undefined) setFilterColor(filters.filterColor);
    if (filters.filterMaterial !== undefined) setFilterMaterial(filters.filterMaterial);
    if (filters.filterType !== undefined) setFilterType(filters.filterType);
    if (filters.filterGroup !== undefined) setFilterGroup(filters.filterGroup);
    if (filters.filterDoors !== undefined) setFilterDoors(filters.filterDoors);
    if (filters.filterLegs !== undefined) setFilterLegs(filters.filterLegs);
    if (filters.filterBomStatus !== undefined) setFilterBomStatus(filters.filterBomStatus);
    if (filters.filterLinkedStatus !== undefined) setFilterLinkedStatus(filters.filterLinkedStatus);
    if (filters.page !== undefined) setPage(filters.page);
    if (filters.limit !== undefined) setLimit(filters.limit);
    if (filters.sortBy !== undefined) setSortBy(filters.sortBy);
    if (filters.sortOrder !== undefined) setSortOrder(filters.sortOrder);
    if (filters.compactMode !== undefined) setCompactMode(filters.compactMode);
    if (filters.groupBy !== undefined) setGroupBy(filters.groupBy);
  };

  // Inline editing state
  const [editingCell, setEditingCell] = useState<{productId: number, field: string} | null>(null);
  const [editValue, setEditValue] = useState<string>("");

  // Helper function to format mm to cm
  const formatCm = (mm: number | null) => {
    if (!mm) return null;
    const cm = mm / 10;
    return cm % 1 === 0 ? cm.toFixed(0) : cm.toFixed(1);
  };

  // Bulk edit form state
  const [bulkLength, setBulkLength] = useState("");
  const [bulkWidth, setBulkWidth] = useState("");
  const [bulkHeight, setBulkHeight] = useState("");
  const [bulkColor, setBulkColor] = useState("");
  const [bulkMaterial, setBulkMaterial] = useState("");
  const [bulkPrice, setBulkPrice] = useState("");
  const [bulkProductType, setBulkProductType] = useState("");
  const [bulkProductGroup, setBulkProductGroup] = useState("");
  const [bulkDoors, setBulkDoors] = useState("");
  const [bulkLegs, setBulkLegs] = useState("");

  // Build query parameters for API (grouping and filters are handled client-side only)
  const queryParams = useMemo(() => ({
    page: page.toString(),
    limit: limit.toString(),
    sortBy,
    sortOrder,
  }), [page, limit, sortBy, sortOrder]);
  
  const queryKey = useMemo(() => ['/api/catalog-products', queryParams], [queryParams]);
  
  // Fetch Odoo URL configuration
  const { data: odooUrlData } = useQuery<{ url: string | null; userUrl: string | null }>({
    queryKey: ['/api/odoo/url'],
  });
  
  const { data: productsData, isLoading } = useQuery<{
    products: CatalogProduct[];
    pagination: {
      page: number;
      limit: number;
      total: number;
      totalPages: number;
    };
  }>({
    queryKey,
  });

  const products = productsData?.products || [];
  const pagination = productsData?.pagination;

  const { data: colors } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });
  const { data: materials } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=material"],
  });
  const { data: dimensionLengths } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_length"],
  });
  const { data: dimensionWidths } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_width"],
  });
  const { data: dimensionHeights } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_height"],
  });
  const { data: productTypes } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_type"],
  });
  const { data: productGroups } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_group"],
  });
  const { data: doorsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=door"],
  });
  const { data: legsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=leg"],
  });

  // Auto-expand groups when grouping is first enabled or changed
  useEffect(() => {
    if (groupBy && products.length > 0) {
      const groupedData = getGroupedProducts();
      if (groupedData && expandedGroups.size === 0) {
        // Auto-expand all groups on first load or when grouping is enabled
        setExpandedGroups(new Set(groupedData.map(({ key }) => key)));
      }
    }
  }, [groupBy, products.length]);

  // Parse search query into individual search phrases
  const getSearchPhrases = () => {
    if (!searchQuery) return [];
    // Split by both semicolons and commas, trim and remove empty
    const phrases: string[] = [];
    searchQuery.split(/[;,]/).forEach((phrase: string) => {
      const trimmed = phrase.trim();
      if (trimmed) phrases.push(trimmed);
    });
    return phrases;
  };

  const searchPhrases = getSearchPhrases();

  // Remove a specific search phrase
  const removeSearchPhrase = (phraseToRemove: string) => {
    const updatedPhrases = searchPhrases.filter(p => p !== phraseToRemove);
    setSearchQuery(updatedPhrases.join('; '));
  };

  // Smart search - split by semicolon (AND) and comma (OR)
  const filteredProducts = products.filter(product => {
    // SKU filter (exact partial match)
    if (filterSku && !product.sku?.toLowerCase().includes(filterSku.toLowerCase())) {
      return false;
    }
    
    // Text search - supports AND (;) and OR (,)
    if (searchQuery) {
      const terms = searchQuery.split(';').map((t: string) => t.trim()).filter(Boolean);
      const matches = terms.every((term: string) => {
        const orTerms = term.split(',').map((t: string) => t.trim().toLowerCase()).filter(Boolean);
        return orTerms.some((orTerm: string) => {
          return (
            product.id?.toString().includes(orTerm) ||
            product.sku?.toLowerCase().includes(orTerm) ||
            product.title?.toLowerCase().includes(orTerm) ||
            product.color?.toLowerCase().includes(orTerm) ||
            product.material?.toLowerCase().includes(orTerm)
          );
        });
      });
      if (!matches) return false;
    }

    // Dimension filters - convert both to numbers for comparison (DB returns decimals as strings)
    if (filterLength && Number(product.length) !== Number(filterLength)) return false;
    if (filterWidth && Number(product.width) !== Number(filterWidth)) return false;
    if (filterHeight && Number(product.height) !== Number(filterHeight)) return false;

    // Attribute filters
    if (filterColor && product.color !== filterColor) return false;
    if (filterMaterial && product.material !== filterMaterial) return false;
    if (filterType && product.productType !== filterType) return false;
    if (filterGroup && product.productGroup !== filterGroup) return false;
    if (filterDoors && product.doors !== filterDoors) return false;
    if (filterLegs && product.legs !== filterLegs) return false;
    
    // BOM status filter
    if (filterBomStatus === "withBom" && (!product.bomCount || product.bomCount === 0)) return false;
    if (filterBomStatus === "withoutBom" && product.bomCount && product.bomCount > 0) return false;

    // Linked status filter
    if (filterLinkedStatus === "linked" && (!product.marketplaceLinksCount || product.marketplaceLinksCount === 0)) return false;
    if (filterLinkedStatus === "unlinked" && product.marketplaceLinksCount && product.marketplaceLinksCount > 0) return false;

    return true;
  });

  const activeFiltersCount = [
    ...searchPhrases,
    filterSku,
    filterLength,
    filterWidth,
    filterHeight,
    filterColor,
    filterMaterial,
    filterType,
    filterGroup,
    filterDoors,
    filterLegs,
    filterBomStatus !== "all" ? filterBomStatus : null,
    filterLinkedStatus !== "all" ? filterLinkedStatus : null,
  ].filter(Boolean).length;

  const clearAllFilters = () => {
    setFilterSku("");
    setFilterLength("");
    setFilterWidth("");
    setFilterHeight("");
    setFilterColor("");
    setFilterMaterial("");
    setFilterType("");
    setFilterGroup("");
    setFilterDoors("");
    setFilterLegs("");
    setFilterBomStatus("all");
    setFilterLinkedStatus("all");
    setSearchQuery("");
  };

  // Bulk update mutation
  const bulkUpdateMutation = useMutation({
    mutationFn: async (updates: any) => {
      return apiRequest("POST", "/api/catalog-products/bulk-update", {
        productIds: selectedProducts,
        updates,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.startsWith('/api/catalog-products');
        }
      });
      toast({
        title: "Produkty zaktualizowane",
        description: `Pomyślnie zaktualizowano ${selectedProducts.length} produktów.`,
      });
      setSelectedProducts([]);
      setIsBulkEditOpen(false);
      resetBulkEditForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować produktów.",
        variant: "destructive",
      });
    },
  });

  // Bulk delete mutation
  const bulkDeleteMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", "/api/catalog-products/bulk-delete", {
        productIds: selectedProducts,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.startsWith('/api/catalog-products');
        }
      });
      toast({
        title: "Produkty usunięte",
        description: `Pomyślnie usunięto ${selectedProducts.length} produktów.`,
      });
      setSelectedProducts([]);
      setIsBulkDeleteOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć produktów.",
        variant: "destructive",
      });
    },
  });

  // Bulk BOM generation
  const handleBulkBomGeneration = async () => {
    if (selectedProducts.length === 0) {
      toast({
        title: "Brak zaznaczonych produktów",
        description: "Zaznacz produkty do generowania BOM",
        variant: "destructive",
      });
      return;
    }

    setIsGeneratingBom(true);
    
    try {
      toast({
        title: "Generowanie BOM",
        description: `Rozpoczęto generowanie BOM dla ${selectedProducts.length} produktów...`,
      });

      const res = await apiRequest("POST", "/api/catalog-products/bulk-bom", {
        productIds: selectedProducts,
      });
      const response = await res.json() as { successful: number; failed: number; errors: string[] };

      // Immediately invalidate cache - no waiting!
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0] as string;
          return key?.startsWith('/api/catalog-products');
        }
      });

      toast({
        title: "Generowanie BOM zakończone",
        description: `Wygenerowano BOM dla ${response.successful} z ${selectedProducts.length} produktów`,
      });
      
      // Clear selection
      setSelectedProducts([]);
    } catch (error: any) {
      toast({
        title: "Błąd generowania BOM",
        description: error.message || "Nie udało się wygenerować BOM",
        variant: "destructive",
      });
    } finally {
      setIsGeneratingBom(false);
    }
  };

  // Bulk BOM deletion
  const [isDeletingBom, setIsDeletingBom] = useState(false);
  
  const handleBulkBomDeletion = async () => {
    if (selectedProducts.length === 0) {
      toast({
        title: "Brak zaznaczonych produktów",
        description: "Zaznacz produkty do usunięcia BOM",
        variant: "destructive",
      });
      return;
    }

    setIsDeletingBom(true);
    
    try {
      toast({
        title: "Usuwanie BOM",
        description: `Rozpoczęto usuwanie BOM dla ${selectedProducts.length} produktów...`,
      });

      const res = await apiRequest("DELETE", "/api/catalog-products/bulk-bom", {
        productIds: selectedProducts,
      });
      const response = await res.json() as { successful: number; failed: number; errors: string[] };

      // Immediately invalidate cache - no waiting!
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0] as string;
          return key?.startsWith('/api/catalog-products');
        }
      });

      toast({
        title: "Usuwanie BOM zakończone",
        description: `Usunięto BOM dla ${response.successful} z ${selectedProducts.length} produktów`,
      });
      
      // Clear selection
      setSelectedProducts([]);
    } catch (error: any) {
      toast({
        title: "Błąd usuwania BOM",
        description: error.message || "Nie udało się usunąć BOM",
        variant: "destructive",
      });
    } finally {
      setIsDeletingBom(false);
    }
  };

  // Bulk BOM export to Odoo
  const handleBulkBomExport = async () => {
    if (selectedProducts.length === 0) {
      toast({
        title: "Brak zaznaczonych produktów",
        description: "Zaznacz produkty do eksportu BOM do Odoo",
        variant: "destructive",
      });
      return;
    }

    setIsBulkBomExporting(true);
    
    try {
      toast({
        title: "Eksport BOM do Odoo",
        description: `Rozpoczęto eksport BOM dla ${selectedProducts.length} produktów...`,
      });

      const res = await apiRequest("POST", "/api/catalog-products/bulk-bom-export", {
        productIds: selectedProducts,
      });
      const response = await res.json() as { successful: number; failed: number; errors: string[] };

      // Immediately invalidate cache - no waiting!
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0] as string;
          return key?.startsWith('/api/catalog-products');
        }
      });

      toast({
        title: "Eksport BOM zakończony",
        description: `Wyeksportowano BOM dla ${response.successful} z ${selectedProducts.length} produktów${response.failed > 0 ? `. Błędy: ${response.failed}` : ''}`,
        variant: response.failed > 0 ? "destructive" : "default",
      });
      
      // Clear selection
      setSelectedProducts([]);
    } catch (error: any) {
      toast({
        title: "Błąd eksportu BOM",
        description: error.message || "Nie udało się wyeksportować BOM do Odoo",
        variant: "destructive",
      });
    } finally {
      setIsBulkBomExporting(false);
    }
  };

  // Bulk Odoo export
  const handleBulkOdooExport = async () => {
    if (selectedProducts.length === 0) {
      toast({
        title: "Brak zaznaczonych produktów",
        description: "Zaznacz produkty do eksportu do Odoo",
        variant: "destructive",
      });
      return;
    }

    setIsBulkExporting(true);
    
    let successCount = 0;
    let failCount = 0;
    const errors: string[] = [];
    const exportedProducts = new Map<number, number>(); // productId -> odooProductId

    toast({
      title: "Eksport do Odoo",
      description: `Rozpoczęto eksport ${selectedProducts.length} produktów...`,
    });

    // Export products in parallel
    const exportPromises = selectedProducts.map(async (productId) => {
      setExportingProducts(prev => new Set(prev).add(productId));
      
      try {
        const response = await fetch(`/api/catalog-products/${productId}/export-to-odoo`, {
          method: 'POST',
          credentials: 'include',
        });
        
        const data = await response.json();
        
        if (response.ok) {
          successCount++;
          exportedProducts.set(productId, data.odoo_product_id);
        } else {
          failCount++;
          errors.push(`Produkt ID ${productId}: ${data.error}`);
        }
      } catch (error: any) {
        failCount++;
        errors.push(`Produkt ID ${productId}: ${error.message}`);
      } finally {
        setExportingProducts(prev => {
          const newSet = new Set(prev);
          newSet.delete(productId);
          return newSet;
        });
      }
    });

    await Promise.all(exportPromises);

    // Invalidate and refetch - instant UI update!
    if (exportedProducts.size > 0) {
      await queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0] as string;
          return typeof key === 'string' && key.startsWith('/api/catalog-products');
        }
      });
    }

    toast({
      title: "Eksport zakończony",
      description: `Wyeksportowano ${successCount} z ${selectedProducts.length} produktów${failCount > 0 ? `. Błędy: ${failCount}` : ''}`,
      variant: failCount > 0 ? "destructive" : "default",
    });

    setSelectedProducts([]);
    setIsBulkExporting(false);
  };

  // Single product export handler
  const handleProductExport = async (productId: number) => {
    setExportingProducts(prev => new Set(prev).add(productId));
    
    try {
      const response = await fetch(`/api/catalog-products/${productId}/export-to-odoo`, {
        method: 'POST',
        credentials: 'include',
      });
      const data = await response.json();
      
      if (!response.ok) {
        toast({
          title: "Błąd eksportu do Odoo",
          description: `${data.error}${data.details ? `\n\n${data.details}` : ''}`,
          variant: "destructive",
        });
      } else {
        // Invalidate and refetch - instant UI update!
        await queryClient.invalidateQueries({ 
          predicate: (query) => {
            const key = query.queryKey[0] as string;
            return typeof key === 'string' && key.startsWith('/api/catalog-products');
          }
        });
        
        toast({
          title: "Eksport do Odoo",
          description: data.message,
        });
      }
    } catch (error: any) {
      toast({
        title: "Błąd eksportu",
        description: error.message,
        variant: "destructive",
      });
    } finally {
      // Remove spinner - button color already changed!
      setExportingProducts(prev => {
        const newSet = new Set(prev);
        newSet.delete(productId);
        return newSet;
      });
    }
  };

  // Batch export functions
  const handleBatchExport = async (format: 'csv' | 'pdf') => {
    if (selectedProducts.length === 0) {
      toast({
        title: "Brak zaznaczonych produktów",
        description: "Zaznacz produkty do eksportu",
        variant: "destructive",
      });
      return;
    }

    try {
      const response = await fetch('/api/catalog-products/batch-export', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          productIds: selectedProducts,
          format,
        }),
      });

      if (!response.ok) {
        throw new Error('Export failed');
      }

      // Get filename from Content-Disposition header or use default
      const contentDisposition = response.headers.get('Content-Disposition');
      let filename = `products-${new Date().toISOString().split('T')[0]}.${format}`;
      if (contentDisposition) {
        const filenameMatch = contentDisposition.match(/filename="?(.+)"?/i);
        if (filenameMatch) {
          filename = filenameMatch[1];
        }
      }

      // Download file
      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);

      toast({
        title: "Eksport zakończony",
        description: `Plik ${filename} został pobrany (${selectedProducts.length} produktów)`,
      });
    } catch (error) {
      console.error('Batch export error:', error);
      toast({
        title: "Błąd eksportu",
        description: "Nie udało się wyeksportować produktów",
        variant: "destructive",
      });
    }
  };

  // Inline edit mutation
  const inlineUpdateMutation = useMutation({
    mutationFn: async ({ productId, field, value }: { productId: number, field: string, value: any }) => {
      const updates: any = {};
      
      if (field === 'sku') updates.sku = value;
      if (field === 'basePrice') updates.basePrice = value;
      if (field === 'color') updates.color = value;
      if (field === 'isActive') updates.isActive = value;
      
      return apiRequest("PATCH", `/api/catalog-products/${productId}`, updates);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.startsWith('/api/catalog-products');
        }
      });
      setEditingCell(null);
      setEditValue("");
      toast({
        title: "Zapisano",
        description: "Produkt został zaktualizowany.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd zapisu",
        description: error.message || "Nie udało się zapisać zmiany.",
        variant: "destructive",
      });
    },
  });

  const toggleProduct = (productId: number) => {
    setSelectedProducts(prev =>
      prev.includes(productId)
        ? prev.filter(id => id !== productId)
        : [...prev, productId]
    );
  };

  const toggleAll = () => {
    if (selectedProducts.length === filteredProducts.length) {
      setSelectedProducts([]);
    } else {
      setSelectedProducts(filteredProducts.map(p => p.id));
    }
  };

  // Inline editing helpers
  const startEditing = (productId: number, field: string, currentValue: any) => {
    setEditingCell({ productId, field });
    setEditValue(currentValue?.toString() || "");
  };

  const cancelEditing = () => {
    setEditingCell(null);
    setEditValue("");
  };

  const saveEdit = (productId: number, field: string) => {
    let value: any = editValue;
    
    if (field === 'basePrice') {
      value = editValue ? parseFloat(editValue) : null;
    }
    
    inlineUpdateMutation.mutate({ productId, field, value });
  };

  const copySkuAirtableToSku = (productId: number, skuAirtable: string) => {
    inlineUpdateMutation.mutate({ 
      productId, 
      field: 'sku', 
      value: skuAirtable 
    });
    
    toast({
      title: "SKU skopiowane",
      description: `Wartość "${skuAirtable}" została skopiowana do pola SKU`,
    });
  };

  const toggleActive = (productId: number, currentValue: boolean) => {
    inlineUpdateMutation.mutate({ 
      productId, 
      field: 'isActive', 
      value: !currentValue 
    });
  };

  const resetBulkEditForm = () => {
    setBulkLength("");
    setBulkWidth("");
    setBulkHeight("");
    setBulkColor("");
    setBulkMaterial("");
    setBulkPrice("");
    setBulkProductType("");
    setBulkProductGroup("");
    setBulkDoors("");
    setBulkLegs("");
  };

  const handleBulkUpdate = () => {
    const updates: any = {};
    if (bulkLength) updates.length = parseFloat(bulkLength);
    if (bulkWidth) updates.width = parseFloat(bulkWidth);
    if (bulkHeight) updates.height = parseFloat(bulkHeight);
    if (bulkColor) updates.color = bulkColor;
    if (bulkMaterial) updates.material = bulkMaterial;
    if (bulkPrice) updates.basePrice = parseFloat(bulkPrice);
    if (bulkProductType) updates.productType = bulkProductType;
    if (bulkProductGroup) updates.productGroup = bulkProductGroup;
    if (bulkDoors) updates.doors = bulkDoors;
    if (bulkLegs) updates.legs = bulkLegs;

    if (Object.keys(updates).length === 0) {
      toast({
        title: "Brak zmian",
        description: "Wybierz przynajmniej jedno pole do aktualizacji.",
        variant: "destructive",
      });
      return;
    }

    bulkUpdateMutation.mutate(updates);
  };

  // Sorting function
  const handleSort = (column: string) => {
    if (sortBy === column) {
      setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
    } else {
      setSortBy(column);
      setSortOrder('DESC');
    }
    setPage(1); // Reset to first page on sort change
  };

  // Grouping functions
  const getGroupKey = (product: CatalogProduct, field: string): string => {
    const value = product[field as keyof CatalogProduct];
    if (value === null || value === undefined) return '(puste)';
    
    // For dimensions, format as cm
    if (field === 'length' || field === 'width' || field === 'height') {
      const cm = formatCm(value as number);
      return cm ? `${cm} cm` : '(puste)';
    }
    
    // For dictionary fields, find readable name
    if (field === 'color' && colors) {
      const color = colors.find(c => c.name === value || c.code === value);
      return color?.name || String(value);
    }
    if (field === 'material' && materials) {
      const material = materials.find(m => m.name === value || m.code === value);
      return material?.name || String(value);
    }
    if (field === 'productType' && productTypes) {
      const type = productTypes.find(t => t.name === value || t.code === value);
      return type?.name || String(value);
    }
    if (field === 'productGroup' && productGroups) {
      const group = productGroups.find(g => g.name === value || g.code === value);
      return group?.name || String(value);
    }
    
    return String(value);
  };

  const getGroupedProducts = () => {
    if (!groupBy) return null;

    const groups = new Map<string, CatalogProduct[]>();
    
    filteredProducts.forEach(product => {
      const key = getGroupKey(product, groupBy);
      if (!groups.has(key)) {
        groups.set(key, []);
      }
      groups.get(key)!.push(product);
    });

    // Sort groups by key
    return Array.from(groups.entries())
      .sort(([a], [b]) => a.localeCompare(b, 'pl'))
      .map(([key, products]) => ({ key, products }));
  };

  const toggleGroup = (groupKey: string) => {
    setExpandedGroups(prev => {
      const newSet = new Set(prev);
      if (newSet.has(groupKey)) {
        newSet.delete(groupKey);
      } else {
        newSet.add(groupKey);
      }
      return newSet;
    });
  };

  const toggleAllGroups = () => {
    const groupedData = getGroupedProducts();
    if (!groupedData) return;

    // If all groups are expanded, collapse all. Otherwise expand all.
    const allExpanded = groupedData.every(({ key }) => expandedGroups.has(key));
    
    if (allExpanded) {
      setExpandedGroups(new Set());
    } else {
      setExpandedGroups(new Set(groupedData.map(({ key }) => key)));
    }
  };

  // Pagination component
  const PaginationControls = () => {
    if (!pagination) return null;

    return (
      <div className="flex flex-col md:flex-row items-center justify-between gap-2 md:gap-4">
        <div className="text-xs md:text-sm text-muted-foreground order-2 md:order-1">
          <span className="hidden md:inline">
            Wyniki {((page - 1) * limit) + 1} - {Math.min(page * limit, pagination.total)} z {pagination.total}
          </span>
          <span className="md:hidden">
            {((page - 1) * limit) + 1}-{Math.min(page * limit, pagination.total)} / {pagination.total}
          </span>
        </div>

        <div className="flex items-center gap-2 order-1 md:order-2">
          <Select
            value={limit.toString()}
            onValueChange={(value) => {
              setLimit(parseInt(value));
              setPage(1);
            }}
          >
            <SelectTrigger className="w-16 md:w-20 h-8" data-testid="select-page-size">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="25">25</SelectItem>
              <SelectItem value="50">50</SelectItem>
              <SelectItem value="250">250</SelectItem>
            </SelectContent>
          </Select>

          <div className="flex items-center gap-0.5 md:gap-1">
            <Button
              variant="outline"
              size="sm"
              onClick={() => setPage(1)}
              disabled={page === 1}
              data-testid="button-first-page"
              className="h-8 w-8 p-0 hidden md:inline-flex"
            >
              <ChevronsLeft className="h-4 w-4" />
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setPage(page - 1)}
              disabled={page === 1}
              data-testid="button-prev-page"
              className="h-8 w-8 p-0"
            >
              <ChevronLeft className="h-4 w-4" />
            </Button>
            <span className="px-2 md:px-3 text-xs md:text-sm whitespace-nowrap">
              {page} / {pagination.totalPages}
            </span>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setPage(page + 1)}
              disabled={page >= pagination.totalPages}
              data-testid="button-next-page"
              className="h-8 w-8 p-0"
            >
              <ChevronRight className="h-4 w-4" />
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setPage(pagination.totalPages)}
              disabled={page >= pagination.totalPages}
              data-testid="button-last-page"
              className="h-8 w-8 p-0 hidden md:inline-flex"
            >
              <ChevronsRight className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </div>
    );
  };

  return (
    <div className="p-2 sm:p-4 space-y-2">
      <Card>
        <CardHeader className="h-10 px-3 py-0 flex items-center">
          <div className="flex flex-row gap-2 justify-between items-center w-full">
            <div className="flex items-center gap-2 flex-wrap">
              <Package className="h-4 w-4 text-primary" />
              <CardTitle className="text-base" data-testid="title-catalog">Katalog Produktów</CardTitle>
              {!isLoading && (
                <Badge variant="secondary" className="h-5 text-xs">
                  {filteredProducts.length} {filteredProducts.length === 1 ? 'produkt' : 'produktów'}
                </Badge>
              )}
              {/* Search phrases badges */}
              {searchPhrases.map((phrase, index) => (
                <Badge 
                  key={`search-${index}`} 
                  variant="default" 
                  className="gap-1 pr-0.5 h-5 text-xs"
                  data-testid={`badge-search-phrase-${index}`}
                >
                  {phrase}
                  <Button
                    variant="ghost"
                    size="sm"
                    className="h-4 w-4 p-0 hover:bg-primary-foreground/20 ml-0.5"
                    onClick={(e) => {
                      e.stopPropagation();
                      removeSearchPhrase(phrase);
                    }}
                    data-testid={`button-remove-search-${index}`}
                  >
                    <X className="h-2.5 w-2.5" />
                  </Button>
                </Badge>
              ))}
            </div>
            <Button
              onClick={() => setLocation("/catalog-products/new")}
              data-testid="button-add-product"
              size="sm"
              className="h-8 shrink-0"
            >
              <Plus className="h-3.5 w-3.5 mr-1.5" />
              Dodaj Produkt
            </Button>
          </div>
        </CardHeader>
        <CardContent className="space-y-3 pt-3">
          {/* Search Bar */}
          <div className="flex gap-2">
            <div className="flex-1 relative">
              <Search className="absolute left-2 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Szukaj... (; dla AND, , dla OR)"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-8 h-8"
                data-testid="input-search-main"
              />
            </div>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setFiltersExpanded(!filtersExpanded)}
              data-testid="button-toggle-filters"
              className="h-8"
            >
              Filtry {activeFiltersCount > 0 && `(${activeFiltersCount})`}
              {filtersExpanded ? <ChevronUp className="ml-1 h-4 w-4" /> : <ChevronDown className="ml-1 h-4 w-4" />}
            </Button>
          </div>

          {/* Expanded Filters */}
          {filtersExpanded && (
            <div className="grid grid-cols-1 md:grid-cols-4 gap-2 p-3 border rounded-md bg-muted/50">
              <div className="space-y-1">
                <Label className="text-xs">SKU</Label>
                <Input
                  placeholder="Filtruj SKU"
                  value={filterSku}
                  onChange={(e) => setFilterSku(e.target.value)}
                  className="h-8"
                  data-testid="input-filter-sku"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Długość</Label>
                <DictionaryCombobox
                  items={dimensionLengths || []}
                  value={filterLength}
                  onChange={setFilterLength}
                  placeholder="Wybierz długość"
                  testId="select-filter-length"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Szerokość</Label>
                <DictionaryCombobox
                  items={dimensionWidths || []}
                  value={filterWidth}
                  onChange={setFilterWidth}
                  placeholder="Wybierz szerokość"
                  testId="select-filter-width"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Wysokość</Label>
                <DictionaryCombobox
                  items={dimensionHeights || []}
                  value={filterHeight}
                  onChange={setFilterHeight}
                  placeholder="Wybierz wysokość"
                  testId="select-filter-height"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Kolor</Label>
                <DictionaryCombobox
                  items={colors || []}
                  value={filterColor}
                  onChange={setFilterColor}
                  placeholder="Wybierz kolor"
                  testId="select-filter-color"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Materiał</Label>
                <DictionaryCombobox
                  items={materials || []}
                  value={filterMaterial}
                  onChange={setFilterMaterial}
                  placeholder="Wybierz materiał"
                  testId="select-filter-material"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Typ</Label>
                <DictionaryCombobox
                  items={productTypes || []}
                  value={filterType}
                  onChange={setFilterType}
                  placeholder="Wybierz typ"
                  testId="select-filter-type"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Grupa</Label>
                <DictionaryCombobox
                  items={productGroups || []}
                  value={filterGroup}
                  onChange={setFilterGroup}
                  placeholder="Wybierz grupę"
                  testId="select-filter-group"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Drzwi</Label>
                <DictionaryCombobox
                  items={doorsOptions || []}
                  value={filterDoors}
                  onChange={setFilterDoors}
                  placeholder="Wybierz drzwi"
                  testId="select-filter-doors"
                />
              </div>
              <div className="space-y-1">
                <Label className="text-xs">Nogi</Label>
                <DictionaryCombobox
                  items={legsOptions || []}
                  value={filterLegs}
                  onChange={setFilterLegs}
                  placeholder="Wybierz nogi"
                  testId="select-filter-legs"
                />
              </div>
              <div className="flex items-end md:col-span-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={clearAllFilters}
                  className="w-full h-8"
                  data-testid="button-clear-filters"
                >
                  <X className="h-4 w-4 mr-1" />
                  Wyczyść filtry
                </Button>
              </div>
            </div>
          )}

          {/* Grouping and View Options */}
          <div className="flex gap-2 items-center flex-wrap">
            <Label className="text-xs text-muted-foreground">Grupuj po:</Label>
            <Select value={groupBy || "none"} onValueChange={(val) => setGroupBy(val === "none" ? null : val)}>
              <SelectTrigger className="w-32 h-7 text-xs" data-testid="select-group-by">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="none">Brak</SelectItem>
                <SelectItem value="type">Typ</SelectItem>
                <SelectItem value="group">Grupa</SelectItem>
                <SelectItem value="color">Kolor</SelectItem>
                <SelectItem value="material">Materiał</SelectItem>
              </SelectContent>
            </Select>

            <Label className="text-xs text-muted-foreground ml-2">Sortuj:</Label>
            <Select value={sortBy} onValueChange={(val) => setSortBy(val)}>
              <SelectTrigger className="w-32 h-7 text-xs" data-testid="select-sort-by">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="id">ID</SelectItem>
                <SelectItem value="sku">SKU</SelectItem>
                <SelectItem value="title">Nazwa</SelectItem>
                <SelectItem value="length">Długość</SelectItem>
                <SelectItem value="width">Szerokość</SelectItem>
                <SelectItem value="height">Wysokość</SelectItem>
                <SelectItem value="color">Kolor</SelectItem>
                <SelectItem value="price">Cena</SelectItem>
              </SelectContent>
            </Select>
            <Button
              variant="ghost"
              size="sm"
              onClick={() => setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC')}
              className="h-7 px-2"
              data-testid="button-toggle-sort"
            >
              {sortOrder === 'ASC' ? '↑' : '↓'}
            </Button>

            <div className="ml-auto flex items-center gap-2">
              <SavedFiltersManager
                context="catalog_products"
                currentFilters={getCurrentFilters()}
                onFiltersLoad={loadFiltersFromSaved}
              />
            </div>
          </div>

          {/* Bulk Actions Bar */}
          {selectedProducts.length > 0 && (
            <div className="flex items-center gap-2 p-2 bg-muted rounded-md">
              <span className="text-sm font-medium">
                Wybrano: {selectedProducts.length}
              </span>
              <Button
                size="sm"
                variant="outline"
                onClick={() => setIsBulkEditOpen(true)}
                data-testid="button-bulk-edit"
                className="h-7"
              >
                <Edit className="h-3 w-3 mr-1" />
                Edytuj
              </Button>
              <Button
                size="sm"
                variant="destructive"
                onClick={() => setIsBulkDeleteOpen(true)}
                data-testid="button-bulk-delete"
                className="h-7"
              >
                <Trash2 className="h-3 w-3 mr-1" />
                Usuń
              </Button>
              <Button
                size="sm"
                variant="ghost"
                onClick={() => setSelectedProducts([])}
                data-testid="button-clear-selection"
                className="h-7 ml-auto"
              >
                Wyczyść
              </Button>
            </div>
          )}
        </CardContent>
      </Card>

        {selectedProducts.length > 0 && (
          <div className="fixed bottom-0 left-0 right-0 z-50 border-t border-primary bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/80 shadow-lg">
            <div className="container mx-auto px-4 py-3">
              <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-2">
                <div className="flex items-center gap-2">
                  <Check className="h-4 w-4 text-primary" />
                  <span className="font-medium text-sm">
                    Zaznaczono: {selectedProducts.length} {selectedProducts.length === 1 ? 'produkt' : 'produktów'}
                  </span>
                </div>
                <div className="flex items-center gap-2 overflow-x-auto">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setIsBulkEditOpen(true)}
                    data-testid="button-bulk-edit"
                    className="whitespace-nowrap"
                  >
                    <Edit className="h-4 w-4 md:mr-2" />
                    <span className="hidden md:inline">Edytuj zaznaczone</span>
                  </Button>
                  <Button
                    variant="default"
                    size="sm"
                    onClick={() => handleBulkBomGeneration()}
                    disabled={isGeneratingBom}
                    data-testid="button-bulk-bom"
                    className="whitespace-nowrap"
                  >
                    {isGeneratingBom ? (
                      <>
                        <div className="h-4 w-4 md:mr-2 animate-spin rounded-full border-2 border-background border-t-transparent" />
                        <span className="hidden md:inline">Generowanie...</span>
                      </>
                    ) : (
                      <>
                        <FileText className="h-4 w-4 md:mr-2" />
                        <span className="hidden md:inline">Generuj BOM</span>
                      </>
                    )}
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleBulkBomDeletion()}
                    disabled={isDeletingBom}
                    data-testid="button-bulk-bom-delete"
                    className="whitespace-nowrap"
                  >
                    {isDeletingBom ? (
                      <>
                        <div className="h-4 w-4 md:mr-2 animate-spin rounded-full border-2 border-foreground border-t-transparent" />
                        <span className="hidden md:inline">Usuwanie...</span>
                      </>
                    ) : (
                      <>
                        <Trash2 className="h-4 w-4 md:mr-2" />
                        <span className="hidden md:inline">Usuń BOM</span>
                      </>
                    )}
                  </Button>
                  <Button
                    variant="default"
                    size="sm"
                    onClick={() => handleBulkBomExport()}
                    disabled={isBulkBomExporting}
                    data-testid="button-bulk-bom-export-odoo"
                    className="whitespace-nowrap bg-blue-600 hover:bg-blue-700"
                  >
                    {isBulkBomExporting ? (
                      <>
                        <Loader2 className="h-4 w-4 md:mr-2 animate-spin" />
                        <span className="hidden md:inline">Eksport BOM...</span>
                      </>
                    ) : (
                      <>
                        <Upload className="h-4 w-4 md:mr-2" />
                        <span className="hidden md:inline">Eksport BOM do Odoo</span>
                      </>
                    )}
                  </Button>
                  <Button
                    variant="default"
                    size="sm"
                    onClick={() => handleBulkOdooExport()}
                    disabled={isBulkExporting}
                    data-testid="button-bulk-odoo-export"
                    className="whitespace-nowrap bg-green-600 hover:bg-green-700"
                  >
                    {isBulkExporting ? (
                      <>
                        <Loader2 className="h-4 w-4 md:mr-2 animate-spin" />
                        <span className="hidden md:inline">Eksportowanie...</span>
                      </>
                    ) : (
                      <>
                        <Upload className="h-4 w-4 md:mr-2" />
                        <span className="hidden md:inline">Eksport do Odoo</span>
                      </>
                    )}
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleBatchExport('csv')}
                    data-testid="button-batch-export-csv"
                    className="whitespace-nowrap"
                  >
                    <FileSpreadsheet className="h-4 w-4 md:mr-2" />
                    <span className="hidden md:inline">Eksport CSV</span>
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleBatchExport('pdf')}
                    data-testid="button-batch-export-pdf"
                    className="whitespace-nowrap"
                  >
                    <FileText className="h-4 w-4 md:mr-2" />
                    <span className="hidden md:inline">Eksport PDF</span>
                  </Button>
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={() => setIsBulkDeleteOpen(true)}
                    data-testid="button-bulk-delete"
                    className="whitespace-nowrap"
                  >
                    <Trash2 className="h-4 w-4 md:mr-2" />
                    <span className="hidden md:inline">Usuń</span>
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => setSelectedProducts([])}
                    data-testid="button-deselect-all"
                    className="whitespace-nowrap"
                  >
                    Odznacz
                  </Button>
                </div>
              </div>
            </div>
          </div>
        )}

        <Card>
          <CardContent className="pt-6">
            {isLoading ? (
              <div className="space-y-2">
                {[...Array(5)].map((_, i) => (
                  <Skeleton key={i} className="h-16 w-full" />
                ))}
              </div>
            ) : filteredProducts.length === 0 ? (
              <div className="text-center py-12">
                <Package className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                <h3 className="text-lg font-medium mb-2">Brak produktów</h3>
                <p className="text-muted-foreground mb-4">
                  {searchQuery || activeFiltersCount > 0
                    ? "Nie znaleziono produktów pasujących do wyszukiwania lub filtrów"
                    : "Dodaj pierwszy produkt do katalogu"}
                </p>
                {!searchQuery && activeFiltersCount === 0 && (
                  <Button onClick={() => setLocation("/catalog-products/new")}>
                    <Plus className="h-4 w-4 mr-2" />
                    Dodaj Produkt
                  </Button>
                )}
              </div>
            ) : (
              <>
                <div className="mb-3 md:mb-4 space-y-2 md:space-y-0 md:flex md:items-center md:justify-between md:gap-4">
                  {/* Mobile: Stacked controls */}
                  <div className="flex flex-col gap-2 md:hidden">
                    <div className="flex items-center justify-between gap-2">
                      <div className="flex items-center gap-2">
                        <Switch
                          checked={compactMode}
                          onCheckedChange={setCompactMode}
                          data-testid="switch-compact-mode"
                        />
                        <Label className="text-xs cursor-pointer" onClick={() => setCompactMode(!compactMode)}>
                          Kompaktowy
                        </Label>
                      </div>
                      <Select
                        value={groupBy || "none"}
                        onValueChange={(value) => {
                          setGroupBy(value === "none" ? null : value);
                          setExpandedGroups(new Set());
                        }}
                      >
                        <SelectTrigger className="w-32 h-8 text-xs" data-testid="select-group-by">
                          <SelectValue placeholder="Grupuj" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="none">Brak</SelectItem>
                          <SelectItem value="color">Kolor</SelectItem>
                          <SelectItem value="length">Długość</SelectItem>
                          <SelectItem value="width">Szerokość</SelectItem>
                          <SelectItem value="productType">Rodzaj</SelectItem>
                          <SelectItem value="material">Materiał</SelectItem>
                          <SelectItem value="productGroup">Grupa</SelectItem>
                        </SelectContent>
                      </Select>
                      {groupBy && (() => {
                        const groupedData = getGroupedProducts();
                        if (!groupedData || groupedData.length === 0) return null;
                        const allExpanded = groupedData.every(({ key }) => expandedGroups.has(key));
                        return (
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={toggleAllGroups}
                            data-testid="button-toggle-all-groups"
                            className="h-8 px-2"
                          >
                            {allExpanded ? (
                              <ChevronUp className="h-4 w-4" />
                            ) : (
                              <ChevronDown className="h-4 w-4" />
                            )}
                          </Button>
                        );
                      })()}
                    </div>
                    <PaginationControls />
                  </div>

                  {/* Desktop: Horizontal layout */}
                  <div className="hidden md:flex md:items-center md:gap-4">
                    <div className="flex items-center gap-2">
                      <Switch
                        checked={compactMode}
                        onCheckedChange={setCompactMode}
                        data-testid="switch-compact-mode"
                      />
                      <Label className="text-sm cursor-pointer" onClick={() => setCompactMode(!compactMode)}>
                        Widok kompaktowy
                      </Label>
                    </div>
                    <div className="flex items-center gap-2">
                      <Label className="text-sm whitespace-nowrap">
                        Grupuj według:
                      </Label>
                      <Select
                        value={groupBy || "none"}
                        onValueChange={(value) => {
                          setGroupBy(value === "none" ? null : value);
                          setExpandedGroups(new Set());
                        }}
                      >
                        <SelectTrigger className="w-40 h-8" data-testid="select-group-by">
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="none">Brak</SelectItem>
                          <SelectItem value="color">Kolor</SelectItem>
                          <SelectItem value="length">Długość</SelectItem>
                          <SelectItem value="width">Szerokość</SelectItem>
                          <SelectItem value="productType">Rodzaj</SelectItem>
                          <SelectItem value="material">Materiał</SelectItem>
                          <SelectItem value="productGroup">Grupa</SelectItem>
                        </SelectContent>
                      </Select>
                      {groupBy && (() => {
                        const groupedData = getGroupedProducts();
                        if (!groupedData || groupedData.length === 0) return null;
                        const allExpanded = groupedData.every(({ key }) => expandedGroups.has(key));
                        return (
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={toggleAllGroups}
                            data-testid="button-toggle-all-groups"
                          >
                            {allExpanded ? (
                              <>
                                <ChevronUp className="h-4 w-4 mr-2" />
                                Zwiń wszystko
                              </>
                            ) : (
                              <>
                                <ChevronDown className="h-4 w-4 mr-2" />
                                Rozwiń wszystko
                              </>
                            )}
                          </Button>
                        );
                      })()}
                    </div>
                  </div>
                  <div className="hidden md:block flex-1 min-w-max">
                    <PaginationControls />
                  </div>
                </div>

                {/* Mobile Card View - visible on screens < md */}
                <div className="md:hidden space-y-2">
                  {groupBy ? (
                    // Grouped mobile view
                    getGroupedProducts()?.map(({ key: groupKey, products: groupProducts }) => {
                      const isExpanded = expandedGroups.has(groupKey);
                      return (
                        <div key={`mobile-group-${groupKey}`} className="space-y-2">
                          {/* Group Header */}
                          <button
                            onClick={() => toggleGroup(groupKey)}
                            className="w-full flex items-center gap-2 bg-muted/30 rounded-md px-3 py-2 text-left hover:bg-muted/50 transition-colors"
                            data-testid={`button-mobile-group-${groupKey}`}
                          >
                            {isExpanded ? (
                              <ChevronUp className="h-4 w-4 flex-shrink-0" />
                            ) : (
                              <ChevronDown className="h-4 w-4 flex-shrink-0" />
                            )}
                            <span className="font-semibold text-sm flex-1 min-w-0 truncate">
                              {groupKey}
                            </span>
                            <span className="text-xs text-muted-foreground flex-shrink-0">
                              ({groupProducts.length})
                            </span>
                          </button>
                          
                          {/* Group Products */}
                          {isExpanded && groupProducts.map((product) => (
                            <Card key={product.id} className="hover-elevate" data-testid={`card-product-${product.id}`}>
                              <CardContent className="p-3">
                                <div className="flex gap-2">
                                  {/* Checkbox */}
                                  <div className="flex-shrink-0 pt-1">
                                    <Checkbox
                                      checked={selectedProducts.includes(product.id)}
                                      onCheckedChange={() => toggleProduct(product.id)}
                                      data-testid={`checkbox-mobile-product-${product.id}`}
                                    />
                                  </div>
                                  
                                  {/* Product Image */}
                                  <div 
                                    className="flex-shrink-0 cursor-pointer"
                                    onClick={() => setLocation(`/catalog-products/${product.id}`)}
                                  >
                                    {product.primaryImageUrl ? (
                                      <img 
                                        src={product.primaryImageUrl} 
                                        alt={product.title}
                                        className="w-20 h-20 object-cover"
                                        data-testid={`img-mobile-product-${product.id}`}
                                      />
                                    ) : (
                                      <div className="w-20 h-20 bg-muted flex items-center justify-center">
                                        <ImageIcon className="h-8 w-8 text-muted-foreground" />
                                      </div>
                                    )}
                                  </div>
                                  
                                  {/* Product Details */}
                                  <div 
                                    className="flex-1 min-w-0 cursor-pointer"
                                    onClick={() => setLocation(`/catalog-products/${product.id}`)}
                                  >
                                    <div className="font-medium text-sm mb-0.5 line-clamp-2">{product.title || "-"}</div>
                                    <div className="text-xs text-muted-foreground font-mono mb-1.5">{product.sku || "-"}</div>
                                    
                                    {/* Badges */}
                                    <div className="flex flex-wrap gap-1 mb-1.5">
                                      {product.productType && (() => {
                                        const typeDict = productTypes?.find(t => t.code === product.productType);
                                        const bgColor = typeDict?.color || '#6366F1';
                                        const textColor = getTextColorForBackground(bgColor);
                                        return (
                                          <Badge 
                                            variant="outline" 
                                            className="text-xs h-5 px-1.5 border-transparent" 
                                            style={{ backgroundColor: bgColor, color: textColor }}
                                          >
                                            {product.productType}
                                          </Badge>
                                        );
                                      })()}
                                      {product.color && (() => {
                                        const colorDict = colors?.find(c => c.code === product.color);
                                        const bgColor = colorDict?.color || '#94A3B8';
                                        const textColor = getTextColorForBackground(bgColor);
                                        const hasBorder = needsVisibleBorder(bgColor);
                                        return (
                                          <Badge 
                                            variant="outline" 
                                            className={`text-xs h-5 px-1.5 ${hasBorder ? '' : 'border-transparent'}`}
                                            style={{ backgroundColor: bgColor, color: textColor }}
                                          >
                                            {product.color}
                                          </Badge>
                                        );
                                      })()}
                                      {product.material && (() => {
                                        const materialDict = materials?.find(m => m.code === product.material);
                                        if (materialDict?.color) {
                                          const bgColor = materialDict.color;
                                          const textColor = getTextColorForBackground(bgColor);
                                          const hasBorder = needsVisibleBorder(bgColor);
                                          return (
                                            <Badge 
                                              variant="outline" 
                                              className={`text-xs h-5 px-1.5 ${hasBorder ? '' : 'border-transparent'}`}
                                              style={{ backgroundColor: bgColor, color: textColor }}
                                            >
                                              {product.material}
                                            </Badge>
                                          );
                                        }
                                        return <Badge variant="outline" className="text-xs h-5 px-1.5 font-normal text-muted-foreground">{product.material}</Badge>;
                                      })()}
                                      {product.colorOptions && product.colorOptions.length > 0 && product.colorOptions.map((option, idx) => (
                                        <Badge 
                                          key={idx}
                                          variant="secondary" 
                                          className="text-xs h-5 px-1.5"
                                        >
                                          {option}
                                        </Badge>
                                      ))}
                                    </div>
                                    
                                    {/* Dimensions */}
                                    {(product.length || product.width || product.height) && (
                                      <div className="text-xs text-muted-foreground mb-1.5">
                                        {product.length && `${formatCm(product.length)}cm`}
                                        {product.length && product.width && ' × '}
                                        {product.width && `${formatCm(product.width)}cm`}
                                        {(product.length || product.width) && product.height && ' × '}
                                        {product.height && `${formatCm(product.height)}cm`}
                                      </div>
                                    )}
                                    
                                    {/* Price & Stats */}
                                    <div className="flex items-center justify-between">
                                      <div className="font-medium text-sm">
                                        {product.basePrice 
                                          ? `${Number(product.basePrice).toFixed(2)} ${product.currency}`
                                          : "-"}
                                      </div>
                                      <div className="flex items-center gap-1.5">
                                        <Badge variant="outline" className="gap-0.5 text-xs px-1 py-0">
                                          <ImageIcon className="h-2.5 w-2.5" />
                                          {product.imagesCount}
                                        </Badge>
                                        <div 
                                          onClick={(e) => {
                                            e.stopPropagation();
                                            toggleActive(product.id, product.isActive);
                                          }}
                                        >
                                          <Switch
                                            checked={product.isActive}
                                            data-testid={`switch-mobile-active-${product.id}`}
                                          />
                                        </div>
                                      </div>
                                    </div>
                                  </div>
                                </div>
                              </CardContent>
                            </Card>
                          ))}
                        </div>
                      );
                    })
                  ) : (
                    // Ungrouped mobile view
                    filteredProducts.map((product) => (
                      <Card key={product.id} className="hover-elevate" data-testid={`card-product-${product.id}`}>
                        <CardContent className="p-3">
                          <div className="flex gap-2">
                            {/* Checkbox */}
                            <div className="flex-shrink-0 pt-1">
                              <Checkbox
                                checked={selectedProducts.includes(product.id)}
                                onCheckedChange={() => toggleProduct(product.id)}
                                data-testid={`checkbox-mobile-product-${product.id}`}
                              />
                            </div>
                            
                            {/* Product Image */}
                            <div 
                              className="flex-shrink-0 cursor-pointer"
                              onClick={() => setLocation(`/catalog-products/${product.id}`)}
                            >
                              {product.primaryImageUrl ? (
                                <img 
                                  src={product.primaryImageUrl} 
                                  alt={product.title}
                                  className="w-20 h-20 object-cover"
                                  data-testid={`img-mobile-product-${product.id}`}
                                />
                              ) : (
                                <div className="w-20 h-20 bg-muted flex items-center justify-center">
                                  <ImageIcon className="h-8 w-8 text-muted-foreground" />
                                </div>
                              )}
                            </div>
                            
                            {/* Product Details */}
                            <div 
                              className="flex-1 min-w-0 cursor-pointer"
                              onClick={() => setLocation(`/catalog-products/${product.id}`)}
                            >
                              <div className="font-medium text-sm mb-0.5 line-clamp-2">{product.title || "-"}</div>
                              <div className="text-xs text-muted-foreground font-mono mb-1.5">{product.sku || "-"}</div>
                              
                              {/* Badges */}
                              <div className="flex flex-wrap gap-1 mb-1.5">
                                {product.productType && (() => {
                                  const typeDict = productTypes?.find(t => t.code === product.productType);
                                  const bgColor = typeDict?.color || '#6366F1';
                                  const textColor = getTextColorForBackground(bgColor);
                                  return (
                                    <Badge 
                                      variant="outline" 
                                      className="text-xs h-5 px-1.5 border-transparent" 
                                      style={{ backgroundColor: bgColor, color: textColor }}
                                    >
                                      {product.productType}
                                    </Badge>
                                  );
                                })()}
                                {product.color && (() => {
                                  const colorDict = colors?.find(c => c.code === product.color);
                                  const bgColor = colorDict?.color || '#94A3B8';
                                  const textColor = getTextColorForBackground(bgColor);
                                  return (
                                    <Badge 
                                      variant="outline" 
                                      className="text-xs h-5 px-1.5 border-transparent" 
                                      style={{ backgroundColor: bgColor, color: textColor }}
                                    >
                                      {product.color}
                                    </Badge>
                                  );
                                })()}
                                {product.material && (() => {
                                  const materialDict = materials?.find(m => m.code === product.material);
                                  if (materialDict?.color) {
                                    const bgColor = materialDict.color;
                                    const textColor = getTextColorForBackground(bgColor);
                                    return (
                                      <Badge 
                                        variant="outline" 
                                        className="text-xs h-5 px-1.5 border-transparent" 
                                        style={{ backgroundColor: bgColor, color: textColor }}
                                      >
                                        {product.material}
                                      </Badge>
                                    );
                                  }
                                  return <Badge variant="outline" className="text-xs px-1.5 py-0">{product.material}</Badge>;
                                })()}
                                {product.colorOptions && product.colorOptions.length > 0 && product.colorOptions.map((option, idx) => (
                                  <Badge 
                                    key={idx}
                                    variant="secondary" 
                                    className="text-xs h-5 px-1.5"
                                  >
                                    {option}
                                  </Badge>
                                ))}
                              </div>
                              
                              {/* Dimensions */}
                              {(product.length || product.width || product.height) && (
                                <div className="text-xs text-muted-foreground mb-1.5">
                                  {product.length && `${formatCm(product.length)}cm`}
                                  {product.length && product.width && ' × '}
                                  {product.width && `${formatCm(product.width)}cm`}
                                  {(product.length || product.width) && product.height && ' × '}
                                  {product.height && `${formatCm(product.height)}cm`}
                                </div>
                              )}
                              
                              {/* Price & Stats */}
                              <div className="flex items-center justify-between">
                                <div className="font-medium text-sm">
                                  {product.basePrice 
                                    ? `${Number(product.basePrice).toFixed(2)} ${product.currency}`
                                    : "-"}
                                </div>
                                <div className="flex items-center gap-1.5">
                                  <Badge variant="outline" className="gap-0.5 text-xs px-1 py-0">
                                    <ImageIcon className="h-2.5 w-2.5" />
                                    {product.imagesCount}
                                  </Badge>
                                  <div 
                                    onClick={(e) => {
                                      e.stopPropagation();
                                      toggleActive(product.id, product.isActive);
                                    }}
                                  >
                                    <Switch
                                      checked={product.isActive}
                                      data-testid={`switch-mobile-active-${product.id}`}
                                    />
                                  </div>
                                </div>
                              </div>
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    ))
                  )}
                </div>

                {/* Desktop Table View - visible on screens >= md */}
                <Table className="hidden md:table">
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-12">
                      <Checkbox
                        checked={selectedProducts.length === filteredProducts.length && filteredProducts.length > 0}
                        onCheckedChange={toggleAll}
                        data-testid="checkbox-select-all"
                      />
                    </TableHead>
                    <TableHead className="w-16"></TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50 w-16"
                      onClick={() => handleSort('id')}
                    >
                      <div className="flex items-center gap-1">
                        ID
                        {sortBy === 'id' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('sku')}
                    >
                      <div className="flex items-center gap-1">
                        SKU
                        {sortBy === 'sku' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead className="font-medium text-muted-foreground">
                      SKU Airtable
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('productType')}
                    >
                      <div className="flex items-center gap-1">
                        Rodzaj
                        {sortBy === 'productType' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('length')}
                    >
                      <div className="flex items-center gap-1">
                        Dł.
                        {sortBy === 'length' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('width')}
                    >
                      <div className="flex items-center gap-1">
                        Szer.
                        {sortBy === 'width' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('color')}
                    >
                      <div className="flex items-center gap-1">
                        Kolor
                        {sortBy === 'color' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead className="font-medium text-muted-foreground">
                      Opcje kolor
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('doors')}
                    >
                      <div className="flex items-center gap-1">
                        Drzwi
                        {sortBy === 'doors' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('legs')}
                    >
                      <div className="flex items-center gap-1">
                        Nogi
                        {sortBy === 'legs' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead className="text-center">BOM</TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('title')}
                    >
                      <div className="flex items-center gap-1">
                        Nazwa
                        {sortBy === 'title' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('basePrice')}
                    >
                      <div className="flex items-center gap-1">
                        Cena
                        {sortBy === 'basePrice' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead className="text-center">Zdj.</TableHead>
                    <TableHead 
                      className="cursor-pointer select-none hover:bg-muted/50"
                      onClick={() => handleSort('isActive')}
                    >
                      <div className="flex items-center gap-1">
                        Status
                        {sortBy === 'isActive' && (
                          sortOrder === 'ASC' ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />
                        )}
                      </div>
                    </TableHead>
                    <TableHead className="text-center">Połączenie</TableHead>
                    <TableHead className="text-right">Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {groupBy ? (
                    // Grouped view
                    getGroupedProducts()?.map(({ key: groupKey, products: groupProducts }) => {
                      const isExpanded = expandedGroups.has(groupKey);
                      return (
                        <Fragment key={groupKey}>
                          {/* Group Header Row */}
                          <TableRow
                            className="bg-muted/30 font-semibold hover-elevate cursor-pointer"
                            onClick={() => toggleGroup(groupKey)}
                            data-testid={`row-group-${groupKey}`}
                          >
                            <TableCell colSpan={16} className="py-2">
                              <div className="flex items-center gap-2">
                                {isExpanded ? (
                                  <ChevronUp className="h-4 w-4" />
                                ) : (
                                  <ChevronDown className="h-4 w-4" />
                                )}
                                <span className="text-sm">
                                  {groupKey} <span className="text-xs text-muted-foreground font-normal">({groupProducts.length} produktów)</span>
                                </span>
                              </div>
                            </TableCell>
                          </TableRow>
                          {/* Group Products */}
                          {isExpanded && groupProducts.map((product) => {
                            const isSelected = selectedProducts.includes(product.id);
                            const rowBgClass = isSelected ? 'bg-green-500/20 hover:bg-green-500/25' : '';
                            
                            return (
                              <TableRow 
                                key={product.id} 
                                data-testid={`row-product-${product.id}`}
                                className={`cursor-pointer hover-elevate ${compactMode ? 'h-8' : ''} ${rowBgClass}`}
                                onClick={() => setLocation(`/catalog-products/${product.id}`)}
                              >
                              <TableCell className={compactMode ? 'py-1' : ''} onClick={(e) => e.stopPropagation()}>
                                <Checkbox
                                  checked={selectedProducts.includes(product.id)}
                                  onCheckedChange={() => toggleProduct(product.id)}
                                  data-testid={`checkbox-product-${product.id}`}
                                />
                              </TableCell>
                              <TableCell className={compactMode ? 'py-1' : ''}>
                                {product.primaryImageUrl ? (
                                  <img 
                                    src={product.primaryImageUrl} 
                                    alt={product.title}
                                    className={compactMode ? 'w-10 h-10 object-cover' : 'w-20 h-20 object-cover'}
                                    data-testid={`img-product-${product.id}`}
                                  />
                                ) : (
                                  <div className={compactMode ? 'w-10 h-10 bg-muted flex items-center justify-center' : 'w-20 h-20 bg-muted flex items-center justify-center'}>
                                    <Image className={compactMode ? 'h-5 w-5 text-muted-foreground' : 'h-10 w-10 text-muted-foreground'} />
                                  </div>
                                )}
                              </TableCell>
                              <TableCell className={`text-muted-foreground font-mono ${compactMode ? 'text-xs py-1' : 'text-sm'}`}>
                                {product.id}
                              </TableCell>
                              <TableCell 
                                className={`font-mono cursor-pointer hover:bg-muted/50 ${compactMode ? 'text-xs py-1' : 'text-sm'}`}
                                onClick={(e) => {
                                  e.stopPropagation();
                                  startEditing(product.id, 'sku', product.sku);
                                }}
                              >
                                {editingCell?.productId === product.id && editingCell?.field === 'sku' ? (
                                  <Input
                                    value={editValue}
                                    onChange={(e) => setEditValue(e.target.value)}
                                    onBlur={() => saveEdit(product.id, 'sku')}
                                    onKeyDown={(e) => {
                                      if (e.key === 'Enter') saveEdit(product.id, 'sku');
                                      if (e.key === 'Escape') cancelEditing();
                                    }}
                                    autoFocus
                                    className="h-8"
                                    data-testid={`input-sku-${product.id}`}
                                  />
                                ) : (
                                  <span>{product.sku || "-"}</span>
                                )}
                              </TableCell>
                              <TableCell 
                                className={`font-mono ${compactMode ? 'text-xs py-1' : 'text-sm'} text-muted-foreground`}
                                onClick={(e) => e.stopPropagation()}
                              >
                                {product.skuAirtable ? (
                                  <button
                                    onClick={(e) => {
                                      e.stopPropagation();
                                      copySkuAirtableToSku(product.id, product.skuAirtable!);
                                    }}
                                    className="flex items-center gap-1 hover:text-foreground transition-colors cursor-pointer group"
                                    title="Kliknij aby skopiować do pola SKU"
                                    data-testid={`button-copy-sku-${product.id}`}
                                  >
                                    <span>{product.skuAirtable}</span>
                                    <Copy className="w-3 h-3 opacity-0 group-hover:opacity-100 transition-opacity" />
                                  </button>
                                ) : (
                                  "-"
                                )}
                              </TableCell>
                              <TableCell className={compactMode ? 'text-xs py-1' : 'text-sm'}>
                                {product.productType ? (() => {
                                  const typeDict = productTypes?.find(t => t.code === product.productType);
                                  const bgColor = typeDict?.color;
                                  const textColorClass = getTextColorForBackground(bgColor);
                                  
                                  return (
                                    <Badge 
                                      variant="outline" 
                                      className={`${compactMode ? 'text-xs h-5' : ''} ${textColorClass} border-transparent`}
                                      style={bgColor ? { backgroundColor: bgColor } : undefined}
                                      data-testid={`badge-type-${product.id}`}
                                    >
                                      {product.productType}
                                    </Badge>
                                  );
                                })() : (
                                  <span className="text-muted-foreground">-</span>
                                )}
                              </TableCell>
                              <TableCell className={`${compactMode ? 'py-1' : ''}`}>
                                <div className="bg-black/90 px-2 py-1 rounded text-gray-300 font-mono text-xs text-center inline-block min-w-[50px]">
                                  {product.length ? `${formatCm(product.length)}` : "-"}
                                </div>
                              </TableCell>
                              <TableCell className={`${compactMode ? 'py-1' : ''}`}>
                                <div className="bg-black/90 px-2 py-1 rounded text-gray-300 font-mono text-xs text-center inline-block min-w-[50px]">
                                  {product.width ? `${formatCm(product.width)}` : "-"}
                                </div>
                              </TableCell>
                              <TableCell 
                                className={`cursor-pointer hover:bg-muted/50 ${compactMode ? 'text-xs py-1' : 'text-sm'}`}
                                onClick={(e) => {
                                  e.stopPropagation();
                                  startEditing(product.id, 'color', product.color);
                                }}
                              >
                                {editingCell?.productId === product.id && editingCell?.field === 'color' ? (
                                  <Select 
                                    value={editValue}
                                    onValueChange={(colorCode) => {
                                      setEditValue(colorCode);
                                      inlineUpdateMutation.mutate({ 
                                        productId: product.id, 
                                        field: 'color', 
                                        value: colorCode 
                                      });
                                    }}
                                  >
                                    <SelectTrigger className="h-8" data-testid={`select-color-${product.id}`}>
                                      <SelectValue placeholder="Wybierz kolor" />
                                    </SelectTrigger>
                                    <SelectContent>
                                      {colors?.filter(c => c.isActive).map((color) => (
                                        <SelectItem key={color.id} value={color.code}>
                                          {color.name || color.code}
                                        </SelectItem>
                                      ))}
                                    </SelectContent>
                                  </Select>
                                ) : (
                                  <div className="flex flex-wrap gap-1">
                                    {product.color ? (() => {
                                      const colorDict = colors?.find(c => c.code === product.color);
                                      const bgColor = colorDict?.color || '#94A3B8';
                                      const textColor = getTextColorForBackground(bgColor);
                                      return (
                                        <Badge 
                                          variant="outline" 
                                          className="text-xs h-5 px-1.5 border-transparent" 
                                          style={{ backgroundColor: bgColor, color: textColor }}
                                        >
                                          {product.color}
                                        </Badge>
                                      );
                                    })() : "-"}
                                  </div>
                                )}
                              </TableCell>
                              <TableCell className={compactMode ? 'text-xs py-1' : 'text-sm'}>
                                {product.colorOptions && product.colorOptions.length > 0 ? (
                                  <div className="flex flex-wrap gap-1">
                                    {product.colorOptions.map((option, idx) => (
                                      <Badge 
                                        key={idx}
                                        variant="secondary" 
                                        className={compactMode ? 'text-xs h-5 px-1.5' : 'text-xs'}
                                        data-testid={`badge-color-option-${product.id}-${idx}`}
                                      >
                                        {option}
                                      </Badge>
                                    ))}
                                  </div>
                                ) : (
                                  <span className="text-muted-foreground">-</span>
                                )}
                              </TableCell>
                              <TableCell className={compactMode ? 'text-xs py-1' : 'text-sm'}>
                                {product.doors ? (
                                  <Badge variant="outline" className={compactMode ? 'text-xs h-5' : ''}>
                                    {product.doors}
                                  </Badge>
                                ) : (
                                  <span className="text-muted-foreground">-</span>
                                )}
                              </TableCell>
                              <TableCell className={compactMode ? 'text-xs py-1' : 'text-sm'}>
                                {product.legs ? (
                                  <Badge variant="outline" className={compactMode ? 'text-xs h-5' : ''}>
                                    {product.legs}
                                  </Badge>
                                ) : (
                                  <span className="text-muted-foreground">-</span>
                                )}
                              </TableCell>
                              <TableCell className={`text-center ${compactMode ? 'py-1' : ''}`}>
                                <Badge 
                                  variant="outline" 
                                  className={`${compactMode ? 'text-xs h-5' : ''} border-transparent`}
                                  style={{
                                    backgroundColor: product.bomCount && product.bomCount > 0 ? '#22c55e' : '#ef4444',
                                    color: 'white'
                                  }}
                                >
                                  {product.bomCount || 0}
                                </Badge>
                              </TableCell>
                              <TableCell className={`font-medium max-w-xs truncate ${compactMode ? 'text-xs py-1' : ''}`}>
                                {product.title || "-"}
                              </TableCell>
                              <TableCell 
                                className={`font-medium cursor-pointer hover:bg-muted/50 ${compactMode ? 'text-xs py-1' : ''}`}
                                onClick={(e) => {
                                  e.stopPropagation();
                                  startEditing(product.id, 'basePrice', product.basePrice);
                                }}
                              >
                                {editingCell?.productId === product.id && editingCell?.field === 'basePrice' ? (
                                  <Input
                                    type="number"
                                    step="0.01"
                                    value={editValue}
                                    onChange={(e) => setEditValue(e.target.value)}
                                    onBlur={() => saveEdit(product.id, 'basePrice')}
                                    onKeyDown={(e) => {
                                      if (e.key === 'Enter') saveEdit(product.id, 'basePrice');
                                      if (e.key === 'Escape') cancelEditing();
                                    }}
                                    autoFocus
                                    className="h-8 w-24"
                                    data-testid={`input-price-${product.id}`}
                                  />
                                ) : (
                                  <span>
                                    {product.basePrice 
                                      ? `${Number(product.basePrice).toFixed(2)} ${product.currency}`
                                      : "-"}
                                  </span>
                                )}
                              </TableCell>
                              <TableCell className={`text-center ${compactMode ? 'py-1' : ''}`}>
                                <Badge variant="outline" className={compactMode ? 'gap-0.5 text-xs h-5' : 'gap-1'}>
                                  <Image className={compactMode ? 'h-2.5 w-2.5' : 'h-3 w-3'} />
                                  {product.imagesCount}
                                </Badge>
                              </TableCell>
                              <TableCell className={compactMode ? 'py-1' : ''} onClick={(e) => e.stopPropagation()}>
                                <div className="flex items-center gap-2">
                                  <Switch
                                    checked={product.isActive}
                                    onCheckedChange={() => toggleActive(product.id, product.isActive)}
                                    data-testid={`switch-active-${product.id}`}
                                  />
                                  {!compactMode && (
                                    <span className="text-sm text-muted-foreground">
                                      {product.isActive ? "Aktywny" : "Nieaktywny"}
                                    </span>
                                  )}
                                </div>
                              </TableCell>
                              <TableCell className={`text-center ${compactMode ? 'py-1' : ''}`} onClick={(e) => e.stopPropagation()}>
                                <ConnectionStatusIcon
                                  isLinked={product.marketplaceLinksCount > 0}
                                  targetPath={product.firstMarketplaceSku ? `/product/${product.firstMarketplaceSku}` : null}
                                  targetSku={product.firstMarketplaceSku}
                                  productId={product.id}
                                  onManageLinks={() => {
                                    setSelectedProductForLinking(product);
                                    setLinkDialogOpen(true);
                                  }}
                                />
                              </TableCell>
                              <TableCell className={compactMode ? 'py-1' : ''} onClick={(e) => e.stopPropagation()}>
                                <div className="flex items-center justify-end gap-1">
                                  {product.odooProductId && odooUrlData?.userUrl && (
                                    <Button
                                      size="icon"
                                      variant="outline"
                                      onClick={(e) => {
                                        e.stopPropagation();
                                        window.open(`${odooUrlData.userUrl}/products/${product.odooProductId}`, '_blank');
                                      }}
                                      data-testid={`button-odoo-link-${product.id}`}
                                      title="Otwórz produkt w Odoo"
                                      className="bg-purple-600 hover:bg-purple-700 border-purple-600 text-white"
                                    >
                                      <ExternalLink className="h-4 w-4" />
                                    </Button>
                                  )}
                                  <Button
                                    size="icon"
                                    variant={product.odooProductId ? "default" : "destructive"}
                                    onClick={(e) => {
                                      e.stopPropagation();
                                      handleProductExport(product.id);
                                    }}
                                    disabled={exportingProducts.has(product.id)}
                                    data-testid={`button-export-odoo-${product.id}`}
                                    title={
                                      product.odooExportError 
                                        ? `Błąd eksportu: ${product.odooExportError}` 
                                        : product.odooProductId 
                                          ? `Wyeksportowano: ${new Date(product.odooExportedAt!).toLocaleString('pl-PL')}` 
                                          : "Nie wyeksportowano - kliknij aby wysłać do Odoo"
                                    }
                                    className={
                                      product.odooExportError 
                                        ? "bg-yellow-500 hover:bg-yellow-600 relative" 
                                        : product.odooProductId 
                                          ? "bg-green-600 hover:bg-green-700" 
                                          : ""
                                    }
                                  >
                                    {exportingProducts.has(product.id) ? (
                                      <Loader2 className="h-4 w-4 animate-spin" />
                                    ) : (
                                      <Upload className="h-4 w-4" />
                                    )}
                                    {product.odooExportError && (
                                      <span className="absolute -top-1 -right-1 flex h-3 w-3">
                                        <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-red-400 opacity-75"></span>
                                        <span className="relative inline-flex rounded-full h-3 w-3 bg-red-500"></span>
                                      </span>
                                    )}
                                  </Button>
                                  <Button
                                    size="icon"
                                    variant="ghost"
                                    onClick={() => setLocation(`/catalog-products/${product.id}`)}
                                    data-testid={`button-edit-${product.id}`}
                                    title="Edytuj"
                                  >
                                    <Edit className="h-4 w-4" />
                                  </Button>
                                </div>
                              </TableCell>
                            </TableRow>
                            );
                          })}
                        </Fragment>
                      );
                    })
                  ) : (
                    // Normal view (no grouping)
                    filteredProducts.map((product) => {
                      const isSelected = selectedProducts.includes(product.id);
                      const rowBgClass = isSelected ? 'bg-green-500/20 hover:bg-green-500/25' : '';
                      
                      return (
                        <TableRow 
                          key={product.id} 
                          data-testid={`row-product-${product.id}`}
                          className={`cursor-pointer hover-elevate ${compactMode ? 'h-8' : ''} ${rowBgClass}`}
                          onClick={() => setLocation(`/catalog-products/${product.id}`)}
                        >
                      <TableCell className={compactMode ? 'py-1' : ''} onClick={(e) => e.stopPropagation()}>
                        <Checkbox
                          checked={selectedProducts.includes(product.id)}
                          onCheckedChange={() => toggleProduct(product.id)}
                          data-testid={`checkbox-product-${product.id}`}
                        />
                      </TableCell>
                      <TableCell className={compactMode ? 'py-1' : ''}>
                        {product.primaryImageUrl ? (
                          <img 
                            src={product.primaryImageUrl} 
                            alt={product.title}
                            className={compactMode ? 'w-10 h-10 object-cover' : 'w-20 h-20 object-cover'}
                            data-testid={`img-product-${product.id}`}
                          />
                        ) : (
                          <div className={compactMode ? 'w-10 h-10 bg-muted flex items-center justify-center' : 'w-20 h-20 bg-muted flex items-center justify-center'}>
                            <Image className={compactMode ? 'h-5 w-5 text-muted-foreground' : 'h-10 w-10 text-muted-foreground'} />
                          </div>
                        )}
                      </TableCell>
                      <TableCell className={`text-muted-foreground font-mono ${compactMode ? 'text-xs py-1' : 'text-sm'}`}>
                        {product.id}
                      </TableCell>
                      <TableCell 
                        className={`font-mono cursor-pointer hover:bg-muted/50 ${compactMode ? 'text-xs py-1' : 'text-sm'}`}
                        onClick={(e) => {
                          e.stopPropagation();
                          startEditing(product.id, 'sku', product.sku);
                        }}
                      >
                        {editingCell?.productId === product.id && editingCell?.field === 'sku' ? (
                          <Input
                            value={editValue}
                            onChange={(e) => setEditValue(e.target.value)}
                            onBlur={() => saveEdit(product.id, 'sku')}
                            onKeyDown={(e) => {
                              if (e.key === 'Enter') saveEdit(product.id, 'sku');
                              if (e.key === 'Escape') cancelEditing();
                            }}
                            autoFocus
                            className="h-8"
                            data-testid={`input-sku-${product.id}`}
                          />
                        ) : (
                          <span>{product.sku || "-"}</span>
                        )}
                      </TableCell>
                      <TableCell 
                        className={`font-mono ${compactMode ? 'text-xs py-1' : 'text-sm'} text-muted-foreground`}
                        onClick={(e) => e.stopPropagation()}
                      >
                        {product.skuAirtable ? (
                          <button
                            onClick={(e) => {
                              e.stopPropagation();
                              copySkuAirtableToSku(product.id, product.skuAirtable!);
                            }}
                            className="flex items-center gap-1 hover:text-foreground transition-colors cursor-pointer group"
                            title="Kliknij aby skopiować do pola SKU"
                            data-testid={`button-copy-sku-${product.id}`}
                          >
                            <span>{product.skuAirtable}</span>
                            <Copy className="w-3 h-3 opacity-0 group-hover:opacity-100 transition-opacity" />
                          </button>
                        ) : (
                          "-"
                        )}
                      </TableCell>
                      <TableCell className={compactMode ? 'text-xs py-1' : 'text-sm'}>
                        {product.productType ? (() => {
                          const typeDict = productTypes?.find(t => t.code === product.productType);
                          const bgColor = typeDict?.color;
                          const textColorClass = getTextColorForBackground(bgColor);
                          
                          return (
                            <Badge 
                              variant="outline" 
                              className={`${compactMode ? 'text-xs h-5' : ''} ${textColorClass} border-transparent`}
                              style={bgColor ? { backgroundColor: bgColor } : undefined}
                              data-testid={`badge-type-${product.id}`}
                            >
                              {product.productType}
                            </Badge>
                          );
                        })() : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className={`${compactMode ? 'py-1' : ''}`}>
                        <div className="bg-black/90 px-2 py-1 rounded text-gray-300 font-mono text-xs text-center inline-block min-w-[50px]">
                          {product.length ? `${formatCm(product.length)}` : "-"}
                        </div>
                      </TableCell>
                      <TableCell className={`${compactMode ? 'py-1' : ''}`}>
                        <div className="bg-black/90 px-2 py-1 rounded text-gray-300 font-mono text-xs text-center inline-block min-w-[50px]">
                          {product.width ? `${formatCm(product.width)}` : "-"}
                        </div>
                      </TableCell>
                      <TableCell 
                        className={`cursor-pointer hover:bg-muted/50 ${compactMode ? 'text-xs py-1' : 'text-sm'}`}
                        onClick={(e) => {
                          e.stopPropagation();
                          startEditing(product.id, 'color', product.color);
                        }}
                      >
                        {editingCell?.productId === product.id && editingCell?.field === 'color' ? (
                          <Select 
                            value={editValue}
                            onValueChange={(colorCode) => {
                              setEditValue(colorCode);
                              inlineUpdateMutation.mutate({ 
                                productId: product.id, 
                                field: 'color', 
                                value: colorCode 
                              });
                            }}
                          >
                            <SelectTrigger className="h-8" data-testid={`select-color-${product.id}`}>
                              <SelectValue placeholder="Wybierz kolor" />
                            </SelectTrigger>
                            <SelectContent>
                              {colors?.filter(c => c.isActive).map((color) => (
                                <SelectItem key={color.id} value={color.code}>
                                  {color.name || color.code}
                                </SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                        ) : (
                          <div className="flex flex-wrap gap-1">
                            {product.color ? (() => {
                              const colorDict = colors?.find(c => c.code === product.color);
                              const bgColor = colorDict?.color || '#94A3B8';
                              const textColor = getTextColorForBackground(bgColor);
                              return (
                                <Badge 
                                  variant="outline" 
                                  className="text-xs h-5 px-1.5 border-transparent" 
                                  style={{ backgroundColor: bgColor, color: textColor }}
                                >
                                  {product.color}
                                </Badge>
                              );
                            })() : "-"}
                          </div>
                        )}
                      </TableCell>
                      <TableCell className={compactMode ? 'text-xs py-1' : 'text-sm'}>
                        {product.colorOptions && product.colorOptions.length > 0 ? (
                          <div className="flex flex-wrap gap-1">
                            {product.colorOptions.map((option, idx) => (
                              <Badge 
                                key={idx}
                                variant="secondary" 
                                className={compactMode ? 'text-xs h-5 px-1.5' : 'text-xs'}
                                data-testid={`badge-color-option-${product.id}-${idx}`}
                              >
                                {option}
                              </Badge>
                            ))}
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className={compactMode ? 'text-xs py-1' : 'text-sm'}>
                        {product.doors ? (
                          <Badge variant="outline" className={compactMode ? 'text-xs h-5' : ''}>
                            {product.doors}
                          </Badge>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className={compactMode ? 'text-xs py-1' : 'text-sm'}>
                        {product.legs ? (
                          <Badge variant="outline" className={compactMode ? 'text-xs h-5' : ''}>
                            {product.legs}
                          </Badge>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className={`text-center ${compactMode ? 'py-1' : ''}`}>
                        <Badge 
                          variant="outline" 
                          className={`${compactMode ? 'text-xs h-5' : ''} border-transparent`}
                          style={{
                            backgroundColor: product.bomCount && product.bomCount > 0 ? '#22c55e' : '#ef4444',
                            color: 'white'
                          }}
                        >
                          {product.bomCount || 0}
                        </Badge>
                      </TableCell>
                      <TableCell className={`font-medium max-w-xs truncate ${compactMode ? 'text-xs py-1' : ''}`}>
                        {product.title || "-"}
                      </TableCell>
                      <TableCell 
                        className={`font-medium cursor-pointer hover:bg-muted/50 ${compactMode ? 'text-xs py-1' : ''}`}
                        onClick={(e) => {
                          e.stopPropagation();
                          startEditing(product.id, 'basePrice', product.basePrice);
                        }}
                      >
                        {editingCell?.productId === product.id && editingCell?.field === 'basePrice' ? (
                          <Input
                            type="number"
                            step="0.01"
                            value={editValue}
                            onChange={(e) => setEditValue(e.target.value)}
                            onBlur={() => saveEdit(product.id, 'basePrice')}
                            onKeyDown={(e) => {
                              if (e.key === 'Enter') saveEdit(product.id, 'basePrice');
                              if (e.key === 'Escape') cancelEditing();
                            }}
                            autoFocus
                            className="h-8 w-24"
                            data-testid={`input-price-${product.id}`}
                          />
                        ) : (
                          <span>
                            {product.basePrice 
                              ? `${Number(product.basePrice).toFixed(2)} ${product.currency}`
                              : "-"}
                          </span>
                        )}
                      </TableCell>
                      <TableCell className={`text-center ${compactMode ? 'py-1' : ''}`}>
                        <Badge variant="outline" className={compactMode ? 'gap-0.5 text-xs h-5' : 'gap-1'}>
                          <Image className={compactMode ? 'h-2.5 w-2.5' : 'h-3 w-3'} />
                          {product.imagesCount}
                        </Badge>
                      </TableCell>
                      <TableCell className={compactMode ? 'py-1' : ''} onClick={(e) => e.stopPropagation()}>
                        <div className="flex items-center gap-2">
                          <Switch
                            checked={product.isActive}
                            onCheckedChange={() => toggleActive(product.id, product.isActive)}
                            data-testid={`switch-active-${product.id}`}
                          />
                          {!compactMode && (
                            <span className="text-sm text-muted-foreground">
                              {product.isActive ? "Aktywny" : "Nieaktywny"}
                            </span>
                          )}
                        </div>
                      </TableCell>
                      <TableCell className={`text-center ${compactMode ? 'py-1' : ''}`} onClick={(e) => e.stopPropagation()}>
                        <ConnectionStatusIcon
                          isLinked={product.marketplaceLinksCount > 0}
                          targetPath={product.firstMarketplaceSku ? `/product/${product.firstMarketplaceSku}` : null}
                          targetSku={product.firstMarketplaceSku}
                          productId={product.id}
                          onManageLinks={() => {
                            setSelectedProductForLinking(product);
                            setLinkDialogOpen(true);
                          }}
                        />
                      </TableCell>
                      <TableCell className={compactMode ? 'py-1' : ''} onClick={(e) => e.stopPropagation()}>
                        <div className="flex items-center justify-end gap-1">
                          {product.odooProductId && odooUrlData?.userUrl && (
                            <Button
                              size="icon"
                              variant="outline"
                              onClick={(e) => {
                                e.stopPropagation();
                                window.open(`${odooUrlData.userUrl}/products/${product.odooProductId}`, '_blank');
                              }}
                              data-testid={`button-odoo-link-${product.id}`}
                              title="Otwórz produkt w Odoo"
                              className="bg-purple-600 hover:bg-purple-700 border-purple-600 text-white"
                            >
                              <ExternalLink className="h-4 w-4" />
                            </Button>
                          )}
                          <Button
                            size="icon"
                            variant={product.odooProductId ? "default" : "destructive"}
                            onClick={(e) => {
                              e.stopPropagation();
                              handleProductExport(product.id);
                            }}
                            disabled={exportingProducts.has(product.id)}
                            data-testid={`button-export-odoo-${product.id}`}
                            title={
                              product.odooExportError 
                                ? `Błąd eksportu: ${product.odooExportError}` 
                                : product.odooProductId 
                                  ? `Wyeksportowano: ${new Date(product.odooExportedAt!).toLocaleString('pl-PL')}` 
                                  : "Nie wyeksportowano - kliknij aby wysłać do Odoo"
                            }
                            className={
                              product.odooExportError 
                                ? "bg-yellow-500 hover:bg-yellow-600 relative" 
                                : product.odooProductId 
                                  ? "bg-green-600 hover:bg-green-700" 
                                  : ""
                            }
                          >
                            {exportingProducts.has(product.id) ? (
                              <Loader2 className="h-4 w-4 animate-spin" />
                            ) : (
                              <Upload className="h-4 w-4" />
                            )}
                            {product.odooExportError && (
                              <span className="absolute -top-1 -right-1 flex h-3 w-3">
                                <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-red-400 opacity-75"></span>
                                <span className="relative inline-flex rounded-full h-3 w-3 bg-red-500"></span>
                              </span>
                            )}
                          </Button>
                          <Button
                            size="icon"
                            variant="ghost"
                            onClick={() => setLocation(`/catalog-products/${product.id}`)}
                            data-testid={`button-edit-${product.id}`}
                            title="Edytuj"
                          >
                            <Edit className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                      );
                  })
                  )}
                </TableBody>
              </Table>

              <div className="mt-4">
                <PaginationControls />
              </div>
              </>
            )}
          </CardContent>
        </Card>

        {/* Bulk Edit Dialog */}
        <Dialog open={isBulkEditOpen} onOpenChange={setIsBulkEditOpen}>
          <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Edytuj zaznaczone produkty ({selectedProducts.length})</DialogTitle>
              <DialogDescription>
                Wprowadź wartości dla pól które chcesz zaktualizować. Puste pola nie zostaną zmienione.
              </DialogDescription>
            </DialogHeader>
            <div className="grid grid-cols-2 gap-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="bulk-length">Długość (mm)</Label>
                <Input
                  id="bulk-length"
                  type="number"
                  placeholder="np. 500"
                  value={bulkLength}
                  onChange={(e) => setBulkLength(e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-width">Szerokość (mm)</Label>
                <Input
                  id="bulk-width"
                  type="number"
                  placeholder="np. 300"
                  value={bulkWidth}
                  onChange={(e) => setBulkWidth(e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-height">Wysokość (mm)</Label>
                <Input
                  id="bulk-height"
                  type="number"
                  placeholder="np. 800"
                  value={bulkHeight}
                  onChange={(e) => setBulkHeight(e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-price">Cena</Label>
                <Input
                  id="bulk-price"
                  type="number"
                  step="0.01"
                  placeholder="np. 299.99"
                  value={bulkPrice}
                  onChange={(e) => setBulkPrice(e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-color">Kolor</Label>
                <Select value={bulkColor || "none"} onValueChange={(val) => setBulkColor(val === "none" ? "" : val)}>
                  <SelectTrigger id="bulk-color">
                    <SelectValue placeholder="Bez zmiany" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Bez zmiany</SelectItem>
                    {colors?.filter(c => c.isActive).map((color) => (
                      <SelectItem key={color.id} value={color.code}>
                        {color.name || color.code}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-material">Materiał</Label>
                <Select value={bulkMaterial || "none"} onValueChange={(val) => setBulkMaterial(val === "none" ? "" : val)}>
                  <SelectTrigger id="bulk-material">
                    <SelectValue placeholder="Bez zmiany" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Bez zmiany</SelectItem>
                    {materials?.filter(m => m.isActive).map((material) => (
                      <SelectItem key={material.id} value={material.code}>
                        {material.name || material.code}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-type">Typ produktu</Label>
                <Select value={bulkProductType || "none"} onValueChange={(val) => setBulkProductType(val === "none" ? "" : val)}>
                  <SelectTrigger id="bulk-type">
                    <SelectValue placeholder="Bez zmiany" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Bez zmiany</SelectItem>
                    {productTypes?.filter(t => t.isActive).map((type) => (
                      <SelectItem key={type.id} value={type.code}>
                        {type.name || type.code}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-group">Grupa produktu</Label>
                <Select value={bulkProductGroup || "none"} onValueChange={(val) => setBulkProductGroup(val === "none" ? "" : val)}>
                  <SelectTrigger id="bulk-group">
                    <SelectValue placeholder="Bez zmiany" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Bez zmiany</SelectItem>
                    {productGroups?.filter(g => g.isActive).map((group) => (
                      <SelectItem key={group.id} value={group.code}>
                        {group.name || group.code}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-doors">Drzwi</Label>
                <Select value={bulkDoors || "none"} onValueChange={(val) => setBulkDoors(val === "none" ? "" : val)}>
                  <SelectTrigger id="bulk-doors">
                    <SelectValue placeholder="Bez zmiany" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Bez zmiany</SelectItem>
                    {doorsOptions?.filter(d => d.isActive).map((door) => (
                      <SelectItem key={door.id} value={door.code}>
                        {door.name || door.code}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="bulk-legs">Nóżki</Label>
                <Select value={bulkLegs || "none"} onValueChange={(val) => setBulkLegs(val === "none" ? "" : val)}>
                  <SelectTrigger id="bulk-legs">
                    <SelectValue placeholder="Bez zmiany" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Bez zmiany</SelectItem>
                    {legsOptions?.filter(l => l.isActive).map((leg) => (
                      <SelectItem key={leg.id} value={leg.code}>
                        {leg.name || leg.code}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsBulkEditOpen(false)}>
                Anuluj
              </Button>
              <Button onClick={handleBulkUpdate} disabled={bulkUpdateMutation.isPending}>
                {bulkUpdateMutation.isPending ? "Zapisywanie..." : "Zapisz zmiany"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        {/* Bulk Delete Dialog */}
        <Dialog open={isBulkDeleteOpen} onOpenChange={setIsBulkDeleteOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Czy na pewno chcesz usunąć produkty?</DialogTitle>
              <DialogDescription>
                Ta operacja usunie {selectedProducts.length} {selectedProducts.length === 1 ? 'produkt' : 'produktów'} i nie może być cofnięta.
              </DialogDescription>
            </DialogHeader>
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsBulkDeleteOpen(false)}>
                Anuluj
              </Button>
              <Button 
                variant="destructive" 
                onClick={() => bulkDeleteMutation.mutate()}
                disabled={bulkDeleteMutation.isPending}
              >
                {bulkDeleteMutation.isPending ? "Usuwanie..." : "Usuń"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        {/* Marketplace Link Management Dialog */}
        {selectedProductForLinking && (
          <MarketplaceLinkDialog
            open={linkDialogOpen}
            onOpenChange={setLinkDialogOpen}
            mode="catalog-to-marketplace"
            catalogProductId={selectedProductForLinking.id}
            onSuccess={() => {
              queryClient.invalidateQueries({ queryKey: ['/api/catalog/products'] });
            }}
          />
        )}
    </div>
  );
}
