import { Link, useRoute } from "wouter";
import { Plus, BookOpen, Palette, Box, Grid3x3, DoorClosed, Footprints, Shirt, Package, Ruler } from "lucide-react";
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarGroupLabel,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarHeader,
} from "@/components/ui/sidebar";
import { Button } from "@/components/ui/button";
import type { DictionaryType } from "@shared/schema";

const DICTIONARY_TYPES: { 
  value: DictionaryType; 
  label: string; 
  icon: typeof BookOpen;
}[] = [
  { value: "product_type", label: "Rodzaj produktu", icon: Box },
  { value: "unit", label: "Jednostka", icon: Grid3x3 },
  { value: "color", label: "Kolor", icon: Palette },
  { value: "material", label: "Materiał", icon: Package },
  { value: "product_group", label: "Grupa produktów", icon: Grid3x3 },
  { value: "door", label: "Drzwi", icon: DoorClosed },
  { value: "leg", label: "Nóżki", icon: Footprints },
  { value: "fabric", label: "Tkaniny", icon: Shirt },
  { value: "dimension_length", label: "Wymiar - długość", icon: Ruler },
  { value: "dimension_width", label: "Wymiar - szerokość", icon: Ruler },
  { value: "dimension_height", label: "Wymiar - wysokość", icon: Ruler },
];

interface DictionariesSidebarProps {
  onAddNew?: () => void;
}

export function DictionariesSidebar({ onAddNew }: DictionariesSidebarProps) {
  const [, params] = useRoute("/dictionaries/:type");
  const activeType = params?.type as DictionaryType | undefined;

  return (
    <Sidebar>
      <SidebarHeader className="border-b px-6 py-4">
        <div className="flex items-center justify-between gap-2">
          <h2 className="text-lg font-semibold">Słowniki</h2>
          {onAddNew && (
            <Button 
              size="icon" 
              variant="ghost"
              onClick={onAddNew}
              data-testid="button-add-dictionary-type"
            >
              <Plus className="h-4 w-4" />
            </Button>
          )}
        </div>
      </SidebarHeader>
      <SidebarContent>
        <SidebarGroup>
          <SidebarGroupLabel>Typy słowników</SidebarGroupLabel>
          <SidebarGroupContent>
            <SidebarMenu>
              {DICTIONARY_TYPES.map((dict) => {
                const Icon = dict.icon;
                const isActive = activeType === dict.value;
                
                return (
                  <SidebarMenuItem key={dict.value}>
                    <SidebarMenuButton asChild isActive={isActive}>
                      <Link href={`/dictionaries/${dict.value}`} data-testid={`link-dictionary-${dict.value}`}>
                        <Icon className="h-4 w-4" />
                        <span>{dict.label}</span>
                      </Link>
                    </SidebarMenuButton>
                  </SidebarMenuItem>
                );
              })}
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>
    </Sidebar>
  );
}
