import { useQuery } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Loader2, History, Eye } from "lucide-react";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useState } from "react";
import { Textarea } from "@/components/ui/textarea";

interface AIGenerationHistoryProps {
  productId: number;
  onReuse?: (description: string) => void;
}

interface AIHistoryItem {
  id: number;
  productId: number;
  templateId: number | null;
  status: string;
  response: string | null;
  costMetadata: {
    model?: string;
    tokens?: { prompt: number; completion: number; total: number };
    cost?: number;
    templateName?: string;
    tone?: string;
    language?: string;
  } | null;
  createdAt: string;
  templateName?: string;
  userName?: string;
}

export function AIGenerationHistory({ productId, onReuse }: AIGenerationHistoryProps) {
  const [open, setOpen] = useState(false);
  const [previewItem, setPreviewItem] = useState<AIHistoryItem | null>(null);

  const { data: history, isLoading } = useQuery<AIHistoryItem[]>({
    queryKey: ["/api/catalog-products", productId, "ai-history"],
    queryFn: async () => {
      const response = await fetch(`/api/catalog-products/${productId}/ai-history?limit=20`);
      if (!response.ok) throw new Error("Failed to fetch AI history");
      return await response.json();
    },
    enabled: open,
  });

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "completed":
        return <Badge variant="default">Ukończone</Badge>;
      case "failed":
        return <Badge variant="destructive">Błąd</Badge>;
      case "processing":
        return <Badge variant="secondary">Przetwarzanie</Badge>;
      default:
        return <Badge variant="outline">{status}</Badge>;
    }
  };

  const getToneName = (tone?: string) => {
    switch (tone) {
      case "professional":
        return "Profesjonalny";
      case "friendly":
        return "Przyjazny";
      case "casual":
        return "Swobodny";
      case "formal":
        return "Formalny";
      default:
        return tone || "-";
    }
  };

  const getLanguageName = (language?: string) => {
    switch (language) {
      case "pl":
        return "Polski";
      case "en":
        return "Angielski";
      default:
        return language || "-";
    }
  };

  return (
    <>
      <Dialog open={open} onOpenChange={setOpen}>
        <DialogTrigger asChild>
          <Button
            type="button"
            variant="ghost"
            size="sm"
            data-testid="button-ai-history"
          >
            <History className="h-4 w-4 mr-2" />
            Historia AI
          </Button>
        </DialogTrigger>
        <DialogContent className="max-w-5xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Historia generacji AI</DialogTitle>
            <DialogDescription>
              Poprzednie generacje opisów dla tego produktu
            </DialogDescription>
          </DialogHeader>

          {isLoading && (
            <div className="flex items-center justify-center py-8">
              <Loader2 className="h-8 w-8 animate-spin" />
            </div>
          )}

          {!isLoading && history && history.length === 0 && (
            <div className="text-center text-muted-foreground py-8">
              Brak historii generacji AI dla tego produktu
            </div>
          )}

          {!isLoading && history && history.length > 0 && (
            <div className="rounded-md border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Data</TableHead>
                    <TableHead>Szablon</TableHead>
                    <TableHead>Ton</TableHead>
                    <TableHead>Język</TableHead>
                    <TableHead>Model</TableHead>
                    <TableHead>Koszt</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead className="text-right">Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {history.map((item) => (
                    <TableRow key={item.id} data-testid={`row-history-${item.id}`}>
                      <TableCell className="whitespace-nowrap">
                        {format(new Date(item.createdAt), "dd MMM yyyy, HH:mm", { locale: pl })}
                      </TableCell>
                      <TableCell>
                        {item.costMetadata?.templateName || item.templateName || "-"}
                      </TableCell>
                      <TableCell>{getToneName(item.costMetadata?.tone)}</TableCell>
                      <TableCell>{getLanguageName(item.costMetadata?.language)}</TableCell>
                      <TableCell className="font-mono text-xs">
                        {item.costMetadata?.model || "-"}
                      </TableCell>
                      <TableCell className="font-mono">
                        ${item.costMetadata?.cost?.toFixed(4) || "0.0000"}
                      </TableCell>
                      <TableCell>{getStatusBadge(item.status)}</TableCell>
                      <TableCell className="text-right">
                        {item.response && (
                          <Button
                            type="button"
                            variant="ghost"
                            size="sm"
                            onClick={() => setPreviewItem(item)}
                            data-testid={`button-preview-${item.id}`}
                          >
                            <Eye className="h-4 w-4" />
                          </Button>
                        )}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {previewItem && (
        <Dialog open={!!previewItem} onOpenChange={() => setPreviewItem(null)}>
          <DialogContent className="max-w-3xl">
            <DialogHeader>
              <DialogTitle>Podgląd wygenerowanego opisu</DialogTitle>
              <DialogDescription>
                {format(new Date(previewItem.createdAt), "dd MMMM yyyy, HH:mm", { locale: pl })}
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-muted-foreground">Szablon:</span>{" "}
                  {previewItem.costMetadata?.templateName || "-"}
                </div>
                <div>
                  <span className="text-muted-foreground">Ton:</span>{" "}
                  {getToneName(previewItem.costMetadata?.tone)}
                </div>
                <div>
                  <span className="text-muted-foreground">Język:</span>{" "}
                  {getLanguageName(previewItem.costMetadata?.language)}
                </div>
                <div>
                  <span className="text-muted-foreground">Koszt:</span>{" "}
                  <span className="font-mono">
                    ${previewItem.costMetadata?.cost?.toFixed(4) || "0.0000"}
                  </span>
                </div>
              </div>
              <Textarea
                value={previewItem.response || ""}
                readOnly
                rows={15}
                className="font-mono text-sm"
                data-testid="textarea-preview-description"
              />
              {onReuse && (
                <div className="flex gap-2 justify-end">
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => setPreviewItem(null)}
                    data-testid="button-close-preview"
                  >
                    Zamknij
                  </Button>
                  <Button
                    type="button"
                    onClick={() => {
                      if (previewItem.response) {
                        onReuse(previewItem.response);
                        setPreviewItem(null);
                        setOpen(false);
                      }
                    }}
                    data-testid="button-reuse-description"
                  >
                    Użyj tego opisu
                  </Button>
                </div>
              )}
            </div>
          </DialogContent>
        </Dialog>
      )}
    </>
  );
}
