import type { Express } from "express";
import { requirePermission } from "../../auth";
import { pool } from "../../postgres";
import { 
  getLocations, 
  getLocationById, 
  createLocation, 
  updateLocation, 
  deleteLocation,
  getLocationPath,
  getLocationChildren
} from "../../services/production/locations";
import { insertProductionLocationSchema } from "@shared/schema";

const partialLocationSchema = insertProductionLocationSchema.partial();

export function setupLocationRoutes(app: Express) {
  // GET /api/production/locations - Get all locations (optionally filtered by parent)
  app.get("/api/production/locations", requirePermission("view_production"), async (req, res) => {
    try {
      const parentId = req.query.parentId 
        ? (req.query.parentId === 'null' ? null : parseInt(req.query.parentId as string))
        : undefined;
      
      const locations = await getLocations(pool, parentId);
      res.json(locations);
    } catch (error: any) {
      console.error("Error fetching locations:", error);
      res.status(500).json({ error: "Failed to fetch locations" });
    }
  });

  // GET /api/production/locations/:id - Get single location
  app.get("/api/production/locations/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const location = await getLocationById(pool, id);
      
      if (!location) {
        return res.status(404).json({ error: "Location not found" });
      }
      
      res.json(location);
    } catch (error: any) {
      console.error("Error fetching location:", error);
      res.status(500).json({ error: "Failed to fetch location" });
    }
  });

  // GET /api/production/locations/:id/path - Get location breadcrumb path
  app.get("/api/production/locations/:id/path", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const path = await getLocationPath(pool, id);
      res.json(path);
    } catch (error: any) {
      console.error("Error fetching location path:", error);
      res.status(500).json({ error: "Failed to fetch location path" });
    }
  });

  // GET /api/production/locations/:id/children - Get all child locations
  app.get("/api/production/locations/:id/children", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const children = await getLocationChildren(pool, id);
      res.json(children);
    } catch (error: any) {
      console.error("Error fetching location children:", error);
      res.status(500).json({ error: "Failed to fetch location children" });
    }
  });

  // POST /api/production/locations - Create location
  app.post("/api/production/locations", requirePermission("manage_production"), async (req, res) => {
    try {
      const parsed = insertProductionLocationSchema.safeParse(req.body);
      
      if (!parsed.success) {
        return res.status(400).json({ 
          error: "Invalid request data", 
          details: parsed.error.errors 
        });
      }
      
      const location = await createLocation(pool, parsed.data);
      res.status(201).json(location);
    } catch (error: any) {
      console.error("Error creating location:", error);
      res.status(500).json({ error: "Failed to create location" });
    }
  });

  // PATCH /api/production/locations/:id - Update location
  app.patch("/api/production/locations/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const parsed = partialLocationSchema.safeParse(req.body);
      
      if (!parsed.success) {
        return res.status(400).json({ 
          error: "Invalid request data", 
          details: parsed.error.errors 
        });
      }
      
      const location = await updateLocation(pool, id, parsed.data);
      
      if (!location) {
        return res.status(404).json({ error: "Location not found" });
      }
      
      res.json(location);
    } catch (error: any) {
      console.error("Error updating location:", error);
      // Return specific error message for validation errors
      if (error.message?.includes("parent") || error.message?.includes("descendants")) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "Failed to update location" });
    }
  });

  // DELETE /api/production/locations/:id - Delete location
  app.delete("/api/production/locations/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const deleted = await deleteLocation(pool, id);
      
      if (!deleted) {
        return res.status(404).json({ error: "Location not found" });
      }
      
      res.status(204).send();
    } catch (error: any) {
      console.error("Error deleting location:", error);
      res.status(500).json({ error: "Failed to delete location" });
    }
  });
}
