import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { 
  Table, 
  TableBody, 
  TableCell, 
  TableHead, 
  TableHeader, 
  TableRow 
} from "@/components/ui/table";
import { 
  AlertCircle, 
  AlertTriangle, 
  Info, 
  CheckCircle, 
  Search, 
  Trash2,
  RefreshCw,
  ChevronLeft,
  ChevronRight
} from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface ErrorLog {
  id: number;
  type: string;
  message: string;
  stack_trace?: string;
  context?: any;
  severity: string;
  timestamp: string;
  resolved_at?: string;
  resolved_by?: number;
  created_at: string;
}

interface ErrorLogsResponse {
  logs: ErrorLog[];
  total: number;
  limit: number;
  offset: number;
}

const ERROR_TYPES = [
  { value: "all", label: "Wszystkie typy" },
  { value: "allegro", label: "Allegro" },
  { value: "shoper", label: "Shoper" },
  { value: "odoo", label: "Odoo" },
  { value: "database", label: "Baza danych" },
  { value: "api", label: "API" },
  { value: "webhook", label: "Webhook" },
  { value: "sync", label: "Synchronizacja" },
  { value: "auth", label: "Autentykacja" },
];

const SEVERITY_LEVELS = [
  { value: "all", label: "Wszystkie poziomy" },
  { value: "error", label: "Error" },
  { value: "warning", label: "Warning" },
  { value: "info", label: "Info" },
];

const RESOLVED_STATUSES = [
  { value: "all", label: "Wszystkie" },
  { value: "false", label: "Nierozwiązane" },
  { value: "true", label: "Rozwiązane" },
];

function getSeverityIcon(severity: string) {
  switch (severity) {
    case "error":
      return <AlertCircle className="h-4 w-4 text-destructive" />;
    case "warning":
      return <AlertTriangle className="h-4 w-4 text-yellow-500" />;
    case "info":
      return <Info className="h-4 w-4 text-blue-500" />;
    default:
      return <Info className="h-4 w-4" />;
  }
}

function getSeverityBadge(severity: string) {
  switch (severity) {
    case "error":
      return <Badge variant="destructive">Error</Badge>;
    case "warning":
      return <Badge className="bg-yellow-500 text-white">Warning</Badge>;
    case "info":
      return <Badge className="bg-blue-500 text-white">Info</Badge>;
    default:
      return <Badge variant="secondary">{severity}</Badge>;
  }
}

function getTypeBadge(type: string) {
  const typeConfig: Record<string, { variant: any; label: string }> = {
    allegro: { variant: "default", label: "Allegro" },
    shoper: { variant: "default", label: "Shoper" },
    odoo: { variant: "default", label: "Odoo" },
    database: { variant: "secondary", label: "Database" },
    api: { variant: "secondary", label: "API" },
    webhook: { variant: "secondary", label: "Webhook" },
    sync: { variant: "secondary", label: "Sync" },
    auth: { variant: "secondary", label: "Auth" },
  };

  const config = typeConfig[type] || { variant: "outline", label: type };
  return <Badge variant={config.variant}>{config.label}</Badge>;
}

export default function ErrorLogsPage() {
  const { toast } = useToast();
  const [typeFilter, setTypeFilter] = useState("all");
  const [severityFilter, setSeverityFilter] = useState("all");
  const [resolvedFilter, setResolvedFilter] = useState("false");
  const [searchQuery, setSearchQuery] = useState("");
  const [currentPage, setCurrentPage] = useState(0);
  const [selectedLog, setSelectedLog] = useState<ErrorLog | null>(null);
  const [detailsOpen, setDetailsOpen] = useState(false);
  
  const limit = 50;

  const queryParams = new URLSearchParams({
    type: typeFilter,
    severity: severityFilter,
    resolved: resolvedFilter,
    limit: limit.toString(),
    offset: (currentPage * limit).toString(),
    ...(searchQuery && { search: searchQuery }),
  });

  const { data, isLoading, refetch } = useQuery<ErrorLogsResponse>({
    queryKey: ["/api/error-logs", typeFilter, severityFilter, resolvedFilter, searchQuery, currentPage],
    queryFn: async () => {
      const response = await fetch(`/api/error-logs?${queryParams}`);
      if (!response.ok) throw new Error("Failed to fetch error logs");
      return response.json();
    },
  });

  const resolveMutation = useMutation({
    mutationFn: async (logId: number) => {
      const response = await fetch(`/api/error-logs/${logId}/resolve`, {
        method: "PATCH",
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to resolve error log");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/error-logs"] });
      toast({
        title: "Sukces",
        description: "Błąd został oznaczony jako rozwiązany",
      });
      setDetailsOpen(false);
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się oznaczyć błędu jako rozwiązanego",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (logId: number) => {
      const response = await fetch(`/api/error-logs/${logId}`, {
        method: "DELETE",
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to delete error log");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/error-logs"] });
      toast({
        title: "Sukces",
        description: "Błąd został usunięty",
      });
      setDetailsOpen(false);
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć błędu",
        variant: "destructive",
      });
    },
  });

  const cleanupMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch("/api/error-logs/cleanup", {
        method: "POST",
        credentials: "include",
        body: JSON.stringify({ days: 30 }),
        headers: { "Content-Type": "application/json" },
      });
      if (!response.ok) throw new Error("Failed to cleanup error logs");
      return response.json();
    },
    onSuccess: (response: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/error-logs"] });
      toast({
        title: "Sukces",
        description: `Usunięto ${response.deletedCount} starych logów`,
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się wyczyścić starych logów",
        variant: "destructive",
      });
    },
  });

  const totalPages = data ? Math.ceil(data.total / limit) : 0;

  const handleViewDetails = (log: ErrorLog) => {
    setSelectedLog(log);
    setDetailsOpen(true);
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Logi błędów</h1>
          <p className="text-muted-foreground mt-1">
            Przeglądaj i zarządzaj logami błędów systemu
          </p>
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            onClick={() => refetch()}
            disabled={isLoading}
            data-testid="button-refresh"
          >
            <RefreshCw className={`h-4 w-4 mr-2 ${isLoading ? "animate-spin" : ""}`} />
            Odśwież
          </Button>
          <Button
            variant="outline"
            onClick={() => cleanupMutation.mutate()}
            disabled={cleanupMutation.isPending}
            data-testid="button-cleanup"
          >
            <Trash2 className="h-4 w-4 mr-2" />
            Wyczyść stare (30+ dni)
          </Button>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Filtry</CardTitle>
          <CardDescription>Filtruj logi według typu, poziomu i statusu</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            <div className="space-y-2">
              <label className="text-sm font-medium">Typ</label>
              <Select value={typeFilter} onValueChange={setTypeFilter}>
                <SelectTrigger data-testid="select-type-filter">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {ERROR_TYPES.map((type) => (
                    <SelectItem key={type.value} value={type.value}>
                      {type.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <label className="text-sm font-medium">Poziom</label>
              <Select value={severityFilter} onValueChange={setSeverityFilter}>
                <SelectTrigger data-testid="select-severity-filter">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {SEVERITY_LEVELS.map((level) => (
                    <SelectItem key={level.value} value={level.value}>
                      {level.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <label className="text-sm font-medium">Status</label>
              <Select value={resolvedFilter} onValueChange={setResolvedFilter}>
                <SelectTrigger data-testid="select-resolved-filter">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {RESOLVED_STATUSES.map((status) => (
                    <SelectItem key={status.value} value={status.value}>
                      {status.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <label className="text-sm font-medium">Wyszukiwanie</label>
              <div className="relative">
                <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Szukaj w komunikatach..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-8"
                  data-testid="input-search"
                />
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Logi ({data?.total || 0})</CardTitle>
              <CardDescription>
                Wyświetlanie {data?.offset || 0 + 1}-
                {Math.min((data?.offset || 0) + (data?.limit || 0), data?.total || 0)} z {data?.total || 0}
              </CardDescription>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <div className="rounded-md border">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-[100px]">Poziom</TableHead>
                  <TableHead className="w-[120px]">Typ</TableHead>
                  <TableHead>Komunikat</TableHead>
                  <TableHead className="w-[180px]">Data</TableHead>
                  <TableHead className="w-[100px]">Status</TableHead>
                  <TableHead className="w-[100px]">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {isLoading ? (
                  <TableRow>
                    <TableCell colSpan={6} className="text-center py-8">
                      <RefreshCw className="h-6 w-6 animate-spin mx-auto mb-2" />
                      Ładowanie...
                    </TableCell>
                  </TableRow>
                ) : data?.logs.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                      Brak logów do wyświetlenia
                    </TableCell>
                  </TableRow>
                ) : (
                  data?.logs.map((log) => (
                    <TableRow 
                      key={log.id} 
                      className="cursor-pointer hover-elevate"
                      onClick={() => handleViewDetails(log)}
                      data-testid={`row-log-${log.id}`}
                    >
                      <TableCell>
                        <div className="flex items-center gap-2">
                          {getSeverityIcon(log.severity)}
                          {getSeverityBadge(log.severity)}
                        </div>
                      </TableCell>
                      <TableCell>{getTypeBadge(log.type)}</TableCell>
                      <TableCell className="max-w-md">
                        <div className="truncate" title={log.message}>
                          {log.message}
                        </div>
                      </TableCell>
                      <TableCell className="text-sm text-muted-foreground">
                        {format(new Date(log.timestamp), "dd MMM yyyy HH:mm", { locale: pl })}
                      </TableCell>
                      <TableCell>
                        {log.resolved_at ? (
                          <Badge variant="outline" className="gap-1">
                            <CheckCircle className="h-3 w-3" />
                            Rozwiązane
                          </Badge>
                        ) : (
                          <Badge variant="secondary">Aktywne</Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={(e) => {
                            e.stopPropagation();
                            handleViewDetails(log);
                          }}
                          data-testid={`button-view-${log.id}`}
                        >
                          Szczegóły
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </div>

          {totalPages > 1 && (
            <div className="flex items-center justify-between mt-4">
              <div className="text-sm text-muted-foreground">
                Strona {currentPage + 1} z {totalPages}
              </div>
              <div className="flex gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setCurrentPage(currentPage - 1)}
                  disabled={currentPage === 0}
                  data-testid="button-prev-page"
                >
                  <ChevronLeft className="h-4 w-4 mr-1" />
                  Poprzednia
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setCurrentPage(currentPage + 1)}
                  disabled={currentPage >= totalPages - 1}
                  data-testid="button-next-page"
                >
                  Następna
                  <ChevronRight className="h-4 w-4 ml-1" />
                </Button>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={detailsOpen} onOpenChange={setDetailsOpen}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>Szczegóły błędu #{selectedLog?.id}</DialogTitle>
            <DialogDescription>
              {selectedLog && format(new Date(selectedLog.timestamp), "dd MMMM yyyy, HH:mm:ss", { locale: pl })}
            </DialogDescription>
          </DialogHeader>

          {selectedLog && (
            <ScrollArea className="max-h-[60vh]">
              <div className="space-y-4">
                <div className="flex gap-2">
                  {getTypeBadge(selectedLog.type)}
                  {getSeverityBadge(selectedLog.severity)}
                  {selectedLog.resolved_at && (
                    <Badge variant="outline" className="gap-1">
                      <CheckCircle className="h-3 w-3" />
                      Rozwiązane {format(new Date(selectedLog.resolved_at), "dd MMM yyyy HH:mm", { locale: pl })}
                    </Badge>
                  )}
                </div>

                <Separator />

                <div>
                  <h4 className="font-semibold mb-2">Komunikat</h4>
                  <p className="text-sm bg-muted p-3 rounded-md">{selectedLog.message}</p>
                </div>

                {selectedLog.stack_trace && (
                  <div>
                    <h4 className="font-semibold mb-2">Stack Trace</h4>
                    <pre className="text-xs bg-muted p-3 rounded-md overflow-x-auto whitespace-pre-wrap">
                      {selectedLog.stack_trace}
                    </pre>
                  </div>
                )}

                {selectedLog.context && Object.keys(selectedLog.context).length > 0 && (
                  <div>
                    <h4 className="font-semibold mb-2">Kontekst</h4>
                    <pre className="text-xs bg-muted p-3 rounded-md overflow-x-auto">
                      {JSON.stringify(selectedLog.context, null, 2)}
                    </pre>
                  </div>
                )}

                <Separator />

                <div className="flex gap-2">
                  {!selectedLog.resolved_at && (
                    <Button
                      onClick={() => resolveMutation.mutate(selectedLog.id)}
                      disabled={resolveMutation.isPending}
                      data-testid="button-resolve"
                    >
                      <CheckCircle className="h-4 w-4 mr-2" />
                      Oznacz jako rozwiązane
                    </Button>
                  )}
                  <Button
                    variant="destructive"
                    onClick={() => deleteMutation.mutate(selectedLog.id)}
                    disabled={deleteMutation.isPending}
                    data-testid="button-delete"
                  >
                    <Trash2 className="h-4 w-4 mr-2" />
                    Usuń
                  </Button>
                </div>
              </div>
            </ScrollArea>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
