import { Router } from "express";
import { pool } from "../../postgres";
import {
  getProductionSettings,
  getProductionSettingById,
  createProductionSetting,
  updateProductionSetting,
  deleteProductionSetting,
} from "../../services/production/settings";
import { insertProductionSettingSchema } from "@shared/schema";

export const settingsRouter = Router();

// GET /api/production/settings - Get all production settings
settingsRouter.get("/", async (req, res) => {
  try {
    const settings = await getProductionSettings(pool);
    res.json(settings);
  } catch (error: any) {
    console.error("Error fetching production settings:", error);
    res.status(500).json({ error: "Failed to fetch production settings" });
  }
});

// GET /api/production/settings/:id - Get single setting by ID
settingsRouter.get("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    const setting = await getProductionSettingById(pool, id);
    
    if (!setting) {
      return res.status(404).json({ error: "Production setting not found" });
    }
    
    res.json(setting);
  } catch (error: any) {
    console.error("Error fetching production setting:", error);
    res.status(500).json({ error: "Failed to fetch production setting" });
  }
});

// POST /api/production/settings - Create new setting
settingsRouter.post("/", async (req, res) => {
  try {
    const parsed = insertProductionSettingSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    const setting = await createProductionSetting(pool, parsed.data);
    res.status(201).json(setting);
  } catch (error: any) {
    console.error("Error creating production setting:", error);
    res.status(500).json({ error: "Failed to create production setting" });
  }
});

// PATCH /api/production/settings/:id - Update setting
settingsRouter.patch("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    
    // Validate partial update with Zod
    const partialSchema = insertProductionSettingSchema.partial();
    const parsed = partialSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    // Reject empty updates
    if (Object.keys(parsed.data).length === 0) {
      return res.status(400).json({ error: "No fields to update" });
    }
    
    const setting = await updateProductionSetting(pool, id, parsed.data);
    res.json(setting);
  } catch (error: any) {
    console.error("Error updating production setting:", error);
    
    if (error.message.includes("not found")) {
      return res.status(404).json({ error: error.message });
    }
    
    res.status(500).json({ error: "Failed to update production setting" });
  }
});

// DELETE /api/production/settings/:id - Delete setting
settingsRouter.delete("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    await deleteProductionSetting(pool, id);
    res.status(204).send();
  } catch (error: any) {
    console.error("Error deleting production setting:", error);
    res.status(500).json({ error: "Failed to delete production setting" });
  }
});
