import { pool } from './postgres.js';
import axios from 'axios';

async function manualInsert() {
  const client = await pool.connect();
  
  try {
    console.log('🔄 Ręczne dodawanie zamówienia #00676...\n');

    const tokenResult = await client.query(`
      SELECT access_token FROM allegro_connections WHERE is_active = true LIMIT 1
    `);
    const accessToken = tokenResult.rows[0]?.access_token;

    if (!accessToken) {
      console.error('❌ Brak access token');
      return;
    }

    const orderId = 'ee8f0010-b1c8-11f0-831f-cf35cc48a3c0';

    console.log(`📡 Pobieranie zamówienia ${orderId} z Allegro API...`);
    
    const response = await axios.get(
      `https://api.allegro.pl/order/checkout-forms/${orderId}`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        }
      }
    );

    const data = response.data;
    
    console.log(`✅ Pobrano dane zamówienia`);
    console.log(`   Produkty: ${data.lineItems?.length || 0}`);

    await client.query('BEGIN');

    // Pobierz order_date z allegro.orders
    const allegroOrderResult = await client.query(
      'SELECT order_date FROM allegro.orders WHERE order_id = $1',
      [orderId]
    );
    
    let orderDate = new Date();
    if (allegroOrderResult.rows.length > 0) {
      orderDate = new Date(allegroOrderResult.rows[0].order_date);
    }

    // Wstaw zamówienie
    const orderResult = await client.query(`
      INSERT INTO commerce.orders (
        source, source_order_id, order_date, status,
        buyer_login, buyer_email, buyer_first_name, buyer_last_name,
        buyer_phone, buyer_address, buyer_zip, buyer_city, buyer_country_code,
        payment_status, payment_type, payment_provider, payment_amount, payment_currency,
        delivery_method, delivery_amount, delivery_currency,
        total_to_pay_amount, total_to_pay_currency,
        raw_payload, order_number, created_at, updated_at
      ) VALUES (
        'ALLEGRO', $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17,
        $18, $19, $20, $21, $22, $23, '00676', NOW(), NOW()
      )
      RETURNING id
    `, [
      orderId,
      orderDate,
      data.status || 'READY_FOR_PROCESSING',
      data.buyer?.login,
      data.buyer?.email,
      data.buyer?.firstName,
      data.buyer?.lastName,
      data.buyer?.phoneNumber,
      data.delivery?.address?.street ? JSON.stringify(data.delivery.address) : null,
      data.delivery?.address?.zipCode,
      data.delivery?.address?.city,
      data.delivery?.address?.countryCode || 'PL',
      data.payment?.paidAmount ? 'PAID' : 'NOT_PAID',
      data.payment?.type,
      data.payment?.provider,
      parseFloat(data.payment?.paidAmount?.amount || data.summary?.totalToPay?.amount || '0'),
      data.payment?.paidAmount?.currency || 'PLN',
      data.delivery?.method?.name,
      parseFloat(data.delivery?.cost?.amount || '0'),
      data.delivery?.cost?.currency || 'PLN',
      parseFloat(data.summary?.totalToPay?.amount || '0'),
      data.summary?.totalToPay?.currency || 'PLN',
      JSON.stringify(data)
    ]);

    const commerceOrderId = orderResult.rows[0].id;
    console.log(`✅ Dodano zamówienie commerce.orders.id=${commerceOrderId}`);

    // Dodaj produkty
    for (const item of data.lineItems || []) {
      const externalId = item.offer?.external?.id;
      const imageUrl = item.imageUrl;

      await client.query(`
        INSERT INTO commerce.order_items (
          order_id, offer_external_id, name, quantity, unit_price, price, image_url, raw_data
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
      `, [
        commerceOrderId,
        externalId,
        item.offer?.name,
        item.quantity || 1,
        parseFloat(item.price?.amount || '0'),
        parseFloat(item.price?.amount || '0') * (item.quantity || 1),
        imageUrl,
        JSON.stringify(item)
      ]);
    }

    console.log(`✅ Dodano ${data.lineItems?.length || 0} produktów`);

    // Dodaj audit log
    await client.query(`
      INSERT INTO commerce.order_number_audit (
        order_id, order_number, source, source_order_id, assigned_at, assigned_by, notes
      ) VALUES ($1, '00676', 'ALLEGRO', $2, NOW(), 'MANUAL_FIX', 'Ręczne przywrócenie zamówienia po błędzie')
    `, [commerceOrderId, orderId]);

    await client.query('COMMIT');
    console.log('\n✅ Zamówienie #00676 zostało przywrócone');

  } catch (error: any) {
    await client.query('ROLLBACK');
    console.error('❌ Błąd:', error.message);
  } finally {
    client.release();
    await pool.end();
  }
}

manualInsert();
