/**
 * Skrypt naprawczy - nadaje numery zamówieniom które nie mają order_number
 * Uruchom: npx tsx server/fix-missing-order-numbers.ts
 */

import { pool } from './postgres';
import { PoolClient } from 'pg';

async function fixMissingOrderNumbers() {
  const client = await pool.connect();
  
  try {
    await client.query('BEGIN');
    
    // Pobierz wszystkie zamówienia bez numerów, posortowane po dacie utworzenia
    const missingNumbersResult = await client.query(`
      SELECT id, source, source_order_id, created_at
      FROM commerce.orders
      WHERE order_number IS NULL
      ORDER BY created_at ASC
    `);
    
    console.log(`🔍 Znaleziono ${missingNumbersResult.rows.length} zamówień bez numerów`);
    
    if (missingNumbersResult.rows.length === 0) {
      console.log('✅ Wszystkie zamówienia mają numery - nic do naprawy');
      await client.query('COMMIT');
      return;
    }
    
    // Dla każdego zamówienia nadaj kolejny numer z sekwencji
    for (const order of missingNumbersResult.rows) {
      // Pobierz i zablokuj sekwencję
      const sequenceResult = await client.query(`
        SELECT current_value 
        FROM commerce.order_sequences 
        WHERE sequence_name = 'main_order_sequence'
        FOR UPDATE
      `);
      
      const currentValue = sequenceResult.rows[0].current_value;
      const nextValue = currentValue + 1;
      const nextOrderNumber = String(nextValue).padStart(5, '0');
      
      // Zaktualizuj sekwencję
      await client.query(`
        UPDATE commerce.order_sequences 
        SET current_value = $1, last_updated = NOW()
        WHERE sequence_name = 'main_order_sequence'
      `, [nextValue]);
      
      // Nadaj numer zamówieniu
      await client.query(`
        UPDATE commerce.orders 
        SET order_number = $1, updated_at = NOW()
        WHERE id = $2
      `, [nextOrderNumber, order.id]);
      
      // Zapisz w audyt logu
      await client.query(`
        INSERT INTO commerce.order_number_audit (
          order_id, order_number, source, source_order_id, assigned_at, assigned_by, notes
        ) VALUES ($1, $2, $3, $4, NOW(), 'FIX_SCRIPT', $5)
      `, [
        order.id,
        nextOrderNumber,
        order.source,
        order.source_order_id,
        `Numer nadany przez skrypt naprawczy dla zamówienia które nie miało numeru (created_at: ${order.created_at})`
      ]);
      
      console.log(`✅ Nadano numer #${nextOrderNumber} dla zamówienia ${order.source}-${order.source_order_id} (id=${order.id})`);
    }
    
    await client.query('COMMIT');
    console.log(`🎉 Naprawiono ${missingNumbersResult.rows.length} zamówień!`);
    
  } catch (error) {
    await client.query('ROLLBACK');
    console.error('❌ Błąd podczas naprawy numerów:', error);
    throw error;
  } finally {
    client.release();
  }
}

// Uruchom skrypt
fixMissingOrderNumbers()
  .then(async () => {
    console.log('✅ Skrypt zakończony pomyślnie');
    await pool.end();
    process.exit(0);
  })
  .catch(async (error) => {
    console.error('❌ Skrypt zakończony błędem:', error);
    await pool.end();
    process.exit(1);
  });
