import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { ArrowLeft, Edit, Trash2, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import type { ProductionWorkCenter, ProductionLocation } from "@shared/schema";

export default function ProductionWorkCenterDetail() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const { data: workCenter, isLoading } = useQuery<ProductionWorkCenter>({
    queryKey: ["/api/production/work-centers", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/work-centers/${id}`);
      if (!res.ok) throw new Error("Failed to fetch work center");
      return res.json();
    },
  });

  const { data: locations = [] } = useQuery<ProductionLocation[]>({
    queryKey: ["/api/production/locations"],
  });

  const deleteMutation = useMutation({
    mutationFn: async () => {
      const res = await fetch(`/api/production/work-centers/${id}`, { method: "DELETE" });
      if (!res.ok) throw new Error("Failed to delete");
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/work-centers"] });
      toast({ title: "Gniazdo robocze zostało usunięte" });
      navigate("/production/work-centers");
    },
  });

  const getLocationName = (locationId: number | null) => {
    if (!locationId) return "-";
    return locations.find(l => l.id === locationId)?.name || "Nieznana lokalizacja";
  };

  const getStatusBadge = (status: string) => {
    const variants = {
      operational: { variant: "default" as const, label: "Operacyjne" },
      maintenance: { variant: "secondary" as const, label: "Konserwacja" },
      offline: { variant: "outline" as const, label: "Offline" },
    };
    const config = variants[status as keyof typeof variants] || variants.operational;
    return <Badge variant={config.variant}>{config.label}</Badge>;
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="flex justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-workcenter-detail" />
        </div>
      </div>
    );
  }

  if (!workCenter) {
    return (
      <div className="container mx-auto p-6">
        <Card>
          <CardContent className="p-8 text-center text-muted-foreground">
            Nie znaleziono gniazda roboczego
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => navigate("/production/work-centers")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div>
            <h1 className="text-3xl font-bold" data-testid="text-workcenter-name">
              {workCenter.name}
            </h1>
            <p className="text-muted-foreground font-mono" data-testid="text-workcenter-code">
              {workCenter.code}
            </p>
          </div>
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            onClick={() => navigate(`/production/work-centers/${id}/edit`)}
            data-testid="button-edit"
          >
            <Edit className="h-4 w-4 mr-2" />
            Edytuj
          </Button>
          <Button
            variant="outline"
            onClick={() => {
              if (confirm("Czy na pewno chcesz usunąć to gniazdo robocze?")) {
                deleteMutation.mutate();
              }
            }}
            disabled={deleteMutation.isPending}
            data-testid="button-delete"
          >
            <Trash2 className="h-4 w-4 mr-2" />
            Usuń
          </Button>
        </div>
      </div>

      <div className="grid gap-6 md:grid-cols-2">
        <Card>
          <CardHeader>
            <CardTitle>Podstawowe Informacje</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <p className="text-sm text-muted-foreground">Kod</p>
              <p className="font-mono" data-testid="detail-code">{workCenter.code}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Nazwa</p>
              <p data-testid="detail-name">{workCenter.name}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Lokalizacja</p>
              <p data-testid="detail-location">{getLocationName(workCenter.locationId)}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Status</p>
              <div data-testid="detail-status">{getStatusBadge(workCenter.status)}</div>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Aktywne</p>
              <Badge variant={workCenter.isActive ? "default" : "secondary"} data-testid="detail-active">
                {workCenter.isActive ? "Tak" : "Nie"}
              </Badge>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Wydajność i OEE</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <p className="text-sm text-muted-foreground">Przepustowość</p>
              <p data-testid="detail-throughput">
                {workCenter.throughputPerHour ? `${workCenter.throughputPerHour} ${workCenter.throughputUnit || ''}/h` : '-'}
              </p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">OEE Target</p>
              <p data-testid="detail-oee-target">
                {workCenter.oeeTarget ? `${workCenter.oeeTarget}%` : '-'}
              </p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Bieżące OEE</p>
              <p data-testid="detail-oee-current">
                {workCenter.currentOee ? `${workCenter.currentOee}%` : '-'}
              </p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Koszt operacyjny</p>
              <p data-testid="detail-cost">
                {workCenter.operatingCostPerHour ? `${workCenter.operatingCostPerHour} PLN/h` : '-'}
              </p>
            </div>
          </CardContent>
        </Card>

        {workCenter.description && (
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle>Opis</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground" data-testid="detail-description">
                {workCenter.description}
              </p>
            </CardContent>
          </Card>
        )}

        {workCenter.capabilities && workCenter.capabilities.length > 0 && (
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle>Możliwości</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="flex flex-wrap gap-2" data-testid="detail-capabilities">
                {workCenter.capabilities.map((cap, idx) => (
                  <Badge key={idx} variant="secondary">{cap}</Badge>
                ))}
              </div>
            </CardContent>
          </Card>
        )}

        {workCenter.notes && (
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle>Notatki</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground whitespace-pre-wrap" data-testid="detail-notes">
                {workCenter.notes}
              </p>
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
}
