import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Search, FileText, Calendar, CheckCircle, Clock } from "lucide-react";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { formatDistanceToNow } from "date-fns";
import { pl } from "date-fns/locale";

export default function BomListPage() {
  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");

  const { data: bomsData, isLoading } = useQuery({
    queryKey: ['/api/boms', { 
      search: searchTerm || undefined,
      status: statusFilter !== "all" ? statusFilter : undefined,
      limit: 50,
      offset: 0
    }],
  });

  const boms = (bomsData as any)?.boms || [];
  const total = (bomsData as any)?.total || 0;

  return (
    <div className="flex flex-col gap-6 p-6">
      <div className="flex items-center justify-between">
        <div className="flex flex-col gap-2">
          <h1 className="text-3xl font-bold" data-testid="text-page-title">
            Lista BOM
          </h1>
          <p className="text-muted-foreground">
            Bill of Materials - zestawienia komponentów dla produktów
          </p>
        </div>
        <Badge variant="outline" data-testid="badge-total-boms">
          {total} BOM-ów
        </Badge>
      </div>

      <Card>
        <CardHeader>
          <div className="flex items-center gap-4">
            <div className="relative flex-1 max-w-sm">
              <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Szukaj po nazwie produktu, SKU..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-8"
                data-testid="input-search-boms"
              />
            </div>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-[200px]" data-testid="select-status-filter">
                <SelectValue placeholder="Status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all" data-testid="option-all-status">
                  Wszystkie statusy
                </SelectItem>
                <SelectItem value="approved">Zatwierdzone</SelectItem>
                <SelectItem value="pending">Oczekujące</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center py-8 text-muted-foreground">
              Ładowanie BOM-ów...
            </div>
          ) : boms.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">
              Brak BOM-ów. Wygeneruj BOM dla produktu w edytorze produktu.
            </div>
          ) : (
            <div className="rounded-md border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Produkt</TableHead>
                    <TableHead>SKU</TableHead>
                    <TableHead>Typ</TableHead>
                    <TableHead>Komponenty</TableHead>
                    <TableHead>Wersja</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Synchronizacja</TableHead>
                    <TableHead>Ostatnia aktualizacja</TableHead>
                    <TableHead>Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {boms.map((bom: any) => (
                    <TableRow key={bom.id} data-testid={`row-bom-${bom.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex flex-col">
                          <span>{bom.product_title || bom.name}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline">{bom.sku}</Badge>
                      </TableCell>
                      <TableCell>
                        <Badge variant="secondary">{bom.product_type || "-"}</Badge>
                      </TableCell>
                      <TableCell>
                        <Badge>{bom.components_count} szt.</Badge>
                      </TableCell>
                      <TableCell className="text-sm">{bom.version || "1.0"}</TableCell>
                      <TableCell>
                        {bom.is_approved ? (
                          <Badge variant="default" className="gap-1">
                            <CheckCircle className="h-3 w-3" />
                            Zatwierdzone
                          </Badge>
                        ) : (
                          <Badge variant="secondary" className="gap-1">
                            <Clock className="h-3 w-3" />
                            Oczekujące
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        {bom.odoo_bom_id ? (
                          <Badge variant="outline" className="gap-1">
                            <CheckCircle className="h-3 w-3" />
                            Odoo: {bom.odoo_bom_id}
                          </Badge>
                        ) : (
                          <Badge variant="secondary">
                            Nie zsynchronizowane
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell className="text-sm text-muted-foreground">
                        {bom.updated_at
                          ? formatDistanceToNow(new Date(bom.updated_at), {
                              addSuffix: true,
                              locale: pl,
                            })
                          : "-"}
                      </TableCell>
                      <TableCell>
                        <Link href={`/bom/${bom.id}`}>
                          <Button variant="ghost" size="sm" data-testid={`button-view-bom-${bom.id}`}>
                            <FileText className="h-4 w-4 mr-1" />
                            Szczegóły
                          </Button>
                        </Link>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
