import { UseFormReturn } from "react-hook-form";
import { Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { z } from "zod";

export const orderFormSchema = z.object({
  orderNumber: z.string().optional(),
  productId: z.coerce.number().positive("Wybierz produkt"),
  bomId: z.coerce.number().optional().nullable(),
  routingId: z.coerce.number().optional().nullable(),
  status: z.enum(['draft', 'confirmed', 'planned', 'in_progress', 'paused', 'done', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  quantityPlanned: z.coerce.number().positive("Ilość musi być większa od 0"),
  unitOfMeasure: z.string().optional(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  responsibleUserId: z.coerce.number().optional().nullable(),
  sourceOrderNumber: z.string().optional().nullable(),
  locationId: z.coerce.number().optional().nullable(),
  notes: z.string().optional().nullable(),
});

export type OrderFormValues = z.infer<typeof orderFormSchema>;

interface ProductionOrderFormProps {
  form: UseFormReturn<OrderFormValues>;
  onSubmit: (data: OrderFormValues) => void | Promise<void>;
  onCancel: () => void;
  isLoading: boolean;
  mode: 'create' | 'edit';
  showFooter?: boolean;
}

export function ProductionOrderForm({
  form,
  onSubmit,
  onCancel,
  isLoading,
  mode,
  showFooter = true,
}: ProductionOrderFormProps) {
  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <div className="grid grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="orderNumber"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Numer Zlecenia (opcjonalnie)</FormLabel>
                <FormControl>
                  <Input {...field} placeholder="Auto-generowany jeśli pusty" data-testid="input-order-number" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="productId"
            render={({ field }) => (
              <FormItem>
                <FormLabel>ID Produktu *</FormLabel>
                <FormControl>
                  <Input type="number" {...field} placeholder="ID produktu" data-testid="input-product-id" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <div className="grid grid-cols-3 gap-4">
          <FormField
            control={form.control}
            name="quantityPlanned"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Ilość Planowana *</FormLabel>
                <FormControl>
                  <Input type="number" {...field} min="1" data-testid="input-quantity" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="unitOfMeasure"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Jednostka</FormLabel>
                <FormControl>
                  <Input {...field} placeholder="szt" data-testid="input-unit" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="priority"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Priorytet</FormLabel>
                <Select onValueChange={field.onChange} value={field.value}>
                  <FormControl>
                    <SelectTrigger data-testid="select-priority">
                      <SelectValue placeholder="Wybierz priorytet" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    <SelectItem value="low">Niski</SelectItem>
                    <SelectItem value="normal">Normalny</SelectItem>
                    <SelectItem value="high">Wysoki</SelectItem>
                    <SelectItem value="urgent">Pilny</SelectItem>
                  </SelectContent>
                </Select>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <div className="grid grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="status"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Status</FormLabel>
                <Select onValueChange={field.onChange} value={field.value}>
                  <FormControl>
                    <SelectTrigger data-testid="select-status">
                      <SelectValue placeholder="Wybierz status" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    <SelectItem value="draft">Projekt</SelectItem>
                    <SelectItem value="confirmed">Potwierdzone</SelectItem>
                    <SelectItem value="planned">Zaplanowane</SelectItem>
                    <SelectItem value="in_progress">W realizacji</SelectItem>
                    <SelectItem value="paused">Wstrzymane</SelectItem>
                    <SelectItem value="done">Zakończone</SelectItem>
                    <SelectItem value="cancelled">Anulowane</SelectItem>
                  </SelectContent>
                </Select>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="sourceOrderNumber"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Źródłowe Zamówienie</FormLabel>
                <FormControl>
                  <Input {...field} value={field.value ?? ""} placeholder="np. ORD-2024-001" data-testid="input-source-order" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <div className="grid grid-cols-3 gap-4">
          <FormField
            control={form.control}
            name="bomId"
            render={({ field }) => (
              <FormItem>
                <FormLabel>ID BOM</FormLabel>
                <FormControl>
                  <Input type="number" {...field} value={field.value ?? ""} placeholder="ID" data-testid="input-bom-id" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="routingId"
            render={({ field }) => (
              <FormItem>
                <FormLabel>ID Marszruty</FormLabel>
                <FormControl>
                  <Input type="number" {...field} value={field.value ?? ""} placeholder="ID" data-testid="input-routing-id" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="locationId"
            render={({ field }) => (
              <FormItem>
                <FormLabel>ID Lokalizacji</FormLabel>
                <FormControl>
                  <Input type="number" {...field} value={field.value ?? ""} placeholder="ID" data-testid="input-location-id" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <div className="grid grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="plannedStartDate"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Planowany Start</FormLabel>
                <FormControl>
                  <Input type="date" {...field} value={field.value ?? ""} data-testid="input-start-date" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="plannedEndDate"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Planowany Koniec</FormLabel>
                <FormControl>
                  <Input type="date" {...field} value={field.value ?? ""} data-testid="input-end-date" />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <FormField
          control={form.control}
          name="responsibleUserId"
          render={({ field }) => (
            <FormItem>
              <FormLabel>ID Odpowiedzialnego Użytkownika</FormLabel>
              <FormControl>
                <Input type="number" {...field} value={field.value ?? ""} placeholder="ID użytkownika" data-testid="input-responsible-user-id" />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="notes"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Uwagi</FormLabel>
              <FormControl>
                <Textarea {...field} value={field.value ?? ""} placeholder="Dodatkowe informacje..." rows={3} data-testid="input-notes" />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        {showFooter && (
          <div className="flex justify-end gap-2 pt-4">
            <Button
              type="button"
              variant="outline"
              onClick={onCancel}
              disabled={isLoading}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
            <Button type="submit" disabled={isLoading} data-testid="button-submit">
              {isLoading ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Zapisywanie...
                </>
              ) : mode === 'edit' ? (
                "Zaktualizuj"
              ) : (
                "Utwórz"
              )}
            </Button>
          </div>
        )}
      </form>
    </Form>
  );
}
