import type { Express } from "express";
import { z } from "zod";
import { pool } from "../../postgres";
import { materialMovementsService } from "../../services/production/material-movements";
import { insertProductionMaterialMovementSchema } from "@shared/schema";
import { requirePermission } from "../../auth";

const partialMovementSchema = insertProductionMaterialMovementSchema.partial();

// Filters schema for query parameters
const filtersSchema = z.object({
  productionOrderId: z.coerce.number().optional(),
  carrierId: z.coerce.number().optional(),
  operatorId: z.coerce.number().optional(),
  materialType: z.string().optional(),
  materialCode: z.string().optional(),
  status: z.string().optional(),
  sourceLocationId: z.coerce.number().optional(),
  targetLocationId: z.coerce.number().optional(),
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  movementBatchId: z.string().optional(),
  limit: z.coerce.number().optional(),
  offset: z.coerce.number().optional(),
});

export function registerMaterialMovementRoutes(app: Express) {
  // Get all movements with filtering
  app.get("/api/production/material-movements", requirePermission("view_production"), async (req, res) => {
    try {
      const filters = filtersSchema.parse(req.query);
      
      // Convert string dates to Date objects
      const processedFilters = {
        ...filters,
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
      };

      const movements = await materialMovementsService.getAllMovements(pool, processedFilters);
      res.json(movements);
    } catch (error: any) {
      console.error("Error fetching material movements:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to fetch material movements" });
    }
  });

  // Get single movement by ID
  app.get("/api/production/material-movements/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const movement = await materialMovementsService.getMovementById(pool, id);
      
      if (!movement) {
        return res.status(404).json({ error: "Material movement not found" });
      }
      
      res.json(movement);
    } catch (error: any) {
      console.error("Error fetching material movement:", error);
      res.status(500).json({ error: "Failed to fetch material movement" });
    }
  });

  // Create new movement
  app.post("/api/production/material-movements", requirePermission("manage_production"), async (req, res) => {
    try {
      const validated = insertProductionMaterialMovementSchema.parse(req.body);
      const movement = await materialMovementsService.createMovement(pool, validated);
      res.status(201).json(movement);
    } catch (error: any) {
      console.error("Error creating material movement:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create material movement" });
    }
  });

  // Batch create movements (for scanning multiple items)
  app.post("/api/production/material-movements/batch", requirePermission("manage_production"), async (req, res) => {
    try {
      const { movements } = req.body;
      
      if (!Array.isArray(movements)) {
        return res.status(400).json({ error: "Expected array of movements" });
      }

      const validated = movements.map(m => insertProductionMaterialMovementSchema.parse(m));
      const result = await materialMovementsService.createBatchMovements(pool, validated);
      res.status(201).json(result);
    } catch (error: any) {
      console.error("Error creating batch movements:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create batch movements" });
    }
  });

  // Update movement
  app.patch("/api/production/material-movements/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const validated = partialMovementSchema.parse(req.body);
      const movement = await materialMovementsService.updateMovement(pool, id, validated);
      res.json(movement);
    } catch (error: any) {
      console.error("Error updating material movement:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to update material movement" });
    }
  });

  // Update movement status (quick endpoint for status changes)
  app.patch("/api/production/material-movements/:id/status", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const { status } = req.body;
      
      if (!status || typeof status !== 'string') {
        return res.status(400).json({ error: "Status is required" });
      }

      const movement = await materialMovementsService.updateMovementStatus(pool, id, status);
      res.json(movement);
    } catch (error: any) {
      console.error("Error updating movement status:", error);
      res.status(500).json({ error: "Failed to update movement status" });
    }
  });

  // Delete movement
  app.delete("/api/production/material-movements/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      await materialMovementsService.deleteMovement(pool, id);
      res.status(204).send();
    } catch (error: any) {
      console.error("Error deleting material movement:", error);
      res.status(500).json({ error: "Failed to delete material movement" });
    }
  });

  // Get movement flow for a production order (chronological)
  app.get("/api/production/orders/:orderId/material-flow", requirePermission("view_production"), async (req, res) => {
    try {
      const orderId = parseInt(req.params.orderId);
      const flow = await materialMovementsService.getProductionOrderFlow(pool, orderId);
      res.json(flow);
    } catch (error: any) {
      console.error("Error fetching production order flow:", error);
      res.status(500).json({ error: "Failed to fetch production order flow" });
    }
  });

  // Get carrier history (track carrier movements)
  app.get("/api/production/carriers/:carrierId/history", requirePermission("view_production"), async (req, res) => {
    try {
      const carrierId = parseInt(req.params.carrierId);
      const history = await materialMovementsService.getCarrierHistory(pool, carrierId);
      res.json(history);
    } catch (error: any) {
      console.error("Error fetching carrier history:", error);
      res.status(500).json({ error: "Failed to fetch carrier history" });
    }
  });

  // Get location inventory (real-time inventory in location)
  app.get("/api/production/locations/:locationId/inventory", requirePermission("view_production"), async (req, res) => {
    try {
      const locationId = parseInt(req.params.locationId);
      const inventory = await materialMovementsService.getLocationInventory(pool, locationId);
      res.json(inventory);
    } catch (error: any) {
      console.error("Error fetching location inventory:", error);
      res.status(500).json({ error: "Failed to fetch location inventory" });
    }
  });
}
