import { Request, Response, NextFunction } from 'express';
import { pool } from './postgres';
import crypto from 'crypto';

// Hash token using SHA256
export function hashToken(token: string): string {
  return crypto.createHash('sha256').update(token).digest('hex');
}

// Generate random API token (64 characters)
export function generateApiToken(): string {
  return crypto.randomBytes(32).toString('hex');
}

// Middleware to validate Bearer token for external API
export async function requireApiToken(req: Request, res: Response, next: NextFunction) {
  try {
    const authHeader = req.headers.authorization;
    
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return res.status(401).json({ 
        error: 'Unauthorized', 
        message: 'Missing or invalid Authorization header' 
      });
    }

    const token = authHeader.substring(7); // Remove 'Bearer ' prefix
    const hashedToken = hashToken(token);

    // Find token in database
    const result = await pool.query(
      'SELECT * FROM api_tokens WHERE token = $1 LIMIT 1',
      [hashedToken]
    );

    if (result.rows.length === 0) {
      return res.status(401).json({ 
        error: 'Unauthorized', 
        message: 'Invalid API token' 
      });
    }

    const apiToken = result.rows[0];

    // Check if token is active
    if (!apiToken.is_active) {
      return res.status(401).json({ 
        error: 'Unauthorized', 
        message: 'API token is inactive' 
      });
    }

    // Check if token is expired
    if (apiToken.expires_at && new Date(apiToken.expires_at) < new Date()) {
      return res.status(401).json({ 
        error: 'Unauthorized', 
        message: 'API token has expired' 
      });
    }

    // Update last used timestamp
    await pool.query(
      'UPDATE api_tokens SET last_used_at = $1 WHERE id = $2',
      [new Date(), apiToken.id]
    );

    // Store token info in request for logging
    (req as any).apiToken = apiToken;

    next();
  } catch (error) {
    console.error('API token validation error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
}

// Middleware to log API requests
export async function logApiRequest(req: Request, res: Response, next: NextFunction) {
  const startTime = Date.now();
  const apiToken = (req as any).apiToken;

  // Log response after it's sent
  res.on('finish', async () => {
    try {
      const responseTime = Date.now() - startTime;
      const ipAddress = req.ip || req.socket.remoteAddress || null;
      const userAgent = req.headers['user-agent'] || null;

      await pool.query(
        `INSERT INTO api_request_logs (token_id, method, path, status_code, response_time, ip_address, user_agent, created_at)
         VALUES ($1, $2, $3, $4, $5, $6, $7, $8)`,
        [
          apiToken?.id || null,
          req.method,
          req.path,
          res.statusCode,
          responseTime,
          ipAddress,
          userAgent,
          new Date()
        ]
      );
    } catch (error) {
      console.error('Failed to log API request:', error);
    }
  });

  next();
}
