import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { QrCode, MapPin, Package2, User, CheckCircle2, XCircle, Loader2 } from "lucide-react";
import type { ProductionLocation, ProductionCarrier } from "@shared/schema";

interface ScanEntry {
  id: string;
  barcode: string;
  timestamp: Date;
  status: 'success' | 'error';
  message?: string;
}

export default function ProductionScanPage() {
  const { toast } = useToast();
  const [barcode, setBarcode] = useState("");
  const [targetLocationId, setTargetLocationId] = useState<string>("");
  const [carrierId, setCarrierId] = useState<string>("");
  const [operatorName, setOperatorName] = useState("");
  const [scanHistory, setScanHistory] = useState<ScanEntry[]>([]);

  // Fetch locations
  const { data: locations = [] } = useQuery<ProductionLocation[]>({
    queryKey: ["/api/production/locations"],
  });

  // Fetch carriers
  const { data: carriers = [] } = useQuery<ProductionCarrier[]>({
    queryKey: ["/api/production/carriers"],
  });

  // Active locations only
  const activeLocations = locations.filter(loc => loc.isActive);
  const activeCarriers = carriers.filter(car => car.isActive);

  // Create movement mutation
  const createMovementMutation = useMutation({
    mutationFn: (data: any) => apiRequest("/api/production/material-movements", "POST", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/material-movements"] });
    },
  });

  const handleScan = async () => {
    if (!barcode.trim()) {
      toast({
        title: "Błąd",
        description: "Wprowadź kod kreskowy",
        variant: "destructive",
      });
      return;
    }

    if (!targetLocationId) {
      toast({
        title: "Błąd",
        description: "Wybierz lokalizację docelową",
        variant: "destructive",
      });
      return;
    }

    const entryId = `scan-${Date.now()}`;
    const timestamp = new Date();

    try {
      // Create movement record
      await createMovementMutation.mutateAsync({
        materialType: "formatka",
        materialCode: barcode,
        materialName: `Materiał ${barcode}`,
        quantity: 1,
        quantityRejected: 0,
        unit: "szt",
        targetLocationId: parseInt(targetLocationId),
        carrierId: carrierId ? parseInt(carrierId) : null,
        scannedBarcode: barcode,
        scanMethod: "manual",
        status: "completed",
        movementType: "transfer",
        movementDate: timestamp,
        notes: operatorName ? `Operator: ${operatorName}` : undefined,
      });

      // Add success entry to history
      setScanHistory(prev => [
        {
          id: entryId,
          barcode,
          timestamp,
          status: 'success',
        },
        ...prev.slice(0, 19), // Keep last 20 entries
      ]);

      toast({
        title: "Zarejestrowano",
        description: `Kod ${barcode} zapisany`,
      });

      // Clear barcode for next scan
      setBarcode("");
    } catch (error: any) {
      // Add error entry to history
      setScanHistory(prev => [
        {
          id: entryId,
          barcode,
          timestamp,
          status: 'error',
          message: error.message || "Błąd zapisu",
        },
        ...prev.slice(0, 19),
      ]);

      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zarejestrować ruchu",
        variant: "destructive",
      });
    }
  };

  const handleBarcodeKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      handleScan();
    }
  };

  const getLocationName = (id: string) => {
    const location = locations.find(loc => loc.id === parseInt(id));
    return location ? location.name : '-';
  };

  const getCarrierName = (id: string) => {
    const carrier = carriers.find(car => car.id === parseInt(id));
    return carrier ? carrier.name : '-';
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold flex items-center gap-2" data-testid="text-page-title">
            <QrCode className="h-8 w-8" />
            Skanowanie Nośników
          </h1>
          <p className="text-muted-foreground mt-1">
            Szybkie rejestrowanie ruchów materiałów
          </p>
        </div>
        <div className="text-right">
          <div className="text-sm text-muted-foreground">Zeskanowano dzisiaj</div>
          <div className="text-2xl font-bold" data-testid="text-scan-count">
            {scanHistory.filter(s => s.status === 'success').length}
          </div>
        </div>
      </div>

      <div className="grid gap-6 md:grid-cols-2">
        {/* Scan Form */}
        <Card>
          <CardHeader>
            <CardTitle>Skanowanie</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="barcode" className="flex items-center gap-2">
                <QrCode className="h-4 w-4" />
                Kod kreskowy
              </Label>
              <Input
                id="barcode"
                value={barcode}
                onChange={(e) => setBarcode(e.target.value)}
                onKeyPress={handleBarcodeKeyPress}
                placeholder="Zeskanuj lub wpisz kod"
                autoFocus
                data-testid="input-barcode"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="targetLocation" className="flex items-center gap-2">
                <MapPin className="h-4 w-4" />
                Lokalizacja docelowa *
              </Label>
              <Select value={targetLocationId} onValueChange={setTargetLocationId}>
                <SelectTrigger id="targetLocation" data-testid="select-target-location">
                  <SelectValue placeholder="Wybierz lokalizację" />
                </SelectTrigger>
                <SelectContent>
                  {activeLocations.map((location) => (
                    <SelectItem key={location.id} value={location.id.toString()}>
                      {location.code} - {location.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="carrier" className="flex items-center gap-2">
                <Package2 className="h-4 w-4" />
                Nośnik (opcjonalnie)
              </Label>
              <Select value={carrierId} onValueChange={(value) => setCarrierId(value === "none" ? "" : value)}>
                <SelectTrigger id="carrier" data-testid="select-carrier">
                  <SelectValue placeholder="Wybierz nośnik" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="none">Brak</SelectItem>
                  {activeCarriers.map((carrier) => (
                    <SelectItem key={carrier.id} value={carrier.id.toString()}>
                      {carrier.code} - {carrier.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="operator" className="flex items-center gap-2">
                <User className="h-4 w-4" />
                Operator (opcjonalnie)
              </Label>
              <Input
                id="operator"
                value={operatorName}
                onChange={(e) => setOperatorName(e.target.value)}
                placeholder="Imię i nazwisko"
                data-testid="input-operator"
              />
            </div>

            <Separator />

            <Button
              onClick={handleScan}
              disabled={createMovementMutation.isPending || !barcode || !targetLocationId}
              className="w-full"
              data-testid="button-scan"
            >
              {createMovementMutation.isPending ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Zapisywanie...
                </>
              ) : (
                <>
                  <QrCode className="h-4 w-4 mr-2" />
                  Zarejestruj (Enter)
                </>
              )}
            </Button>

            {/* Current context summary */}
            {(targetLocationId || carrierId || operatorName) && (
              <Card className="bg-muted/50">
                <CardContent className="p-4 space-y-2 text-sm">
                  <div className="font-semibold">Aktywny kontekst:</div>
                  {targetLocationId && (
                    <div className="flex items-center gap-2">
                      <MapPin className="h-3 w-3" />
                      <span className="text-muted-foreground">Lokalizacja:</span>
                      <span className="font-mono">{getLocationName(targetLocationId)}</span>
                    </div>
                  )}
                  {carrierId && (
                    <div className="flex items-center gap-2">
                      <Package2 className="h-3 w-3" />
                      <span className="text-muted-foreground">Nośnik:</span>
                      <span className="font-mono">{getCarrierName(carrierId)}</span>
                    </div>
                  )}
                  {operatorName && (
                    <div className="flex items-center gap-2">
                      <User className="h-3 w-3" />
                      <span className="text-muted-foreground">Operator:</span>
                      <span>{operatorName}</span>
                    </div>
                  )}
                </CardContent>
              </Card>
            )}
          </CardContent>
        </Card>

        {/* Scan History */}
        <Card>
          <CardHeader>
            <CardTitle>Historia skanowania</CardTitle>
          </CardHeader>
          <CardContent>
            {scanHistory.length === 0 ? (
              <div className="text-center text-muted-foreground py-8">
                Brak zeskanowanych kodów
              </div>
            ) : (
              <div className="space-y-2 max-h-[500px] overflow-y-auto">
                {scanHistory.map((entry) => (
                  <div
                    key={entry.id}
                    className="flex items-center justify-between p-3 rounded-lg border hover-elevate"
                    data-testid={`scan-entry-${entry.id}`}
                  >
                    <div className="flex items-center gap-3 flex-1">
                      {entry.status === 'success' ? (
                        <CheckCircle2 className="h-5 w-5 text-green-500 flex-shrink-0" />
                      ) : (
                        <XCircle className="h-5 w-5 text-destructive flex-shrink-0" />
                      )}
                      <div className="flex-1 min-w-0">
                        <div className="font-mono font-semibold truncate">
                          {entry.barcode}
                        </div>
                        {entry.message && (
                          <div className="text-sm text-destructive truncate">
                            {entry.message}
                          </div>
                        )}
                        <div className="text-xs text-muted-foreground">
                          {entry.timestamp.toLocaleTimeString('pl-PL')}
                        </div>
                      </div>
                    </div>
                    <Badge variant={entry.status === 'success' ? 'default' : 'destructive'}>
                      {entry.status === 'success' ? 'OK' : 'Błąd'}
                    </Badge>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
