import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Link } from "wouter";
import { Plus, Edit, Trash2, Loader2, Clock, ListOrdered, ArrowUp, ArrowDown } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { insertProductionRoutingSchema, insertProductionRoutingOperationSchema } from "@shared/schema";
import type { ProductionRouting, ProductionRoutingOperation, ProductionWorkCenter } from "@shared/schema";
import { z } from "zod";

// Extend shared schemas with frontend validation
const routingFormSchema = insertProductionRoutingSchema.extend({
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  defaultWorkCenterId: z.coerce.number().optional().nullable(),
});

const operationFormSchema = insertProductionRoutingOperationSchema.extend({
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  estimatedTime: z.coerce.number().min(0, "Czas musi być większy lub równy 0"),
  setupTime: z.coerce.number().min(0, "Czas przygotowania musi być większy lub równy 0").optional().nullable(),
  workCenterId: z.coerce.number().optional().nullable(),
}).omit({ routingId: true });

type RoutingFormData = z.infer<typeof routingFormSchema>;
type OperationFormData = z.infer<typeof operationFormSchema>;

export default function ProductionRoutings() {
  const [isRoutingDialogOpen, setIsRoutingDialogOpen] = useState(false);
  const [editingRouting, setEditingRouting] = useState<ProductionRouting | null>(null);
  const [selectedRoutingId, setSelectedRoutingId] = useState<number | null>(null);
  const [isOperationDialogOpen, setIsOperationDialogOpen] = useState(false);
  const [editingOperation, setEditingOperation] = useState<ProductionRoutingOperation | null>(null);
  const { toast } = useToast();

  const routingForm = useForm<RoutingFormData>({
    resolver: zodResolver(routingFormSchema),
    defaultValues: { 
      code: "", 
      name: "", 
      isActive: true,
    },
  });

  const operationForm = useForm<OperationFormData>({
    resolver: zodResolver(operationFormSchema),
    defaultValues: {
      code: "",
      name: "",
      estimatedTime: 0,
      setupTime: 0,
      allowParallel: false,
      qualityCheckRequired: false,
      isActive: true,
    },
  });

  const { data: routings = [], isLoading } = useQuery<ProductionRouting[]>({
    queryKey: ["/api/production/routings"],
  });

  const { data: workCenters = [] } = useQuery<ProductionWorkCenter[]>({
    queryKey: ["/api/production/work-centers"],
  });

  const { data: operations = [], isLoading: isLoadingOperations } = useQuery<ProductionRoutingOperation[]>({
    queryKey: ["/api/production/routings", selectedRoutingId, "operations"],
    enabled: !!selectedRoutingId,
  });

  const createRoutingMutation = useMutation({
    mutationFn: (data: RoutingFormData) => 
      apiRequest("/api/production/routings", "POST", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      setIsRoutingDialogOpen(false);
      routingForm.reset();
      toast({ title: "Marszruta została utworzona" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się utworzyć marszruty",
        variant: "destructive" 
      });
    },
  });

  const updateRoutingMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: RoutingFormData }) =>
      apiRequest(`/api/production/routings/${id}`, "PATCH", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      setIsRoutingDialogOpen(false);
      setEditingRouting(null);
      routingForm.reset();
      toast({ title: "Marszruta została zaktualizowana" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się zaktualizować marszruty",
        variant: "destructive" 
      });
    },
  });

  const deleteRoutingMutation = useMutation({
    mutationFn: (id: number) => apiRequest(`/api/production/routings/${id}`, "DELETE"),
    onSuccess: (_, deletedId) => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      if (selectedRoutingId === deletedId) {
        setSelectedRoutingId(null);
      }
      toast({ title: "Marszruta została usunięta" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się usunąć marszruty",
        variant: "destructive" 
      });
    },
  });

  const createOperationMutation = useMutation({
    mutationFn: (data: OperationFormData & { routingId: number }) =>
      apiRequest(`/api/production/routings/${data.routingId}/operations`, "POST", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings", selectedRoutingId, "operations"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      setIsOperationDialogOpen(false);
      operationForm.reset();
      toast({ title: "Operacja została utworzona" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się utworzyć operacji",
        variant: "destructive" 
      });
    },
  });

  const updateOperationMutation = useMutation({
    mutationFn: ({ id, routingId, data }: { id: number; routingId: number; data: OperationFormData }) =>
      apiRequest(`/api/production/routings/${routingId}/operations/${id}`, "PATCH", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings", selectedRoutingId, "operations"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      setIsOperationDialogOpen(false);
      setEditingOperation(null);
      operationForm.reset();
      toast({ title: "Operacja została zaktualizowana" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się zaktualizować operacji",
        variant: "destructive" 
      });
    },
  });

  const deleteOperationMutation = useMutation({
    mutationFn: ({ id, routingId }: { id: number; routingId: number }) =>
      apiRequest(`/api/production/routings/${routingId}/operations/${id}`, "DELETE"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings", selectedRoutingId, "operations"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      toast({ title: "Operacja została usunięta" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się usunąć operacji",
        variant: "destructive" 
      });
    },
  });

  const resequenceMutation = useMutation({
    mutationFn: ({ routingId, orderedIds }: { routingId: number; orderedIds: number[] }) =>
      apiRequest(`/api/production/routings/${routingId}/operations/resequence`, "POST", { orderedIds }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings", selectedRoutingId, "operations"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      toast({ title: "Kolejność operacji została zaktualizowana" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się zaktualizować kolejności operacji",
        variant: "destructive" 
      });
    },
  });

  useEffect(() => {
    if (editingRouting) {
      routingForm.reset({
        code: editingRouting.code,
        name: editingRouting.name,
        description: editingRouting.description ?? "",
        productId: editingRouting.productId ?? null,
        productType: editingRouting.productType ?? "",
        defaultWorkCenterId: editingRouting.defaultWorkCenterId ?? null,
        isActive: editingRouting.isActive ?? true,
        version: editingRouting.version ?? 1,
        notes: editingRouting.notes ?? "",
      });
    } else {
      routingForm.reset({ 
        code: "", 
        name: "", 
        isActive: true,
      });
    }
  }, [editingRouting, routingForm]);

  useEffect(() => {
    if (editingOperation) {
      operationForm.reset({
        code: editingOperation.code,
        name: editingOperation.name,
        description: editingOperation.description ?? "",
        workCenterId: editingOperation.workCenterId ?? null,
        normId: editingOperation.normId ?? null,
        estimatedTime: editingOperation.estimatedTime ? Number(editingOperation.estimatedTime) : 0,
        setupTime: editingOperation.setupTime ? Number(editingOperation.setupTime) : 0,
        allowParallel: editingOperation.allowParallel ?? false,
        instructions: editingOperation.instructions ?? "",
        qualityCheckRequired: editingOperation.qualityCheckRequired ?? false,
        isActive: editingOperation.isActive ?? true,
      });
    } else {
      operationForm.reset({
        code: "",
        name: "",
        estimatedTime: 0,
        setupTime: 0,
        allowParallel: false,
        qualityCheckRequired: false,
        isActive: true,
      });
    }
  }, [editingOperation, operationForm]);

  const onRoutingSubmit = (data: RoutingFormData) => {
    const normalizedData = {
      ...data,
      defaultWorkCenterId: data.defaultWorkCenterId && data.defaultWorkCenterId !== 0 ? data.defaultWorkCenterId : null,
    };

    if (editingRouting) {
      updateRoutingMutation.mutate({ id: editingRouting.id, data: normalizedData });
    } else {
      createRoutingMutation.mutate(normalizedData);
    }
  };

  const onOperationSubmit = (data: OperationFormData) => {
    if (!selectedRoutingId) return;

    const normalizedData = {
      ...data,
      routingId: selectedRoutingId,
      workCenterId: data.workCenterId && data.workCenterId !== 0 ? data.workCenterId : null,
      normId: data.normId && data.normId !== 0 ? data.normId : null,
    };

    if (editingOperation) {
      updateOperationMutation.mutate({ 
        id: editingOperation.id, 
        routingId: selectedRoutingId, 
        data: normalizedData 
      });
    } else {
      createOperationMutation.mutate(normalizedData);
    }
  };

  const getWorkCenterName = (workCenterId: number | null) => {
    if (!workCenterId) return "-";
    const wc = workCenters.find(w => w.id === workCenterId);
    return wc ? wc.name : "-";
  };

  const formatTime = (minutes: number | string | null) => {
    if (!minutes) return "-";
    const num = typeof minutes === 'string' ? parseFloat(minutes) : minutes;
    if (num < 60) return `${num} min`;
    const hours = Math.floor(num / 60);
    const mins = num % 60;
    return mins > 0 ? `${hours}h ${mins}min` : `${hours}h`;
  };

  const selectedRouting = routings.find(r => r.id === selectedRoutingId);

  const handleMoveOperation = (operationId: number, direction: 'up' | 'down') => {
    if (!selectedRoutingId) return;
    
    const currentIndex = operations.findIndex(op => op.id === operationId);
    if (currentIndex === -1) return;
    
    if (direction === 'up' && currentIndex === 0) return;
    if (direction === 'down' && currentIndex === operations.length - 1) return;
    
    const newOperations = [...operations];
    const targetIndex = direction === 'up' ? currentIndex - 1 : currentIndex + 1;
    
    [newOperations[currentIndex], newOperations[targetIndex]] = [newOperations[targetIndex], newOperations[currentIndex]];
    
    const orderedIds = newOperations.map(op => op.id);
    resequenceMutation.mutate({ routingId: selectedRoutingId, orderedIds });
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <h1 className="text-3xl font-bold" data-testid="text-page-title">Marszruty Produkcyjne</h1>
      
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Routing List */}
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-4">
            <CardTitle>Marszruty</CardTitle>
            <Button
              onClick={() => {
                setEditingRouting(null);
                setIsRoutingDialogOpen(true);
              }}
              data-testid="button-add-routing"
            >
              <Plus className="mr-2 h-4 w-4" />
              Dodaj Marszrutę
            </Button>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="flex justify-center p-8">
                <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-routings" />
              </div>
            ) : (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Kod</TableHead>
                    <TableHead>Nazwa</TableHead>
                    <TableHead>Czas</TableHead>
                    <TableHead>Aktywna</TableHead>
                    <TableHead>Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {routings.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={5} className="text-center text-muted-foreground">
                        Brak marszrut
                      </TableCell>
                    </TableRow>
                  ) : (
                    routings.map((routing) => (
                      <TableRow 
                        key={routing.id} 
                        data-testid={`row-routing-${routing.id}`}
                        className={selectedRoutingId === routing.id ? "bg-accent" : "hover-elevate cursor-pointer"}
                        onClick={() => setSelectedRoutingId(routing.id)}
                      >
                        <TableCell>
                          <Link href={`/production/routings/${routing.id}`}>
                            <span className="font-mono text-primary hover:underline cursor-pointer">
                              {routing.code}
                            </span>
                          </Link>
                        </TableCell>
                        <TableCell>
                          <Link href={`/production/routings/${routing.id}`}>
                            <span className="text-primary hover:underline cursor-pointer">
                              {routing.name}
                            </span>
                          </Link>
                        </TableCell>
                        <TableCell>
                          <div className="flex items-center gap-1">
                            <Clock className="h-3 w-3" />
                            {formatTime(routing.totalEstimatedTime)}
                          </div>
                        </TableCell>
                        <TableCell>
                          <Badge variant={routing.isActive ? "default" : "secondary"}>
                            {routing.isActive ? "Tak" : "Nie"}
                          </Badge>
                        </TableCell>
                        <TableCell onClick={(e) => e.stopPropagation()}>
                          <div className="flex gap-2">
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => {
                                setEditingRouting(routing);
                                setIsRoutingDialogOpen(true);
                              }}
                              data-testid={`button-edit-routing-${routing.id}`}
                            >
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => {
                                if (confirm("Czy na pewno chcesz usunąć tę marszrutę?")) {
                                  deleteRoutingMutation.mutate(routing.id);
                                }
                              }}
                              data-testid={`button-delete-routing-${routing.id}`}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))
                  )}
                </TableBody>
              </Table>
            )}
          </CardContent>
        </Card>

        {/* Operations List */}
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-4">
            <CardTitle>
              {selectedRouting ? (
                <div className="space-y-1">
                  <div>Operacje: {selectedRouting.name}</div>
                  <div className="text-sm font-normal text-muted-foreground">
                    Całkowity czas: {formatTime(selectedRouting.totalEstimatedTime)}
                  </div>
                </div>
              ) : (
                "Operacje"
              )}
            </CardTitle>
            {selectedRoutingId && (
              <Button
                onClick={() => {
                  setEditingOperation(null);
                  setIsOperationDialogOpen(true);
                }}
                data-testid="button-add-operation"
              >
                <Plus className="mr-2 h-4 w-4" />
                Dodaj Operację
              </Button>
            )}
          </CardHeader>
          <CardContent>
            {!selectedRoutingId ? (
              <div className="flex flex-col items-center justify-center p-8 text-muted-foreground">
                <ListOrdered className="h-12 w-12 mb-2 opacity-50" />
                <p>Wybierz marszrutę aby zobaczyć operacje</p>
              </div>
            ) : isLoadingOperations ? (
              <div className="flex justify-center p-8">
                <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-operations" />
              </div>
            ) : (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Kol.</TableHead>
                    <TableHead>Kod</TableHead>
                    <TableHead>Operacja</TableHead>
                    <TableHead>Gniazdo</TableHead>
                    <TableHead>Czas</TableHead>
                    <TableHead>Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {operations.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={6} className="text-center text-muted-foreground">
                        Brak operacji
                      </TableCell>
                    </TableRow>
                  ) : (
                    operations.map((op, index) => (
                      <TableRow key={op.id} data-testid={`row-operation-${op.id}`}>
                        <TableCell>
                          <Badge variant="outline">{op.sequence}</Badge>
                        </TableCell>
                        <TableCell>{op.code}</TableCell>
                        <TableCell>{op.name}</TableCell>
                        <TableCell className="text-sm text-muted-foreground">
                          {getWorkCenterName(op.workCenterId)}
                        </TableCell>
                        <TableCell>
                          <div className="text-sm">
                            {formatTime(op.estimatedTime)}
                            {op.setupTime && Number(op.setupTime) > 0 && (
                              <div className="text-xs text-muted-foreground">
                                +{formatTime(op.setupTime)} przygot.
                              </div>
                            )}
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="flex gap-1">
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleMoveOperation(op.id, 'up')}
                              disabled={index === 0 || resequenceMutation.isPending}
                              data-testid={`button-move-up-${op.id}`}
                            >
                              <ArrowUp className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleMoveOperation(op.id, 'down')}
                              disabled={index === operations.length - 1 || resequenceMutation.isPending}
                              data-testid={`button-move-down-${op.id}`}
                            >
                              <ArrowDown className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => {
                                setEditingOperation(op);
                                setIsOperationDialogOpen(true);
                              }}
                              data-testid={`button-edit-operation-${op.id}`}
                            >
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => {
                                if (confirm("Czy na pewno chcesz usunąć tę operację?")) {
                                  deleteOperationMutation.mutate({ 
                                    id: op.id, 
                                    routingId: selectedRoutingId 
                                  });
                                }
                              }}
                              data-testid={`button-delete-operation-${op.id}`}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))
                  )}
                </TableBody>
              </Table>
            )}
          </CardContent>
        </Card>
      </div>

      {/* Routing Dialog */}
      <Dialog open={isRoutingDialogOpen} onOpenChange={setIsRoutingDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {editingRouting ? "Edytuj Marszrutę" : "Dodaj Marszrutę"}
            </DialogTitle>
            <DialogDescription>
              {editingRouting 
                ? "Zaktualizuj informacje o marszrucie produkcyjnej" 
                : "Wypełnij formularz aby utworzyć nową marszrutę"}
            </DialogDescription>
          </DialogHeader>
          <Form {...routingForm}>
            <form onSubmit={routingForm.handleSubmit(onRoutingSubmit)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={routingForm.control}
                  name="code"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kod *</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-routing-code" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={routingForm.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Nazwa *</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-routing-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={routingForm.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} value={field.value ?? ""} rows={2} data-testid="input-routing-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={routingForm.control}
                  name="productType"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Typ Produktu</FormLabel>
                      <FormControl>
                        <Input {...field} value={field.value ?? ""} data-testid="input-product-type" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={routingForm.control}
                  name="defaultWorkCenterId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Domyślne Gniazdo</FormLabel>
                      <Select 
                        onValueChange={(value) => field.onChange(value === "0" ? null : parseInt(value))}
                        value={field.value?.toString() || "0"}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-default-workcenter">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="0">Brak</SelectItem>
                          {workCenters.map((wc) => (
                            <SelectItem key={wc.id} value={wc.id.toString()}>
                              {wc.name}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={routingForm.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Notatki</FormLabel>
                    <FormControl>
                      <Textarea {...field} value={field.value ?? ""} rows={2} data-testid="input-routing-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={routingForm.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                    <FormControl>
                      <Checkbox
                        checked={field.value ?? false}
                        onCheckedChange={field.onChange}
                        data-testid="checkbox-routing-active"
                      />
                    </FormControl>
                    <div className="space-y-1 leading-none">
                      <FormLabel>Aktywna</FormLabel>
                    </div>
                  </FormItem>
                )}
              />

              <DialogFooter>
                <Button 
                  type="submit" 
                  disabled={createRoutingMutation.isPending || updateRoutingMutation.isPending}
                  data-testid="button-submit-routing"
                >
                  {(createRoutingMutation.isPending || updateRoutingMutation.isPending) && (
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  )}
                  {editingRouting ? "Zaktualizuj" : "Utwórz"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      {/* Operation Dialog */}
      <Dialog open={isOperationDialogOpen} onOpenChange={setIsOperationDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {editingOperation ? "Edytuj Operację" : "Dodaj Operację"}
            </DialogTitle>
            <DialogDescription>
              {editingOperation 
                ? "Zaktualizuj informacje o operacji" 
                : "Wypełnij formularz aby utworzyć nową operację"}
            </DialogDescription>
          </DialogHeader>
          <Form {...operationForm}>
            <form onSubmit={operationForm.handleSubmit(onOperationSubmit)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={operationForm.control}
                  name="code"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kod *</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-operation-code" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={operationForm.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Nazwa *</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-operation-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={operationForm.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} value={field.value ?? ""} rows={2} data-testid="input-operation-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={operationForm.control}
                name="workCenterId"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Gniazdo Robocze</FormLabel>
                    <Select 
                      onValueChange={(value) => field.onChange(value === "0" ? null : parseInt(value))}
                      value={field.value?.toString() || "0"}
                    >
                      <FormControl>
                        <SelectTrigger data-testid="select-operation-workcenter">
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="0">Brak</SelectItem>
                        {workCenters.map((wc) => (
                          <SelectItem key={wc.id} value={wc.id.toString()}>
                            {wc.name}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={operationForm.control}
                  name="estimatedTime"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Czas Szacowany (min) *</FormLabel>
                      <FormControl>
                        <Input 
                          type="number" 
                          step="0.01"
                          {...field}
                          value={field.value ?? ""} 
                          data-testid="input-estimated-time" 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={operationForm.control}
                  name="setupTime"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Czas Przygotowania (min)</FormLabel>
                      <FormControl>
                        <Input 
                          type="number" 
                          step="0.01"
                          {...field}
                          value={field.value ?? ""} 
                          data-testid="input-setup-time" 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={operationForm.control}
                name="instructions"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Instrukcje</FormLabel>
                    <FormControl>
                      <Textarea {...field} value={field.value ?? ""} rows={3} data-testid="input-operation-instructions" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="space-y-3">
                <FormField
                  control={operationForm.control}
                  name="allowParallel"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                      <FormControl>
                        <Checkbox
                          checked={field.value ?? false}
                          onCheckedChange={field.onChange}
                          data-testid="checkbox-allow-parallel"
                        />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel>Dopuść równoległe wykonanie</FormLabel>
                      </div>
                    </FormItem>
                  )}
                />

                <FormField
                  control={operationForm.control}
                  name="qualityCheckRequired"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                      <FormControl>
                        <Checkbox
                          checked={field.value ?? false}
                          onCheckedChange={field.onChange}
                          data-testid="checkbox-quality-check"
                        />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel>Wymaga kontroli jakości</FormLabel>
                      </div>
                    </FormItem>
                  )}
                />

                <FormField
                  control={operationForm.control}
                  name="isActive"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                      <FormControl>
                        <Checkbox
                          checked={field.value ?? false}
                          onCheckedChange={field.onChange}
                          data-testid="checkbox-operation-active"
                        />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel>Aktywna</FormLabel>
                      </div>
                    </FormItem>
                  )}
                />
              </div>

              <DialogFooter>
                <Button 
                  type="submit" 
                  disabled={createOperationMutation.isPending || updateOperationMutation.isPending}
                  data-testid="button-submit-operation"
                >
                  {(createOperationMutation.isPending || updateOperationMutation.isPending) && (
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  )}
                  {editingOperation ? "Zaktualizuj" : "Utwórz"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
