# 📊 Production Planning System - Pełne Wyjaśnienie

## ✅ CO JUŻ DZIAŁA (Stan obecny)

### 1. **Podstawowa infrastruktura** (~4000 linii kodu)
- ✅ **8 Work Centers** (Piła Masterwood, Okleiniarka Falit, Wiertarka Master 1/2/Next, Kompletowanie, Pakowanie)
- ✅ **18 Carriers** (transportery materiałów)
- ✅ **10 Locations** (bufory magazynowe)
- ✅ **1 Routing** z 5 operacjami (Produkcja szafki standardowej)
- ✅ **9 Material Norms** (normy zużycia - płyta 20%, obrzeże +2cm)

### 2. **Backend API**
```
✅ Production Plans - tworzenie planów produkcji
✅ Plan Lines - linie planu (produkty + ilości)
✅ Batch Engine - grupowanie formatek wg koloru/edgingu/drilingu
✅ Routings - marszruty produkcyjne
✅ Orders - zlecenia produkcyjne z workflow_stage
✅ Material Movements - śledzenie przepływu
✅ Norms - normatywy zużycia materiałów
```

### 3. **Frontend** (17 stron)
- ✅ Lista planów produkcji (`/production/plans`)
- ✅ Formularze CRUD dla planów
- ✅ Filtry: status, priorytet, search
- ✅ Statusy: draft, approved, in_progress, completed, cancelled
- ✅ Priorytety: low, normal, high, urgent

---

## ❌ CO BRAKUJE (Gaps)

### 1. **WIZUALIZACJA PRZEPŁYWU** ❌
Obecnie brak graficznej wizualizacji przepływu formatek!

**Co trzeba dodać:**
- ❌ Sankey diagram / Flow chart pokazujący ruch formatek przez stacje
- ❌ Podział na kolory (wizualnie różne strumienie dla CZARNY, WOTAN, etc.)
- ❌ Podział na wiercenie TAK/NIE
- ❌ Podział na oklejanie (wszystkie 4 krawędzie vs wybrane)
- ❌ Gantt chart dla timelineu produkcji
- ❌ Real-time progress tracking

### 2. **ROUTING BRANCHES (rozgałęzienia)** ❌
Tabela `production_routing_branches` istnieje w schemacie, ALE:
- ❌ Brak UI do definiowania split routing
- ❌ Brak logiki wykonywania rozgałęzień
- ❌ Brak wizualizacji rozwidlenia (np. SIEDZISKO → 50% wiertarka + 50% tapicernia)

### 3. **UI DO NORM ZUŻYCIA** ❌
Normy są w bazie (płyta +20%, obrzeże +2cm), ALE:
- ❌ Brak strony `/production/settings` do edycji tych wartości
- ❌ Brak formularza do dodawania własnych norm
- ❌ Brak walidacji norm przy kalkulacji BOM

### 4. **DEMAND TRACKING** ❌
- ❌ Brak automatycznego pobierania popytu z `commerce.orders`
- ❌ Brak agregacji zamówień → plan lines
- ❌ Brak systemu priorytetyzacji według dat dostawy

### 5. **BOM AUTO-GENERATION** ❌
Obecnie BOM trzeba tworzyć ręcznie. Brakuje:
- ❌ Automatycznego wyliczania: płyta (m²), obrzeże (mb), klej (g) dla formatki
- ❌ Wykorzystania dimensions z `product_components` (calculated_length, calculated_width)
- ❌ Auto-generowania BOM podczas tworzenia plan line

---

## 📖 ODPOWIEDZI NA TWOJE PYTANIA

### **Q1: Gdzie jest ustawiony plan?**
**A:** Plan jest w bazie w tabeli `production.production_plans`.

**Obecnie:**
- Strona: `/production/plans`
- Możesz tworzyć plany ręcznie (nazwa, daty, status, priorytet)
- Plan ma linie (`production_plan_lines`) które zawierają: product_id, quantity, routing_id, bom_id

**Brakuje:** 
- Auto-generowania plan lines z zamówień e-commerce
- Dashboard z agregacją popytu

---

### **Q2: Jak program podzieli formatki ze zlecenia na grupy?**
**A:** Używa **Batch Engine** (`server/services/production/batch-engine.ts`)

**Algorytm grupowania:**
```typescript
Batch Key = {
  color: "CZARNY",
  edgingPattern: "ALL_4_EDGES",
  drillingPattern: "STANDARD_HOLES",
  boardType: "PAL_18mm",
  componentType: "DOOR"
}
```

**Proces:**
1. Plan Line ma `product_id` + `quantity` (np. catalog-products/348 x 10 szt)
2. System pobiera BOM dla tego produktu
3. Wyciąga wszystkie komponenty (formatki) z BOM
4. Grupuje komponenty według klucza (kolor + edging + drilling + board + type)
5. Tworzy batch dla każdej unikalnej grupy
6. Zapisuje w `production.production_component_batches`

**Przykład:**
```
Szafka 348 x 10 szt ma BOM:
  - Component 2762 (CZARNY, 4 edges, wiercenie) x 20 szt
  - Component 2763 (WOTAN, 2 edges, bez wiercenia) x 10 szt

Batch Engine tworzy:
  - BATCH-000001: CZARNY + 4edges + wiercenie → 20 szt
  - BATCH-000002: WOTAN + 2edges + bez → 10 szt
```

---

### **Q3: Jak to jest zwizualizowane?**
**A:** ❌ **BRAK WIZUALIZACJI!**

**Co jest teraz:**
- Tabela z listą batchy
- Tabela z listą operacji w routingu (sekwencja)
- Zwykłe tekstowe listy

**Co powinno być:**
- 🎨 **Sankey Diagram** pokazujący przepływ kolorów przez stacje
- 📊 **Gantt Chart** z timelineem produkcji
- 🔄 **Interactive Flow Chart** z węzłami (work centers) i krawędziami (material flow)
- 📈 **Progress Bars** dla każdego batcha na każdej stacji

---

### **Q4: Czy może być interfejs graficzny z symulacją przepływu formatek?**
**A:** ✅ **TAK! To jest BARDZO dobry pomysł i TO BRAKUJE.**

**Wizja:**
```
┌─────────────┐
│  MAGAZYN    │
│  (Buffer 1) │
└──────┬──────┘
       │
       ├─── [CZARNY - 50 formatek] ───┐
       │                               │
       ├─── [WOTAN - 30 formatek]  ───┤
       │                               │
       └─── [BIAŁY - 20 formatek]  ───┤
                                       ▼
                            ┌──────────────────┐
                            │  PIŁA MASTERWOOD │
                            │  (Cięcie)        │
                            └────────┬─────────┘
                                     │
                    ┌────────────────┼────────────────┐
                    │                │                │
             [CZARNY-40]      [WOTAN-30]      [BIAŁY-20]
                    │                │                │
                    ▼                ▼                ▼
         ┌──────────────┐  ┌──────────────┐  ┌──────────────┐
         │ OKLEINIARKA  │  │ OKLEINIARKA  │  │ OKLEINIARKA  │
         │ (4 krawędzie)│  │ (2 krawędzie)│  │ (bez oklejania)│
         └──────┬───────┘  └──────┬───────┘  └──────┬───────┘
                │                 │                 │
                ├─ wiertarka (20) │                 │
                └─ kompletowanie  └─ tapicernia (15)│
                                                     │
                                    ┌────────────────┘
                                    ▼
                            [Wszystkie → PAKOWANIE]
```

**Technologia:**
- React Flow / D3.js / Recharts dla diagramów
- Real-time WebSocket dla live updates
- Color-coded nodes (każdy kolor formatki = inny odcień)
- Animated transitions pokazujące ruch

---

### **Q5: Wytłumacz dokładnie jak ma wyglądać marszruta (routing)?**

**A:** Routing to **sekwencja operacji** przez work centers.

**Struktura:**
```sql
production_routings (marszruta):
  - code: "ROUT-SZAFKA-01"
  - name: "Produkcja szafki standardowej"
  - product_type: "FURNITURE_CABINET"

production_routing_operations (kroki):
  - sequence: 1 → OP-CUT → Piła Masterwood → 15 min
  - sequence: 2 → OP-EDGE → Okleiniarka Falit → 20 min
  - sequence: 3 → OP-DRILL → Wiertarka Master 1 → 10 min
  - sequence: 4 → OP-PICK → Kompletowanie → 5 min
  - sequence: 5 → OP-PACK → Pakowanie → 8 min
```

**Obecnie w bazie (przykład):**
```
ROUT-SZAFKA-01 ma 5 operacji:
1. Cięcie płyty (PILA-MW)
2. Oklejanie krawędzi (OKL-FALIT)
3. Wiercenie otworów (WIERT-M1)
4. Kompletowanie (KOMPL)
5. Pakowanie (PAKOW)
```

---

### **Q6: Przykład na szafkach 348 (10szt) + 354 (15szt) + 5szt na zapas**

**A:** Oto jak to powinno działać:

#### **Krok 1: Tworzenie planu**
```
Plan: "Produkcja tygodnia 48/2025"
  - Line 1: catalog-products/348 → 10 szt → routing ROUT-SZAFKA-01 (full flow)
  - Line 2: catalog-products/354 → 15 szt → routing ROUT-SZAFKA-01 (full flow)
  - Line 3: catalog-products/348 → 5 szt  → routing ROUT-SZAFKA-PARTIAL (skip pakowanie)
```

#### **Krok 2: Generowanie batchy**
```
Batch Engine pobiera BOM dla 348 i 354:
  - Szafka 348: 4 komponenty (drzwi, boki, półki, tył)
  - Szafka 354: 6 komponentów (większa szafka)

Grupuje według koloru/edgingu/drilingu:
  - BATCH-000001: CZARNY + 4edges + drill → 50 formatek (348 x10 + 354 x15)
  - BATCH-000002: WOTAN + 2edges + no drill → 30 formatek
  - BATCH-000003: CZARNY + 4edges + drill → 20 formatek (348 x5 ZAPAS)
```

#### **Krok 3: Routing dla każdego batcha**
```
BATCH-000001 & BATCH-000002 → ROUT-SZAFKA-01 (pełny flow):
  Magazyn → Piła → Okleiniarka → Wiertarka → Kompletowanie → Pakowanie → Wysyłka

BATCH-000003 → ROUT-SZAFKA-PARTIAL (zatrzymaj przed pakowaniem):
  Magazyn → Piła → Okleiniarka → Wiertarka → Kompletowanie → [STOP] → Karton
```

#### **Krok 4: Tracking workflow_stage**
```sql
production_orders:
  - order_id: 1 (BATCH-000001)
    workflow_stage: "cutting" → "edging" → "drilling" → "picking" → "packing" → "shipped"
  
  - order_id: 2 (BATCH-000003 - ZAPAS)
    workflow_stage: "cutting" → "edging" → "drilling" → "picking" → [STOP at warehouse]
```

---

### **Q7: Jak będziemy definiować szafki podczas produkcji?**

**A:** Szafki (produkty finalne) już istnieją w systemie. Oto pełny flow:

**CO JUŻ DZIAŁA:**
1. ✅ Szafka zdefiniowana w `catalog.products` (ID, SKU, nazwa, zdjęcia)
2. ✅ BOM wygenerowany w `bom.product_boms` (lista formatek = komponentów)
3. ✅ Każdy komponent ma wymiary: `calculated_length`, `calculated_width`, `thickness`
4. ✅ Batch Engine grupuje komponenty według koloru/edgingu/drilingu

**JAK DEFINIUJESZ PODCZAS PRODUKCJI:**
```
Krok 1: Tworzysz Production Plan
  - Nazwa: "Produkcja Styczeń 2025"
  - Status: draft → approved

Krok 2: Dodajesz Plan Lines (produkty do wyprodukowania)
  - Line 1: catalog-products/348 (Szafka CZARNA) → 10 szt
  - Line 2: catalog-products/354 (Szafka WOTAN) → 15 szt
  - Line 3: catalog-products/348 (Szafka CZARNA - zapas) → 5 szt + custom_routing

Krok 3: Wybierasz routing dla każdej linii
  - Line 1 & 2 → ROUT-SZAFKA-01 (pełny flow: cięcie → oklejanie → wiercenie → pakowanie)
  - Line 3 → ROUT-SZAFKA-PARTIAL (bez pakowania: cięcie → oklejanie → wiercenie → magazyn)

Krok 4: System automatycznie generuje batche
  - BATCH-000001: CZARNY + 4edges + drill → 60 formatek (10+5 szt szafki 348)
  - BATCH-000002: WOTAN + 2edges + no-drill → 90 formatek (15 szt szafki 354)
```

**CO BRAKUJE:**
❌ UI do wyboru routingu dla każdej plan line (obecnie routing_id w tabeli, ale brak selecta w formularzu)
❌ Możliwość custom routing override (np. zatrzymaj przed pakowaniem)
❌ Wizualizacja BOM podczas tworzenia plan line

---

### **Q8: Zużycie płyty (+20% odpad) i obrzeża (+2cm) - czy zrobić pola w ustawieniach?**

**A:** ✅ **TAK - normy JUŻ są w bazie!** Ale potrzebujesz UI do zarządzania nimi.

**CO JUŻ DZIAŁA:**
✅ Tabela `production.production_material_consumption_norms` zawiera:
```sql
-- Płyty meblowe
PAL:  waste_percentage = 20.00% (płyta wiórowa)
MDF:  waste_percentage = 22.00% (większa kruchość)
HDF:  waste_percentage = 18.00% (wyższa gęstość)

-- Obrzeża
ABS:  edge_overage_cm = 2.00 cm (standardowy naddatek)
PCV:  edge_overage_cm = 2.50 cm (większa elastyczność)

-- Kleje
HOTMELT: consumption_per_unit = 50.00 (gramy na metr bieżący)
EVA:     consumption_per_unit = 45.00 (mniej niż hotmelt)
```

**CO JEST W KODZIE:**
✅ Backend: `server/services/production/norms.ts` (CRUD API gotowy)
✅ Endpoints: `/api/production/norms` (GET, POST, PATCH, DELETE)
❌ Frontend: BRAK strony `/production/settings/norms` z formularzem

**CO TRZEBA ZROBIĆ:**
1. ❌ Stworzyć stronę `/production/settings/norms`
2. ❌ Formularz z polami:
   - `material_type` (board/edging/adhesive)
   - `material_category` (PAL/MDF/ABS/PCV)
   - `waste_percentage` (dla płyt)
   - `edge_overage_cm` (dla obrzeży)
   - `consumption_per_unit` (dla klejów)
3. ❌ Zintegrować z Material Breakdown:
   - Obecnie `getMaterialPrices()` używa hardcoded 20% i 2cm
   - Powinno pobierać z production_material_consumption_norms

**JAK TO BĘDZIE DZIAŁAĆ:**
```typescript
// Przykład użycia norm
const norms = await getNorms(pool, { material_type: 'board', material_category: 'PAL' });
const waste_factor = 1 + (norms.waste_percentage / 100); // 1.20 dla PAL

// Kalkulator BOM
const board_needed = base_area * waste_factor; // 1 m² × 1.20 = 1.20 m²
const edging_needed = perimeter + (edge_count * norms.edge_overage_cm / 100); // +2cm na krawędź
```

---

### **Q9: Czy trzeba tworzyć BOM dla każdej formatki (płyta m², obrzeże mb, klej g)?**

**A:** Zależy od podejścia. Oto stan obecny i propozycja:

**CO JEST TERAZ (stan faktyczny):**
✅ BOM istnieje na poziomie **produktu finalnego** (szafki)
✅ BOM zawiera listę **komponentów** (formatek) z ilościami
✅ Każdy komponent ma wymiary: `calculated_length`, `calculated_width`, `thickness`
❌ **NIE ma** BOM dla pojedynczej formatki (płyta/obrzeże/klej)

**Przykład obecnej struktury:**
```
BOM dla Szafki CZARNA (catalog-products/348):
  ├─ Component 2762 (Drzwi CZARNE 600x400mm) → 2 szt
  ├─ Component 2763 (Bok CZARNY 800x300mm) → 2 szt
  ├─ Component 2764 (Półka CZARNA 580x280mm) → 3 szt
  └─ Component 2765 (Tył CZARNY 598x798mm) → 1 szt

❌ BRAK: BOM dla Component 2762 (ile płyty, obrzeża, kleju?)
```

**PROPOZYCJA: Programowe wyliczanie BOM formatki**
❌ **NIE twórz** osobnego BOM dla każdej formatki (byłoby 10,000+ rekordów!)
✅ **Wyliczaj on-the-fly** przy generowaniu batcha lub Material Breakdown

**Implementacja:**
```typescript
function calculateComponentMaterials(component: ProductComponent, norms: MaterialNorms) {
  const length_m = component.calculated_length / 1000;
  const width_m = component.calculated_width / 1000;
  
  // 1. PŁYTA (m²) z normą odpadu
  const waste_factor = 1 + (norms.board.waste_percentage / 100); // 1.20 dla PAL
  const board_m2 = length_m * width_m * waste_factor;
  
  // 2. OBRZEŻE (mb) z naddatkiem
  const overage_m = norms.edging.edge_overage_cm / 100; // 0.02m dla ABS
  let edging_mb = 0;
  const edges = parseEdgingPattern(component.edging_pattern); // [true,true,false,false]
  
  if (edges[0]) edging_mb += length_m + overage_m; // góra
  if (edges[1]) edging_mb += width_m + overage_m;  // prawo
  if (edges[2]) edging_mb += length_m + overage_m; // dół
  if (edges[3]) edging_mb += width_m + overage_m;  // lewo
  
  // 3. KLEJ (g)
  const adhesive_g = edging_mb * norms.adhesive.consumption_per_unit; // 50g/mb
  
  return {
    board_m2,
    edging_mb,
    adhesive_g,
    color: component.color, // dla warehouse pricing
    board_price: getBoardPrice(component.color, component.thickness),
    edging_price: getEdgingPrice(component.color),
    adhesive_price: 0.05, // hardcoded lub z warehouse
  };
}
```

**GDZIE TO UŻYĆ:**
1. ✅ **Material Breakdown** - obecnie już pobiera ceny z warehouse według koloru
2. ❌ **Batch Generation** - dodać kalkulację zużycia dla całego batcha
3. ❌ **Production Order** - pokazać przewidywane zużycie materiałów

**ZALETY PROGRAMOWEGO PODEJŚCIA:**
- ✅ Nie trzeba tworzyć tysięcy BOM dla formatek
- ✅ Automatyczne aktualizacje przy zmianie dimensions
- ✅ Wykorzystuje normy z `production_material_consumption_norms`
- ✅ Ceny pobierane dynamicznie z warehouse (✅ już działa!)
- ✅ Elastyczność - zmiana normy = instant update wszystkich kalkulacji

---

### **Q10: Gdzie idą formatki SUROWA SIEDZISKO (część tapicernia, część wiertarka)?**

**A:** ❌ **ROUTING BRANCHES - TABELA ISTNIEJE, ALE FUNKCJONALNOŚĆ NIE DZIAŁA!**

**STAN OBECNY:**
✅ Tabela `production.production_routing_branches` istnieje w schemacie
✅ Pola: `routing_operation_id`, `condition_field`, `condition_value`, `percentage`, `next_operation_id`
❌ BRAK UI do definiowania rozgałęzień
❌ BRAK execution engine (kod który wykonuje split)
❌ BRAK wizualizacji rozgałęzień

**JAK TO DZIAŁA TERAZ (problem!):**
```
Routing jest LINIOWY:
  Op1 (Cięcie) → Op2 (Oklejanie) → Op3 (Wiercenie) → Op4 (Kompletowanie) → Op5 (Pakowanie)

Wszystkie formatki idą tą samą ścieżką!
SIEDZISKO nie ma oddzielnej ścieżki do tapicerni ❌
```

**JAK TO POWINNO DZIAŁAĆ:**
```
                    ┌──────────────┐
                    │ OP-EDGE      │
                    │ (Oklejanie)  │
                    └──────┬───────┘
                           │
                ┌──────────┴──────────┐
                │                     │
         IF component_type      ELSE (DOOR, etc.)
           == "SIEDZISKO"              │
                │                     │
        ┌───────┴────────┐            │
        │                │            │
    60% ▼            40% ▼            ▼
┌─────────────┐  ┌──────────┐  ┌──────────┐
│ OP-UPHOLSTER│  │ OP-DRILL │  │ OP-DRILL │
│ (Tapicernia)│  │(Wiertarka)│  │(Wiertarka)│
└─────┬───────┘  └────┬─────┘  └────┬─────┘
      │               │              │
      └───────────────┴──────────────┘
                      ▼
              ┌──────────────┐
              │ OP-PICK      │
              │(Kompletowanie)│
              └──────────────┘
```

**CO TRZEBA ZAIMPLEMENTOWAĆ:**

1. **Execution Engine** (backend logic):
```typescript
async function executeRoutingWithBranches(batch: Batch, routing: Routing) {
  for (const operation of routing.operations) {
    // Check if this operation has branches
    const branches = await getBranches(operation.id);
    
    if (branches.length > 0) {
      // SPLIT LOGIC
      for (const branch of branches) {
        // Check condition
        if (matchCondition(batch.components, branch.condition_field, branch.condition_value)) {
          // Route percentage to next operation
          const split_quantity = batch.total_quantity * (branch.percentage / 100);
          await routeToOperation(split_quantity, branch.next_operation_id);
        }
      }
    } else {
      // Standard linear flow
      await executeOperation(batch, operation);
    }
  }
}
```

2. **UI Builder** (frontend - routing editor):
```
┌─────────────────────────────────────────┐
│ Routing: ROUT-SZAFKA-01                 │
├─────────────────────────────────────────┤
│ Op1: Cięcie (PILA-MW)                   │
│ Op2: Oklejanie (OKL-FALIT)              │
│   ├─ Branch: IF component_type='SIEDZISKO' │
│   │   ├─ 60% → Op6 (Tapicernia)         │
│   │   └─ 40% → Op3 (Wiertarka)          │
│   └─ ELSE → Op3 (Wiertarka)             │
│ Op3: Wiercenie (WIERT-M1)               │
│ Op4: Kompletowanie (KOMPL)              │
│ Op5: Pakowanie (PAKOW)                  │
│ Op6: Tapicernia (TAPIC)                 │
└─────────────────────────────────────────┘

[+ Dodaj rozgałęzienie]
```

3. **Visualizer** (flow diagram):
- React Flow lub D3.js
- Nodes = Operations (work centers)
- Edges = Material flow (z procentami)
- Conditional edges = różne kolory dla różnych warunków

**PRZYKŁAD DEFINICJI W BAZIE (gdy będzie działać):**
```sql
-- Rozgałęzienie po oklejaniu
INSERT INTO production.production_routing_branches
  (routing_operation_id, condition_field, condition_value, percentage, next_operation_id, description)
VALUES
  -- SIEDZISKO → 60% tapicernia
  (2, 'component_type', 'SIEDZISKO', 60.00, 6, 'Siedziska z tapicerowaniem'),
  
  -- SIEDZISKO → 40% wiertarka (surowe bez tapicerowania)
  (2, 'component_type', 'SIEDZISKO', 40.00, 3, 'Siedziska surowe'),
  
  -- Wszystko inne → wiertarka
  (2, 'component_type', '*', 100.00, 3, 'Standardowa ścieżka dla drzwi, boków, etc.');
```

**PODSUMOWANIE Q10:**
- ❌ Obecnie: Wszystkie formatki idą tą samą ścieżką (liniowy routing)
- ✅ Tabela routing_branches istnieje, ale nie jest używana
- ❌ Trzeba: Execution engine + UI builder + Visualizer
- 🎯 Priorytet: WYSOKI (kluczowa funkcjonalność dla produkcji)

---

## 🎯 PLAN IMPLEMENTACJI - CONCRETE ACTION PLAN

### **FAZA 1: QUICK WINS (krytyczne braki UI - 2-3h pracy)**
Status: ⏳ DO ZROBIENIA
Cel: Podstawowa funkcjonalność do zarządzania planami produkcji

**1.1 Production Plan Lines UI** [PRIORYTET: KRYTYCZNY]
- ❌ Strona szczegółów planu (`/production/plans/:id`)
- ❌ Tabela z plan lines (product, quantity, routing, status)
- ❌ Formularz dodawania line z selectem routingu
- ❌ Przycisk "Generate Batches" dla plan line
- Czas: ~1.5h

**1.2 Production Settings - Material Norms** [PRIORYTET: WYSOKI]
- ❌ Strona `/production/settings/norms`
- ❌ Tabela z editable fields (waste_percentage, edge_overage_cm)
- ❌ Integracja z Material Breakdown (pobierz normy zamiast hardcoded)
- Czas: ~1h

**1.3 Partial Routing Support** [PRIORYTET: ŚREDNI]
- ❌ Dodać pole `stop_before_operation_id` do production_plan_lines
- ❌ Select w formularzu: "Zatrzymaj przed operacją"
- ❌ Logika w execution engine (skip operations after stop point)
- Czas: ~0.5h

### **FAZA 2: ROUTING BRANCHES (split logic - 4-6h pracy)**
Status: ⏳ DO ZROBIENIA
Cel: Rozgałęzienia dla SIEDZISKO (tapicernia vs wiertarka)

**2.1 Backend Execution Engine** [PRIORYTET: KRYTYCZNY]
- ❌ `server/services/production/routing-execution.ts`
- ❌ Funkcja `executeRoutingWithBranches(batch, routing)`
- ❌ Condition matching logic (component_type, color, etc.)
- ❌ Split quantity calculation (percentage-based)
- Czas: ~2h

**2.2 Routing Branches CRUD UI** [PRIORYTET: WYSOKI]
- ❌ Strona `/production/routings/:id/branches`
- ❌ Tabela z branches (condition, percentage, next_operation)
- ❌ Formularz dodawania branch (IF condition_field=value THEN goto operation)
- Czas: ~2h

**2.3 Branch Visualizer** [PRIORYTET: ŚREDNI]
- ❌ Prosty tree diagram pokazujący rozgałęzienia
- ❌ Używa React Flow lub prosty CSS tree
- ❌ Color-coded edges dla różnych warunków
- Czas: ~1.5h

### **FAZA 3: WORKFLOW VISUALIZATION (advanced - 6-8h pracy)**
Status: ⏳ DO ZROBIENIA (opcjonalne)
Cel: Graficzna wizualizacja przepływu produkcji

**3.1 Sankey Diagram** [PRIORYTET: NISKI]
- ❌ Biblioteka: Recharts lub react-flow-renderer
- ❌ Dane: batches → operations → work centers
- ❌ Color-coded streams (każdy kolor formatki osobny strumień)
- ❌ Real-time data fetching
- Czas: ~3h

**3.2 Gantt Chart Timeline** [PRIORYTET: NISKI]
- ❌ Biblioteka: react-gantt-timeline lub dhtmlx-gantt
- ❌ Oś czasu: planned_start_date → planned_end_date
- ❌ Drag & drop scheduling (optional)
- Czas: ~4h

**3.3 Interactive Flow Builder** [PRIORYTET: BARDZO NISKI]
- ❌ Drag & drop routing editor (React Flow)
- ❌ Visual representation of branches
- ❌ Auto-save to database
- Czas: ~5h

### **FAZA 4: DEMAND TRACKING (automation - 3-4h pracy)**
Status: ⏳ DO ZROBIENIA (opcjonalne)
Cel: Automatyczne pobieranie popytu z zamówień

**4.1 Order Aggregation** [PRIORYTET: ŚREDNI]
- ❌ Endpoint `/api/production/demand/aggregate`
- ❌ Query: SELECT product_id, SUM(quantity) FROM commerce.orders GROUP BY product_id
- ❌ Filtr: date range, status (paid), priority
- Czas: ~1h

**4.2 Auto Plan Lines Generator** [PRIORYTET: ŚREDNI]
- ❌ Przycisk "Import z zamówień" na stronie planu
- ❌ Dialog z filtrem dat i statusów
- ❌ Auto-tworzenie plan lines z aggregated demand
- Czas: ~2h

**4.3 Priority Scheduler** [PRIORYTET: NISKI]
- ❌ Algorytm priorytetyzacji (earliest delivery date first)
- ❌ Capacity checking (work center availability)
- ❌ Bottleneck detection
- Czas: ~2h

---

## 🚀 REKOMENDACJA - START POINT

### **🎯 Wariant A: Minimum Viable Production Planning (MVP - 3h)**
Cel: Zacząć używać systemu już dziś!

**Task List:**
1. ✅ Production Plan Lines UI (strona szczegółów + formularz)
2. ✅ Generate Batches button
3. ✅ Production Settings - Material Norms UI
4. ✅ Partial Routing support (5szt zatrzymaj przed pakowaniem)

**Rezultat:**
- Możesz tworzyć plany produkcji z produktami
- Widzisz wygenerowane batche
- Edytujesz normy zużycia (20% płyta, 2cm obrzeże)
- Definiujesz partial routing dla zapasu

---

### **🎯 Wariant B: Complete Production System (12h)**
Cel: Pełna funkcjonalność produkcji z wizualizacją

**Task List:**
1. ✅ FAZA 1 (Plan Lines + Settings + Partial Routing) - 3h
2. ✅ FAZA 2 (Routing Branches - split logic) - 6h
3. ✅ FAZA 3.1 (Sankey Diagram) - 3h

**Rezultat:**
- MVP z Wariantu A
- Routing branches dla SIEDZISKO (tapicernia + wiertarka)
- Graficzna wizualizacja przepływu formatek
- Production-ready system

---

### **🎯 Wariant C: Step-by-Step (zacznij od najważniejszego)**
Cel: Implementuj po kawałku, testuj po każdym kroku

**Krok 1:** Plan Lines UI (1.5h) → możesz dodawać produkty do planu
**Krok 2:** Material Norms UI (1h) → edytujesz normy zużycia
**Krok 3:** Routing Branches Backend (2h) → split logic działa
**Krok 4:** Routing Branches UI (2h) → definiujesz rozgałęzienia
**Krok 5:** Sankey Visualization (3h) → widzisz flow formatek

---

## 💬 PYTANIE DO CIEBIE

**Który wariant wybierasz?**
- **A) MVP (3h)** - szybkie wdrożenie, minimum funkcji
- **B) Complete (12h)** - pełny system z wizualizacją
- **C) Step-by-Step** - po kawałku, testuję po każdym kroku
- **D) Coś innego** - opisz co chcesz najpierw

**Lub powiedz:**
- "Zacznij od wizualizacji przepływu" → Faza 3 (Sankey)
- "Zacznij od routing branches" → Faza 2 (split logic)
- "Zacznij od UI" → Faza 1 (Plan Lines + Settings)
- "Zrób wszystko po kolei" → Task list na 15 zadań

---

## 📊 CURRENT STATUS SUMMARY

✅ **CO DZIAŁA:**
- Production Plans (lista, CRUD)
- Batch Engine (grupowanie formatek)
- Material Norms (w bazie, brak UI)
- Routings (liniowe, bez branches)
- Work Centers (8 stacji)

❌ **CO BRAKUJE:**
- Plan Lines UI (krytyczne!)
- Routing Branches execution (krytyczne dla SIEDZISKO!)
- Material Norms UI (wysokie!)
- Workflow visualization (nice to have)
- Demand tracking (nice to have)

**Następny krok:** Czekam na Twoją decyzję! 🚀
