import type { Express } from "express";
import { requirePermission } from "../../auth";
import { z } from "zod";
import { pool } from "../../postgres";
import { insertProductionRoutingSchema, insertProductionRoutingOperationSchema } from "@shared/schema";
import * as routingsService from "../../services/production/routings";
import * as operationsService from "../../services/production/routing-operations";

export function registerRoutingRoutes(app: Express) {
  // GET /api/production/routings - Get all routings
  app.get("/api/production/routings", requirePermission("view_production"), async (req, res) => {
    try {
      const routings = await routingsService.getRoutings(pool);
      res.json(routings);
    } catch (error) {
      console.error("Error fetching routings:", error);
      res.status(500).json({ message: "Failed to fetch routings" });
    }
  });

  // GET /api/production/routings/:id - Get routing by ID
  app.get("/api/production/routings/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid routing ID" });
      }

      const routing = await routingsService.getRoutingById(pool, id);
      if (!routing) {
        return res.status(404).json({ message: "Routing not found" });
      }

      res.json(routing);
    } catch (error) {
      console.error("Error fetching routing:", error);
      res.status(500).json({ message: "Failed to fetch routing" });
    }
  });

  // POST /api/production/routings - Create new routing
  app.post("/api/production/routings", requirePermission("manage_production"), async (req, res) => {
    try {
      const data = insertProductionRoutingSchema.parse(req.body);
      const routing = await routingsService.createRouting(pool, data);
      res.status(201).json(routing);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Routing with this code already exists" });
      }
      console.error("Error creating routing:", error);
      res.status(500).json({ message: "Failed to create routing" });
    }
  });

  // PATCH /api/production/routings/:id - Update routing
  app.patch("/api/production/routings/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid routing ID" });
      }

      const data = insertProductionRoutingSchema.partial().parse(req.body);
      const routing = await routingsService.updateRouting(pool, id, data);
      
      if (!routing) {
        return res.status(404).json({ message: "Routing not found" });
      }

      res.json(routing);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Routing with this code already exists" });
      }
      console.error("Error updating routing:", error);
      res.status(500).json({ message: "Failed to update routing" });
    }
  });

  // DELETE /api/production/routings/:id - Delete routing
  app.delete("/api/production/routings/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid routing ID" });
      }

      const deleted = await routingsService.deleteRouting(pool, id);
      if (!deleted) {
        return res.status(404).json({ message: "Routing not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting routing:", error);
      res.status(500).json({ message: "Failed to delete routing" });
    }
  });

  // GET /api/production/routings/:routingId/operations - Get operations for routing
  app.get("/api/production/routings/:routingId/operations", requirePermission("view_production"), async (req, res) => {
    try {
      const routingId = parseInt(req.params.routingId);
      if (isNaN(routingId)) {
        return res.status(400).json({ message: "Invalid routing ID" });
      }

      const operations = await operationsService.getOperationsByRoutingId(pool, routingId);
      res.json(operations);
    } catch (error) {
      console.error("Error fetching operations:", error);
      res.status(500).json({ message: "Failed to fetch operations" });
    }
  });

  // POST /api/production/routings/:routingId/operations - Create operation
  app.post("/api/production/routings/:routingId/operations", requirePermission("manage_production"), async (req, res) => {
    try {
      const routingId = parseInt(req.params.routingId);
      if (isNaN(routingId)) {
        return res.status(400).json({ message: "Invalid routing ID" });
      }

      const data = insertProductionRoutingOperationSchema.parse({
        ...req.body,
        routingId,
      });

      const operation = await operationsService.createOperation(pool, data);
      res.status(201).json(operation);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Operation with this sequence already exists" });
      }
      console.error("Error creating operation:", error);
      res.status(500).json({ message: "Failed to create operation" });
    }
  });

  // PATCH /api/production/routings/:routingId/operations/:id - Update operation
  app.patch("/api/production/routings/:routingId/operations/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const routingId = parseInt(req.params.routingId);
      const id = parseInt(req.params.id);
      if (isNaN(routingId) || isNaN(id)) {
        return res.status(400).json({ message: "Invalid routing or operation ID" });
      }

      const data = insertProductionRoutingOperationSchema.partial().parse(req.body);
      const operation = await operationsService.updateOperation(pool, id, data);
      
      if (!operation) {
        return res.status(404).json({ message: "Operation not found" });
      }

      // Verify operation belongs to routing
      if (operation.routingId !== routingId) {
        return res.status(400).json({ message: "Operation does not belong to this routing" });
      }

      res.json(operation);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Operation with this sequence already exists" });
      }
      console.error("Error updating operation:", error);
      res.status(500).json({ message: "Failed to update operation" });
    }
  });

  // DELETE /api/production/routings/:routingId/operations/:id - Delete operation
  app.delete("/api/production/routings/:routingId/operations/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid operation ID" });
      }

      const deleted = await operationsService.deleteOperation(pool, id);
      if (!deleted) {
        return res.status(404).json({ message: "Operation not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting operation:", error);
      res.status(500).json({ message: "Failed to delete operation" });
    }
  });

  // POST /api/production/routings/:routingId/operations/resequence - Resequence operations
  app.post("/api/production/routings/:routingId/operations/resequence", requirePermission("manage_production"), async (req, res) => {
    try {
      const routingId = parseInt(req.params.routingId);
      if (isNaN(routingId)) {
        return res.status(400).json({ message: "Invalid routing ID" });
      }

      const schema = z.object({
        orderedIds: z.array(z.number()),
      });

      const { orderedIds } = schema.parse(req.body);
      
      await operationsService.resequenceOperations(pool, routingId, orderedIds);
      res.status(200).json({ message: "Operations resequenced successfully" });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error resequencing operations:", error);
      res.status(500).json({ message: "Failed to resequence operations" });
    }
  });
}
