import { useState, useMemo, useEffect, useRef } from "react";
import { ShoppingCart, Search, X, Plus, LinkIcon, Loader2, Package, List, Layers } from "lucide-react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { ToggleGroup, ToggleGroupItem } from "@/components/ui/toggle-group";
import { useToast } from "@/hooks/use-toast";
import { OrdersTabsBar } from "./OrdersTabsBar";
import { CompactOrdersList } from "./CompactOrdersList";
import { GroupedOrdersList } from "./GroupedOrdersList";
import { OrdersControlBar } from "./OrdersControlBar";
import { useScrollCollapse } from "./use-scroll-collapse";

interface SetComponent {
  component_id: number;
  component_sku: string;
  component_title: string;
  component_color: string | null;
  component_length: string | null;
  component_width: string | null;
  component_height: string | null;
  component_product_type: string | null;
  component_doors: string | null;
  component_legs: string | null;
  quantity: number;
  primary_image_url: string | null;
  parent_set_image_url: string | null;
  is_in_current_plan: boolean;
  is_in_any_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
}

interface MarketplaceOrderItem {
  item_id: number;
  offer_external_id: string;
  name: string;
  quantity: number;
  unit_price: string;
  price: string;
  image_url: string | null;
  product_length: string | null;
  product_width: string | null;
  product_height: string | null;
  product_color: string | null;
  product_color_options: string[] | null;
  product_sku: string | null;
  product_type: string | null;
  product_doors: string | null;
  product_legs: string | null;
  marketplace_product_id: number | null;
  link_type: string | null;
  catalog_product_id: number | null;
  catalog_product_sku: string | null;
  catalog_product_title: string | null;
  catalog_set_id: number | null;
  catalog_set_sku: string | null;
  catalog_set_title: string | null;
  platform_link_id: number | null;
  bom_component_count: number | null;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
  in_current_plan: boolean;
  set_components: SetComponent[] | null;
}

interface MarketplaceOrder {
  order_id: number;
  order_number: string;
  marketplace: string;
  buyer_first_name: string;
  buyer_last_name: string;
  buyer_email: string;
  order_date: string;
  payment_status: string;
  total_to_pay_amount: string;
  currency: string;
  items: MarketplaceOrderItem[];
}

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

interface OrdersPanelProps {
  planId: string;
  orders: MarketplaceOrder[];
  isLoading: boolean;
  statistics: {
    ordersCount: number;
    productsCount: number;
  };
  filters: {
    search?: string;
    color?: string;
    sku?: string;
    limit?: number;
    offset?: number;
    sortBy?: 'order_date' | 'order_number' | 'buyer_name' | 'total_amount' | 'product_sku';
    sortOrder?: 'asc' | 'desc';
    [key: string]: any;
  };
  searchFilter: string;
  onSearchChange: (value: string) => void;
  onFilterChange: (key: string, value: string) => void;
  onClearFilter: (key: string) => void;
  onResetFilters: () => void;
  onAddItem: (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string) => void;
  onLinkItem: (item: MarketplaceOrderItem, marketplace: string) => void;
  activeFiltersCount: number;
  isAddingItem?: boolean;
  recentlyAddedItems?: Set<number>;
  colors?: DictionaryItem[];
  onCollapseChange?: (isCollapsed: boolean) => void;
  enabled?: boolean;
}

export function OrdersPanel({
  planId,
  orders,
  isLoading,
  statistics,
  filters,
  searchFilter,
  onSearchChange,
  onFilterChange,
  onClearFilter,
  onResetFilters,
  onAddItem,
  onLinkItem,
  activeFiltersCount,
  isAddingItem = false,
  recentlyAddedItems = new Set(),
  colors,
  onCollapseChange,
  enabled = true,
}: OrdersPanelProps) {
  const { toast } = useToast();
  
  // View mode with localStorage persistence
  const [viewMode, setViewMode] = useState<"flat" | "grouped">(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-view-mode`);
      return (saved === "flat" || saved === "grouped") ? saved : "flat";
    } catch {
      return "flat";
    }
  });
  
  // Save view mode to localStorage
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-view-mode`, viewMode);
  }, [planId, viewMode]);
  
  const [expandedOrders, setExpandedOrders] = useState<Set<number>>(new Set());
  const [selectedItemIds, setSelectedItemIds] = useState<Set<string>>(new Set());
  
  // Grouped view expansion state - persist in localStorage
  const [expandedColorGroups, setExpandedColorGroups] = useState<Set<string>>(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-expanded-colors`);
      return saved ? new Set(JSON.parse(saved)) : new Set();
    } catch {
      return new Set();
    }
  });
  
  const [expandedTypeGroups, setExpandedTypeGroups] = useState<Set<string>>(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-expanded-types`);
      return saved ? new Set(JSON.parse(saved)) : new Set();
    } catch {
      return new Set();
    }
  });
  
  const [expandedSets, setExpandedSets] = useState<Set<number>>(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-expanded-sets`);
      if (saved) {
        return new Set(JSON.parse(saved));
      }
      // Initialize with all sets expanded by default
      const allSetItemIds = new Set<number>();
      orders.forEach(order => {
        order.items.forEach(item => {
          if (item.set_components && item.set_components.length > 0) {
            allSetItemIds.add(item.item_id);
          }
        });
      });
      return allSetItemIds;
    } catch {
      return new Set();
    }
  });
  
  // Save expansion state to localStorage
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-expanded-colors`, JSON.stringify(Array.from(expandedColorGroups)));
  }, [planId, expandedColorGroups]);
  
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-expanded-types`, JSON.stringify(Array.from(expandedTypeGroups)));
  }, [planId, expandedTypeGroups]);
  
  useEffect(() => {
    localStorage.setItem(`plan-${planId}-expanded-sets`, JSON.stringify(Array.from(expandedSets)));
  }, [planId, expandedSets]);
  
  const allExpanded = useMemo(() => {
    if (viewMode !== 'grouped' || orders.length === 0) return false;
    const allOrderIds = orders.map(o => o.order_id);
    return allOrderIds.length > 0 && allOrderIds.every(id => expandedOrders.has(id));
  }, [viewMode, orders, expandedOrders]);

  useEffect(() => {
    setExpandedOrders(new Set());
  }, [viewMode]);

  const toggleOrder = (orderId: number) => {
    setExpandedOrders(prev => {
      const newSet = new Set(prev);
      if (newSet.has(orderId)) {
        newSet.delete(orderId);
      } else {
        newSet.add(orderId);
      }
      return newSet;
    });
  };

  const handleExpandAll = () => {
    const allOrderIds = orders.map(o => o.order_id);
    setExpandedOrders(new Set(allOrderIds));
  };

  const handleCollapseAll = () => {
    setExpandedOrders(new Set());
  };

  const handlePageSizeChange = (size: number) => {
    onFilterChange('limit', size.toString());
    onFilterChange('offset', '0');
  };

  const handleSortByChange = (sortBy: 'order_date' | 'order_number' | 'buyer_name' | 'total_amount' | 'product_sku') => {
    onFilterChange('sortBy', sortBy);
  };

  const handleSortOrderToggle = () => {
    const currentOrder = filters.sortOrder || 'desc';
    onFilterChange('sortOrder', currentOrder === 'asc' ? 'desc' : 'asc');
  };
  
  // Batch selection handlers
  const toggleItemSelection = (itemId: string) => {
    setSelectedItemIds(prev => {
      const newSet = new Set(prev);
      if (newSet.has(itemId)) {
        newSet.delete(itemId);
      } else {
        newSet.add(itemId);
      }
      return newSet;
    });
  };
  
  const clearSelection = () => {
    setSelectedItemIds(new Set());
  };
  
  const handleBulkAdd = () => {
    if (selectedItemIds.size === 0) return;
    
    let addedCount = 0;
    
    // Process each order and its items/components
    orders.forEach(order => {
      const orderNumber = order.order_number;
      const buyerName = `${order.buyer_first_name} ${order.buyer_last_name}`;
      const marketplace = order.marketplace;
      
      order.items.forEach(item => {
        // Check if main product is selected
        if (selectedItemIds.has(`item-${item.item_id}`)) {
          onAddItem(item, orderNumber, buyerName, marketplace);
          addedCount++;
        }
        
        // Check if any set components are selected
        if (item.set_components && item.set_components.length > 0) {
          item.set_components.forEach(component => {
            const componentKey = `component-${orderNumber}-${component.component_id}`;
            if (selectedItemIds.has(componentKey)) {
              // Transform component into item format for onAddItem
              const componentItem = {
                ...item,
                item_id: component.component_id,
                name: component.component_title,
                product_sku: component.component_sku,
                catalog_product_id: component.component_id,
                quantity: component.quantity,
                image_url: component.primary_image_url,
              };
              onAddItem(componentItem, orderNumber, buyerName, marketplace);
              addedCount++;
            }
          });
        }
      });
    });
    
    clearSelection();
    
    if (addedCount > 0) {
      toast({
        title: "Dodano do planu",
        description: `Dodano ${addedCount} ${addedCount === 1 ? 'produkt' : addedCount > 4 ? 'produktów' : 'produkty'} do planu produkcji`,
      });
    }
  };

  // Scroll collapse setup
  const scrollContainerRef = useRef<HTMLDivElement>(null);
  const { isCollapsed } = useScrollCollapse({ 
    scrollRef: scrollContainerRef, 
    threshold: 80,
    onChange: onCollapseChange,
    enabled
  });
  
  // Save scroll position before data changes
  const savedScrollPositionRef = useRef<number>(0);
  const previousOrdersLengthRef = useRef<number>(orders.length);
  
  // Save scroll position when adding items starts
  useEffect(() => {
    if (isAddingItem && scrollContainerRef.current) {
      savedScrollPositionRef.current = scrollContainerRef.current.scrollTop;
    }
  }, [isAddingItem]);
  
  // Restore scroll position after data changes (refetch)
  useEffect(() => {
    // Only restore if orders count changed (indicating refetch happened)
    if (orders.length !== previousOrdersLengthRef.current && savedScrollPositionRef.current > 0) {
      // Use setTimeout to ensure DOM has updated
      setTimeout(() => {
        if (scrollContainerRef.current) {
          scrollContainerRef.current.scrollTop = savedScrollPositionRef.current;
          // Reset saved position after restoration
          savedScrollPositionRef.current = 0;
        }
      }, 0);
    }
    previousOrdersLengthRef.current = orders.length;
  }, [orders]);

  return (
    <>
      {/* Progress Window - non-blocking, floating small window */}
      {isAddingItem && (
        <div className="fixed top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 z-50 pointer-events-auto">
          <div className="bg-background border-2 border-primary rounded-lg shadow-2xl p-8 max-w-sm">
            <div className="flex flex-col items-center gap-4">
              <Loader2 className="h-12 w-12 animate-spin text-primary" />
              <div className="text-center">
                <p className="text-lg font-semibold">Dodawanie do planu...</p>
                <p className="text-sm text-muted-foreground mt-1">Proszę czekać</p>
              </div>
            </div>
          </div>
        </div>
      )}

      <Card className="h-full flex flex-col rounded-none border-0 overflow-hidden relative">
      
      {/* Collapsible Header + Search + Controls */}
      <div 
        className={`flex-shrink-0 transition-all duration-300 ease-in-out overflow-hidden ${
          isCollapsed ? 'max-h-0 opacity-0' : 'max-h-[500px] opacity-100'
        }`}
        style={{ transitionProperty: 'max-height, opacity' }}
      >
        <CardHeader className="pb-2 px-4 border-b bg-background">
          {/* Compact header: title + inline counters */}
          <div className="flex items-center gap-6">
            <CardTitle className="flex items-center gap-2">
              <ShoppingCart className="h-5 w-5" />
              Dostępne Zamówienia
            </CardTitle>
            <div className="flex items-center gap-4 text-sm font-medium">
              <span data-testid="text-orders-count">Zamówienia: {statistics?.ordersCount ?? 0}</span>
              <Separator orientation="vertical" className="h-4" />
              <span data-testid="text-products-count">Produkty: {statistics?.productsCount ?? 0}</span>
            </div>
          </div>
        </CardHeader>

        {/* View Toggles + Controls - in collapsible section */}
        <div className="px-4 pt-4 pb-2 bg-background">
          {/* Notebook-style Tabs */}
          <div className="flex items-start gap-[2px] mb-2">
            <button
              onClick={() => setViewMode("flat")}
              className={`
                relative px-4 h-[32px] text-sm font-medium
                transition-all flex items-center gap-2
                ${viewMode === "flat" 
                  ? "bg-muted/20 text-foreground" 
                  : "bg-transparent text-muted-foreground hover:text-foreground/80"
                }
              `}
              data-testid="toggle-flat-view"
            >
              <ShoppingCart className="h-4 w-4 text-purple-500" />
              Zamówienia
            </button>
            <button
              onClick={() => setViewMode("grouped")}
              className={`
                relative px-4 h-[32px] text-sm font-medium
                transition-all flex items-center gap-2
                ${viewMode === "grouped" 
                  ? "bg-muted/20 text-foreground" 
                  : "bg-transparent text-muted-foreground hover:text-foreground/80"
                }
              `}
              data-testid="toggle-grouped-view"
            >
              <Layers className="h-4 w-4 text-purple-500" />
              Grupowany
            </button>
          </div>

          {/* Controls Bar + Toggle Filters */}
          <div className="flex items-center justify-between h-[30px] bg-muted/30 px-4">
            {/* Left: Toggle Filters */}
            <div className="flex items-center gap-1">
              <Button
                variant={filters.showSetsOnly === 'true' ? "default" : "outline"}
                size="sm"
                onClick={() => onFilterChange('showSetsOnly', filters.showSetsOnly === 'true' ? 'false' : 'true')}
                className="h-[25px] px-2 text-xs gap-1"
                data-testid="toggle-sets-only"
                title="Pokaż tylko zestawy"
              >
                <Package className="h-3 w-3" />
                Zestawy
              </Button>
              
              <Button
                variant={filters.showCatalogLinked === 'true' ? "default" : "outline"}
                size="sm"
                onClick={() => onFilterChange('showCatalogLinked', filters.showCatalogLinked === 'true' ? 'false' : 'true')}
                className="h-[25px] px-2 text-xs gap-1"
                data-testid="toggle-catalog-linked"
                title="Pokaż tylko połączone z katalogiem"
              >
                <LinkIcon className="h-3 w-3" />
                Połączone
              </Button>
            </div>
            
            {/* Right: Sort Controls */}
            <OrdersControlBar
              viewMode={viewMode}
              allExpanded={allExpanded}
              onExpandAll={handleExpandAll}
              onCollapseAll={handleCollapseAll}
              pageSize={filters.limit || 100}
              onPageSizeChange={handlePageSizeChange}
              sortBy={filters.sortBy || 'order_date'}
              sortOrder={filters.sortOrder || 'desc'}
              onSortByChange={handleSortByChange}
              onSortOrderToggle={handleSortOrderToggle}
            />
          </div>
        </div>
      </div>

      {/* Bulk Actions Toolbar - Sticky when items selected */}
      {selectedItemIds.size > 0 && (
        <div className="sticky top-0 z-30 flex-shrink-0 flex items-center justify-between h-[30px] px-4 bg-primary/10 border-b">
          <span className="text-xs font-medium">
            Zaznaczono: {selectedItemIds.size}
          </span>
          <div className="flex items-center gap-2">
            <Button
              size="sm"
              variant="ghost"
              onClick={clearSelection}
              className="h-6 text-xs"
              data-testid="button-clear-selection"
            >
              Wyczyść
            </Button>
            <Button
              size="sm"
              onClick={handleBulkAdd}
              className="h-6 text-xs"
              data-testid="button-bulk-add"
            >
              <Plus className="h-3 w-3 mr-1" />
              Dodaj do planu
            </Button>
          </div>
        </div>
      )}

      {/* OrdersTabsBar - Sticky Filters */}
      <div 
        className={`sticky ${selectedItemIds.size > 0 ? 'top-[30px]' : 'top-0'} z-20 flex-shrink-0 px-4 pt-4 pb-3 bg-background border-b`} 
        data-testid="orders-tabs-section"
      >
        <OrdersTabsBar
          planId={planId}
          searchFilter={searchFilter}
          onSearchChange={onSearchChange}
          showSetsOnly={filters.showSetsOnly === 'true'}
          showCatalogLinked={filters.showCatalogLinked === 'true'}
          onToggleSetsOnly={() => onFilterChange('showSetsOnly', filters.showSetsOnly === 'true' ? 'false' : 'true')}
          onToggleCatalogLinked={() => onFilterChange('showCatalogLinked', filters.showCatalogLinked === 'true' ? 'false' : 'true')}
        />
      </div>
      {/* Scrollable Orders List */}
      <div ref={scrollContainerRef} className="flex-1 overflow-y-auto px-4 pb-4">
        {isLoading ? (
          <div className="flex items-center justify-center p-8">
            <Loader2 className="h-6 w-6 animate-spin" />
          </div>
        ) : viewMode === "flat" ? (
          <CompactOrdersList 
            orders={orders} 
            colors={colors} 
            onAddItem={onAddItem}
            isAddingItem={isAddingItem}
            recentlyAddedItems={recentlyAddedItems}
            selectedItemIds={selectedItemIds}
            onToggleItemSelection={toggleItemSelection}
          />
        ) : (
          <GroupedOrdersList
            orders={orders}
            colors={colors}
            onAddItem={onAddItem}
            isAddingItem={isAddingItem}
            recentlyAddedItems={recentlyAddedItems}
            selectedItemIds={selectedItemIds}
            onToggleItemSelection={toggleItemSelection}
            expandedColorGroups={expandedColorGroups}
            setExpandedColorGroups={setExpandedColorGroups}
            expandedTypeGroups={expandedTypeGroups}
            setExpandedTypeGroups={setExpandedTypeGroups}
            expandedSets={expandedSets}
            setExpandedSets={setExpandedSets}
          />
        )}
      </div>
    </Card>
    </>
  );
}
