import { useSortable } from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { ArrowRight, Package, ExternalLink, Copy, FileText } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/checkbox";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useLocation } from "wouter";
import { useToast } from "@/hooks/use-toast";

// Shared grid columns for tabular layout
// Always includes Order# column (100px) - use empty placeholder when not showing
// Added 20px column for product links (catalog + marketplace)
export const PRODUCT_GRID_COLUMNS_WITH_CHECKBOX = '24px 20px 40px 100px 80px 60px 60px 100px 60px 60px auto';
export const PRODUCT_GRID_COLUMNS = '20px 40px 100px 80px 60px 60px 100px 60px 60px auto';

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

export interface SetComponent {
  component_id: number;
  component_sku: string;
  component_title: string;
  component_color: string | null;
  component_length: string | null;
  component_width: string | null;
  component_height: string | null;
  component_product_type: string | null;
  component_doors: string | null;
  component_legs: string | null;
  quantity: number;
  primary_image_url: string | null;
  parent_set_image_url: string | null;
  is_in_current_plan: boolean;
  is_in_any_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
}

export interface MarketplaceOrderItem {
  item_id: number;
  offer_external_id: string;
  name: string;
  quantity: number;
  unit_price: string;
  price: string;
  image_url: string | null;
  product_length: string | null;
  product_width: string | null;
  product_height: string | null;
  product_color: string | null;
  product_color_options: string[] | null;
  product_sku: string | null;
  product_type: string | null;
  product_doors: string | null;
  product_legs: string | null;
  marketplace_product_id: number | null;
  link_type: string | null;
  catalog_product_id: number | null;
  catalog_product_sku: string | null;
  catalog_product_title: string | null;
  catalog_set_id: number | null;
  catalog_set_sku: string | null;
  catalog_set_title: string | null;
  platform_link_id: number | null;
  bom_component_count: number | null;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
  in_current_plan: boolean;
  set_components: SetComponent[] | null;
}

function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}

function needsVisibleBorder(hexColor: string | null | undefined): boolean {
  if (!hexColor || !hexColor.startsWith('#')) return false;
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance < 30 || luminance > 230;
}

interface DraggableProductItemProps {
  item: MarketplaceOrderItem;
  orderNumber: string;
  buyerName: string;
  marketplace: string;
  orderDate?: string;
  colors?: DictionaryItem[];
  onAddClick?: (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string, setId?: number, componentId?: number) => void;
  onAddSetComponents?: () => void;
  isAdding?: boolean;
  isRecentlyAdded?: boolean;
  rowHeight?: string;
  showOrderInfo?: boolean;
  isSelected?: boolean;
  onToggleSelect?: (itemId: number) => void;
  orderTone?: 0 | 1;
  setId?: number;
  componentId?: number;
}

export function DraggableProductItem({ 
  item, 
  orderNumber, 
  buyerName, 
  marketplace,
  orderDate,
  colors, 
  onAddClick,
  onAddSetComponents,
  isAdding,
  isRecentlyAdded = false,
  rowHeight = "30px",
  showOrderInfo = false,
  isSelected = false,
  onToggleSelect,
  orderTone,
  setId,
  componentId
}: DraggableProductItemProps) {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  const handleCopyOrderNumber = (e: React.MouseEvent) => {
    e.stopPropagation();
    navigator.clipboard.writeText(orderNumber);
    toast({
      description: `Skopiowano numer zamówienia: ${orderNumber}`,
      duration: 2000,
    });
  };
  
  const handleGoToOrder = (e: React.MouseEvent) => {
    e.stopPropagation();
    setLocation(`/order/${orderNumber}`);
  };
  
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ 
    id: `item-${item.item_id}`,
    data: {
      type: 'order-item',
      item,
      orderNumber,
      buyerName,
      marketplace
    },
    disabled: !!item.in_plan_number // Disable drag if already in any plan
  });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.8 : 1,
    cursor: isDragging ? 'grabbing' : 'grab',
  };

  const hasMapping = !!(item.catalog_product_id || item.catalog_set_id);

  // Find color dictionary item for badge styling - fallback to black if missing
  const colorDict = colors?.find(c => c.code === item.product_color);
  const colorBadgeStyle = colorDict?.color ? {
    backgroundColor: colorDict.color,
    color: getTextColorForBackground(colorDict.color),
    borderColor: 'transparent'
  } : {
    backgroundColor: '#000000',
    color: '#FFFFFF',
    borderColor: 'transparent'
  };

  // Determine background class based on precedence:
  // 1. Recently added (green flash) - highest priority
  // 2. Unmapped products (muted)
  // 3. Order tone alternation (subtle backgrounds)
  const getBackgroundClass = () => {
    if (isRecentlyAdded) {
      return 'bg-green-500/20 dark:bg-green-500/10 transition-colors duration-700';
    }
    if (!hasMapping) {
      return 'bg-muted/30';
    }
    // Order tone: alternating backgrounds for same order grouping
    if (orderTone === 1) {
      return 'bg-muted/40';
    }
    return 'bg-background'; // Default for tone 0
  };

  return (
    <div
      ref={setNodeRef}
      {...(hasMapping ? attributes : {})}
      {...(hasMapping ? listeners : {})}
      className={`
        grid items-center gap-2 px-2 py-0.5 transition-all
        ${hasMapping ? 'hover-elevate cursor-grab active:cursor-grabbing' : 'opacity-50'}
        ${isDragging ? 'shadow-2xl shadow-purple-500/40 scale-105 z-50 border-2 border-purple-400' : ''}
        ${getBackgroundClass()}
      `}
      style={{
        gridTemplateColumns: onToggleSelect ? PRODUCT_GRID_COLUMNS_WITH_CHECKBOX : PRODUCT_GRID_COLUMNS,
        transform: style.transform,
        transition: style.transition,
        opacity: style.opacity,
        minHeight: rowHeight,
        cursor: style.cursor
      }}
      data-testid={`product-row-${item.item_id}`}
      title={!hasMapping ? 'Produkt nie zmapowany - nie można przeciągnąć' : 'Przeciągnąć na prawą stronę aby dodać do planu'}
    >
      {/* Checkbox */}
      {onToggleSelect && (
        <div className="flex items-center justify-center">
          <Checkbox
            checked={isSelected}
            onCheckedChange={() => onToggleSelect(item.item_id)}
            data-testid={`checkbox-select-${item.item_id}`}
          />
        </div>
      )}
      
      {/* Product Links - Catalog & Marketplace */}
      <div className="flex flex-col items-center justify-center gap-0.5">
        {item.catalog_product_id && (
          <button
            className="w-4 h-4 flex items-center justify-center text-blue-500 hover:text-blue-600 transition-colors"
            title="Otwórz produkt katalogowy"
            onClick={(e) => {
              e.stopPropagation();
              setLocation(`/catalog-products/${item.catalog_product_id}`);
            }}
            data-testid={`link-catalog-${item.item_id}`}
          >
            <ExternalLink className="w-3 h-3" />
          </button>
        )}
        {item.catalog_set_id && (
          <button
            className="w-4 h-4 flex items-center justify-center text-green-500 hover:text-green-600 transition-colors"
            title="Otwórz zestaw katalogowy"
            onClick={(e) => {
              e.stopPropagation();
              setLocation(`/catalog-sets/${item.catalog_set_id}`);
            }}
            data-testid={`link-catalog-set-${item.item_id}`}
          >
            <ExternalLink className="w-3 h-3" />
          </button>
        )}
        {item.marketplace_product_id && (
          <button
            className="w-4 h-4 flex items-center justify-center text-purple-500 hover:text-purple-600 transition-colors"
            title="Otwórz produkt marketplace"
            onClick={(e) => {
              e.stopPropagation();
              setLocation(`/product/${item.offer_external_id}`);
            }}
            data-testid={`link-marketplace-${item.item_id}`}
          >
            <ExternalLink className="w-3 h-3" />
          </button>
        )}
      </div>
      
      {/* Image - 40x40px NO rounded corners */}
      <div className="flex items-start justify-center w-10 flex-shrink-0 py-0.5">
        {item.image_url ? (
          <img
            src={item.image_url}
            alt={item.name}
            className="w-10 h-10 object-cover"
          />
        ) : (
          <div className="w-10 h-10 bg-muted flex items-center justify-center">
            <Package className="h-5 w-5 text-muted-foreground" />
          </div>
        )}
      </div>
      
      {/* Order Number - only in grouped view */}
      {showOrderInfo ? (
        <div className="flex flex-col justify-center min-w-0 pb-1">
          <div className="flex items-center gap-1">
            <Badge 
              className="h-5 px-2 text-[10px] bg-black text-white border-transparent cursor-pointer hover:bg-gray-800 transition-colors"
              onClick={handleCopyOrderNumber}
              title="Kliknij aby skopiować numer zamówienia"
            >
              #{orderNumber}
            </Badge>
            <button
              className="w-3 h-3 flex items-center justify-center text-primary hover:text-primary/80 transition-colors"
              title="Przejdź do zamówienia"
              onClick={handleGoToOrder}
            >
              <FileText className="w-3 h-3" />
            </button>
          </div>
          <span className="text-[9px] text-muted-foreground truncate leading-tight">{buyerName}</span>
          {orderDate && (
            <span className="text-[9px] text-muted-foreground truncate leading-tight">
              {format(new Date(orderDate), "dd.MM.yyyy HH:mm", { locale: pl })}
            </span>
          )}
        </div>
      ) : (
        <div className="min-w-0" /> /* Empty placeholder to maintain grid alignment */
      )}
      
      {/* Product Type */}
      <div className="flex items-center justify-center">
        {item.product_type && (
          <Badge variant="secondary" className="h-5 px-2 text-[10px] truncate max-w-full">
            {item.product_type}
          </Badge>
        )}
      </div>
      
      {/* Length */}
      <div className="flex items-center justify-center text-[10px] text-muted-foreground">
        {item.product_length && item.product_length !== "0" ? item.product_length : '-'}
      </div>
      
      {/* Width */}
      <div className="flex items-center justify-center text-[10px] text-muted-foreground">
        {item.product_width && item.product_width !== "0" ? item.product_width : '-'}
      </div>
      
      {/* Color */}
      <div className="flex items-center justify-center min-w-0">
        {item.product_color && (
          <Badge 
            variant="outline" 
            className="h-5 px-2 text-[10px] truncate max-w-full"
            style={colorBadgeStyle}
          >
            {item.product_color}
          </Badge>
        )}
      </div>
      
      {/* Doors */}
      <div className="flex items-center justify-center min-w-0">
        {item.product_doors && (
          <Badge variant="outline" className="h-5 px-2 text-[10px] truncate max-w-full">
            {item.product_doors}
          </Badge>
        )}
      </div>
      
      {/* Legs */}
      <div className="flex items-center justify-center min-w-0">
        {item.product_legs && (
          <Badge variant="outline" className="h-5 px-2 text-[10px] truncate max-w-full">
            {item.product_legs}
          </Badge>
        )}
      </div>
      
      {/* Metadata + Actions (last column auto) */}
      <div className="flex items-center gap-2 justify-end pr-2">
        {item.product_color_options && item.product_color_options.length > 0 && (
          <Badge variant="secondary" className="h-5 px-2 text-[10px]" title={`Opcje: ${item.product_color_options.join(', ')}`}>
            +{item.product_color_options.length}
          </Badge>
        )}
        
        {item.bom_component_count && item.bom_component_count > 0 && (
          <Badge variant="default" className="h-5 px-2 text-[10px]" title={`Komponenty BOM: ${item.bom_component_count}`}>
            BOM:{item.bom_component_count}
          </Badge>
        )}
        
        <span className="text-[10px] font-semibold">
          x{item.quantity}
        </span>

        {!hasMapping && (
          <Badge variant="destructive" className="h-5 px-2 text-[10px]">
            Brak mapowania
          </Badge>
        )}
        
        {hasMapping && onAddSetComponents && (
          <Button
            variant="ghost"
            className="h-[28px] px-3 text-xs bg-green-600 hover:bg-green-700 text-white flex-shrink-0 rounded-sm"
            onClick={(e) => {
              e.stopPropagation();
              onAddSetComponents();
            }}
            disabled={isAdding}
            data-testid={`button-add-set-${item.item_id}`}
            title="Dodaj wszystkie komponenty zestawu do planu"
          >
            Dodaj zestaw
          </Button>
        )}
        
        {hasMapping && onAddClick && !onAddSetComponents && (
          item.in_plan_number ? (
            <Button
              variant="ghost"
              className="h-[28px] px-2 text-xs bg-blue-600 hover:bg-blue-700 text-white flex-shrink-0 rounded-sm"
              onClick={(e) => {
                e.stopPropagation();
                if (item.in_plan_id) {
                  setLocation(`/production/plans/${item.in_plan_id}`);
                }
              }}
              disabled={!item.in_plan_id}
              data-testid={`badge-in-plan-${item.item_id}`}
              title={item.in_plan_id ? `Kliknij aby przejść do planu ${item.in_plan_number}` : `Produkt już w planie ${item.in_plan_number}`}
            >
              {item.in_current_plan ? "w tym planie" : `w planie ${item.in_plan_number}`}
            </Button>
          ) : (
            <Button
              variant="ghost"
              size="icon"
              className="h-[28px] w-[28px] p-0 bg-green-600 hover:bg-green-700 text-white flex-shrink-0 rounded-sm"
              onClick={(e) => {
                e.stopPropagation();
                onAddClick(item, orderNumber, buyerName, marketplace, setId, componentId);
              }}
              disabled={isAdding}
              data-testid={`button-add-${item.item_id}`}
              title="Dodaj do planu produkcyjnego"
            >
              <ArrowRight className="h-4 w-4" />
            </Button>
          )
        )}
      </div>
    </div>
  );
}
