/**
 * Material Breakdown Service - Rozkład materiałowy formatek
 * 
 * Oblicza zużycie płyty (m2), obrzeży (mb), kleju (g) i koszty
 * dla każdej formatki bazując na:
 * - Wymiarach z product_components
 * - Normatywach zużycia z production_material_consumption_norms
 * - Cenach z warehouse.materials
 */

import { Pool } from 'pg';

export interface MaterialBreakdownInput {
  componentId: number;
  width: number; // mm
  height: number; // mm
  thickness: number; // mm
  edgingPattern?: string; // JSON z oklejeniem (np. "[true, true, true, true]")
  boardColor?: string; // Kolor płyty (np. "CZARNY", "WOTAN")
  edgingColor?: string; // Kolor obrzeża (opcjonalne, domyślnie = boardColor)
}

export interface MaterialBreakdownResult {
  // Płyta
  boardAreaM2: number;
  boardWastePercentage: number;
  boardTotalM2: number;
  
  // Obrzeże
  edgingLengthMb: number;
  edgingOverageCm: number;
  edgingTotalMb: number;
  
  // Klej
  adhesiveGramsPerMb: number;
  adhesiveTotalG: number;
  
  // Ceny
  boardUnitPrice: number;
  edgingUnitPrice: number;
  adhesiveUnitPrice: number;
  
  // Koszty
  boardCost: number;
  edgingCost: number;
  adhesiveCost: number;
  totalCost: number;
}

/**
 * Pobiera normatywy zużycia materiałów
 */
async function getMaterialNorms(pool: Pool) {
  const result = await pool.query(`
    SELECT 
      material_category,
      waste_percentage,
      edge_overage_cm AS overage_value,
      input_unit AS overage_unit,
      consumption_per_unit AS consumption_value,
      output_unit AS consumption_unit
    FROM production.production_material_consumption_norms
    WHERE is_active = true
    ORDER BY material_category
  `);
  
  const norms: Record<string, any> = {};
  for (const row of result.rows) {
    norms[row.material_category] = row;
  }
  
  return norms;
}

/**
 * Normalizuje kolor komponentu do formatu warehouse.materials
 * "Czarny mat" → "CZARNY"
 * "6" → "6"
 * "WOTAN" → "WOTAN"
 */
function normalizeColor(color: string | undefined | null): string | undefined {
  if (!color) return undefined;
  
  // Usuń whitespace, uppercase, zamień spacje na underscore
  return color.trim().toUpperCase().replace(/\s+/g, '_');
}

/**
 * Pobiera cenę materiału z magazynu z 3-step lookup
 * 1. Exact token match (np. "18_CZARNY")
 * 2. ILIKE pattern match (np. "%CZARNY%") - prioritize shortest name (closest match)
 * 3. Fallback na średnią kategorii
 */
async function getMaterialPrice(
  pool: Pool,
  category: string,
  color?: string,
  thickness?: string,
  defaultPrice: number = 0
): Promise<{ price: number; matchType: string; materialName?: string }> {
  // Normalizuj category do lowercase i color do uppercase
  const normalizedCategory = category.toLowerCase();
  const normalizedColor = normalizeColor(color);
  
  // Step 1: Exact token match z thickness
  if (normalizedColor && thickness) {
    const exactResult = await pool.query(`
      SELECT m.price, m.name
      FROM warehouse.materials m
      JOIN warehouse.material_groups g ON m.group_id = g.id
      WHERE LOWER(g.category) = $1
        AND m.name = $2
        AND m.is_active = true
        AND m.price IS NOT NULL
      LIMIT 1
    `, [normalizedCategory, `${thickness}_${normalizedColor}`]);
    
    if (exactResult.rows[0]?.price) {
      return {
        price: parseFloat(exactResult.rows[0].price),
        matchType: 'exact',
        materialName: exactResult.rows[0].name
      };
    }
  }
  
  // Step 2: ILIKE pattern match z kolorem - prioritize shortest name (closest match)
  if (normalizedColor) {
    const patternResult = await pool.query(`
      SELECT m.price, m.name
      FROM warehouse.materials m
      JOIN warehouse.material_groups g ON m.group_id = g.id
      WHERE LOWER(g.category) = $1
        AND m.name ILIKE $2
        AND m.is_active = true
        AND m.price IS NOT NULL
      ORDER BY LENGTH(m.name) ASC, m.name ASC
      LIMIT 1
    `, [normalizedCategory, `%${normalizedColor}%`]);
    
    if (patternResult.rows[0]?.price) {
      return {
        price: parseFloat(patternResult.rows[0].price),
        matchType: 'pattern',
        materialName: patternResult.rows[0].name
      };
    }
  }
  
  // Step 3: Fallback na średnią kategorii
  const avgResult = await pool.query(`
    SELECT AVG(m.price::numeric) as avg_price
    FROM warehouse.materials m
    JOIN warehouse.material_groups g ON m.group_id = g.id
    WHERE LOWER(g.category) = $1
      AND m.is_active = true
      AND m.price IS NOT NULL
  `, [normalizedCategory]);
  
  if (avgResult.rows[0]?.avg_price) {
    return {
      price: parseFloat(avgResult.rows[0].avg_price),
      matchType: 'category_avg',
      materialName: undefined
    };
  }
  
  // Ultimate fallback
  console.warn(`⚠️  No price found for category "${category}" (color: ${color || 'none'}), using default: ${defaultPrice}`);
  return {
    price: defaultPrice,
    matchType: 'default',
    materialName: undefined
  };
}

/**
 * Pobiera ceny materiałów z magazynu z color-aware matching
 */
async function getMaterialPrices(
  pool: Pool,
  boardColor?: string,
  edgingColor?: string,
  thickness: string = '18'
) {
  // Płyta meblowa
  const boardResult = await getMaterialPrice(pool, 'plyty', boardColor, thickness, 50);
  if (boardResult.matchType !== 'exact') {
    console.log(`📋 Płyta: ${boardResult.matchType} match for color="${boardColor || 'none'}" → ${boardResult.price} zł/m² ${boardResult.materialName ? `(${boardResult.materialName})` : ''}`);
  }
  
  // Obrzeże
  const edgingResult = await getMaterialPrice(pool, 'obrzeza', edgingColor || boardColor, undefined, 2);
  if (edgingResult.matchType !== 'exact') {
    console.log(`📐 Obrzeże: ${edgingResult.matchType} match for color="${edgingColor || boardColor || 'none'}" → ${edgingResult.price} zł/mb ${edgingResult.materialName ? `(${edgingResult.materialName})` : ''}`);
  }
  
  // Klej (zawsze średnia, sprawdzaj wszystkie warianty nazwy kategorii)
  let adhesiveResult = await getMaterialPrice(pool, 'kleje', undefined, undefined, 0);
  // Kontynuuj próby jeśli nie znaleziono lub cena = 0
  if (!adhesiveResult || adhesiveResult.price === 0 || adhesiveResult.matchType === 'default') {
    adhesiveResult = await getMaterialPrice(pool, 'klej', undefined, undefined, 0);
  }
  if (!adhesiveResult || adhesiveResult.price === 0 || adhesiveResult.matchType === 'default') {
    adhesiveResult = await getMaterialPrice(pool, 'adhesives', undefined, undefined, 0.05);
  }
  
  return {
    boardPrice: String(boardResult.price),
    edgingPrice: String(edgingResult.price),
    adhesivePrice: String(adhesiveResult.price),
  };
}



/**
 * Oblicza powierzchnię płyty w m2
 */
function calculateBoardArea(width: number, height: number): number {
  // width i height w mm, wynik w m2
  const areaM2 = (width * height) / 1_000_000;
  return Math.round(areaM2 * 10000) / 10000; // zaokrąglenie do 4 miejsc
}

/**
 * Oblicza długość obrzeży w mb (metry bieżące)
 */
function calculateEdgingLength(width: number, height: number, edgingPattern?: string): number {
  let edges = [true, true, true, true];
  
  if (edgingPattern && typeof edgingPattern === 'string') {
    if (edgingPattern.includes('[')) {
      try {
        edges = JSON.parse(edgingPattern);
      } catch {
        edges = edgingPattern.split('').slice(0, 4).map(ch => ch.toUpperCase() === 'T');
      }
    } else {
      edges = edgingPattern.split('').slice(0, 4).map(ch => ch.toUpperCase() === 'T');
    }
    
    while (edges.length < 4) {
      edges.push(false);
    }
  }
  
  let lengthMm = 0;
  if (edges[0]) lengthMm += width;
  if (edges[1]) lengthMm += height;
  if (edges[2]) lengthMm += width;
  if (edges[3]) lengthMm += height;
  
  const lengthMb = lengthMm / 1000;
  return Math.round(lengthMb * 100) / 100;
}


/**
 * Główna funkcja obliczająca rozkład materiałowy i koszty
 */
export async function calculateMaterialBreakdown(
  pool: Pool,
  input: MaterialBreakdownInput
): Promise<MaterialBreakdownResult> {
  // 1. Pobierz normatywy
  const norms = await getMaterialNorms(pool);
  
  const boardNorm = norms['boards'] || norms['Plyty'] || { waste_percentage: '20', overage_value: '0' };
  const edgingNorm = norms['edging'] || norms['Obrzeza'] || { overage_value: '2', overage_unit: 'cm' };
  const adhesiveNorm = norms['adhesive'] || norms['Klej'] || { consumption_value: '50', consumption_unit: 'g/mb' };
  
  // 2. Pobierz ceny z color-aware matching
  const prices = await getMaterialPrices(
    pool,
    input.boardColor,
    input.edgingColor,
    String(Math.round(input.thickness))
  );
  
  // 3. Oblicz powierzchnię płyty
  const boardAreaM2 = calculateBoardArea(input.width, input.height);
  const boardWastePercentage = parseFloat(boardNorm.waste_percentage || '20');
  const boardTotalM2 = boardAreaM2 * (1 + boardWastePercentage / 100);
  
  // 4. Oblicz długość obrzeży
  const edgingLengthMb = calculateEdgingLength(input.width, input.height, input.edgingPattern);
  const edgingOverageCm = parseFloat(edgingNorm.overage_value || '2');
  const edgingTotalMb = edgingLengthMb + (edgingOverageCm / 100);
  
  // 5. Oblicz zużycie kleju
  const adhesiveGramsPerMb = parseFloat(adhesiveNorm.consumption_value || '50');
  const adhesiveTotalG = edgingTotalMb * adhesiveGramsPerMb;
  
  // 6. Oblicz koszty
  const boardUnitPrice = parseFloat(prices.boardPrice);
  const edgingUnitPrice = parseFloat(prices.edgingPrice);
  const adhesiveUnitPrice = parseFloat(prices.adhesivePrice);
  
  const boardCost = boardTotalM2 * boardUnitPrice;
  const edgingCost = edgingTotalMb * edgingUnitPrice;
  const adhesiveCost = adhesiveTotalG * adhesiveUnitPrice;
  const totalCost = boardCost + edgingCost + adhesiveCost;
  
  return {
    boardAreaM2: Math.round(boardAreaM2 * 10000) / 10000,
    boardWastePercentage,
    boardTotalM2: Math.round(boardTotalM2 * 10000) / 10000,
    
    edgingLengthMb: Math.round(edgingLengthMb * 100) / 100,
    edgingOverageCm,
    edgingTotalMb: Math.round(edgingTotalMb * 100) / 100,
    
    adhesiveGramsPerMb,
    adhesiveTotalG: Math.round(adhesiveTotalG * 100) / 100,
    
    boardUnitPrice: Math.round(boardUnitPrice * 100) / 100,
    edgingUnitPrice: Math.round(edgingUnitPrice * 100) / 100,
    adhesiveUnitPrice: Math.round(adhesiveUnitPrice * 1000) / 1000,
    
    boardCost: Math.round(boardCost * 100) / 100,
    edgingCost: Math.round(edgingCost * 100) / 100,
    adhesiveCost: Math.round(adhesiveCost * 100) / 100,
    totalCost: Math.round(totalCost * 100) / 100,
  };
}

/**
 * Oblicza i zapisuje breakdown dla komponentu
 */
export async function saveComponentMaterialBreakdown(
  pool: Pool,
  componentId: number
): Promise<number> {
  // 1. Pobierz dane komponentu
  const componentResult = await pool.query(`
    SELECT 
      id,
      calculated_length,
      calculated_width,
      thickness,
      edging_pattern,
      color
    FROM bom.product_components
    WHERE id = $1
  `, [componentId]);
  
  if (componentResult.rows.length === 0) {
    throw new Error(`Component ${componentId} not found`);
  }
  
  const component = componentResult.rows[0];
  
  // 2. Oblicz breakdown z kolorem
  const breakdown = await calculateMaterialBreakdown(pool, {
    componentId,
    width: parseFloat(component.calculated_length || '0'),
    height: parseFloat(component.calculated_width || '0'),
    thickness: parseFloat(component.thickness || '18'),
    edgingPattern: component.edging_pattern,
    boardColor: component.color,
  });
  
  // 3. Usuń stary breakdown (jeśli istnieje)
  await pool.query(
    'DELETE FROM bom.component_material_breakdown WHERE component_id = $1',
    [componentId]
  );
  
  // 4. Zapisz nowy breakdown
  const insertResult = await pool.query(`
    INSERT INTO bom.component_material_breakdown (
      component_id,
      board_area_m2, board_waste_percentage, board_total_m2,
      edging_length_mb, edging_overage_cm, edging_total_mb,
      adhesive_grams_per_mb, adhesive_total_g,
      board_unit_price, edging_unit_price, adhesive_unit_price,
      board_cost, edging_cost, adhesive_cost, total_cost,
      calculated_at
    ) VALUES (
      $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, CURRENT_TIMESTAMP
    )
    RETURNING id
  `, [
    componentId,
    breakdown.boardAreaM2, breakdown.boardWastePercentage, breakdown.boardTotalM2,
    breakdown.edgingLengthMb, breakdown.edgingOverageCm, breakdown.edgingTotalMb,
    breakdown.adhesiveGramsPerMb, breakdown.adhesiveTotalG,
    breakdown.boardUnitPrice, breakdown.edgingUnitPrice, breakdown.adhesiveUnitPrice,
    breakdown.boardCost, breakdown.edgingCost, breakdown.adhesiveCost, breakdown.totalCost,
  ]);
  
  return insertResult.rows[0].id;
}

/**
 * Pobiera breakdown dla komponentu
 */
export async function getComponentMaterialBreakdown(
  pool: Pool,
  componentId: number
) {
  const result = await pool.query(`
    SELECT * FROM bom.component_material_breakdown
    WHERE component_id = $1
    ORDER BY calculated_at DESC
    LIMIT 1
  `, [componentId]);
  
  const row = result.rows[0];
  if (!row) return null;
  
  // Convert decimal/numeric columns to numbers for frontend
  return {
    ...row,
    board_area_m2: row.board_area_m2 ? parseFloat(row.board_area_m2) : null,
    board_waste_percentage: row.board_waste_percentage ? parseFloat(row.board_waste_percentage) : null,
    board_total_m2: row.board_total_m2 ? parseFloat(row.board_total_m2) : null,
    edging_length_mb: row.edging_length_mb ? parseFloat(row.edging_length_mb) : null,
    edging_overage_cm: row.edging_overage_cm ? parseFloat(row.edging_overage_cm) : null,
    edging_total_mb: row.edging_total_mb ? parseFloat(row.edging_total_mb) : null,
    adhesive_grams_per_mb: row.adhesive_grams_per_mb ? parseFloat(row.adhesive_grams_per_mb) : null,
    adhesive_total_g: row.adhesive_total_g ? parseFloat(row.adhesive_total_g) : null,
    board_unit_price: row.board_unit_price ? parseFloat(row.board_unit_price) : null,
    edging_unit_price: row.edging_unit_price ? parseFloat(row.edging_unit_price) : null,
    adhesive_unit_price: row.adhesive_unit_price ? parseFloat(row.adhesive_unit_price) : null,
    board_cost: row.board_cost ? parseFloat(row.board_cost) : null,
    edging_cost: row.edging_cost ? parseFloat(row.edging_cost) : null,
    adhesive_cost: row.adhesive_cost ? parseFloat(row.adhesive_cost) : null,
    total_cost: row.total_cost ? parseFloat(row.total_cost) : null,
  };
}
