import axios from 'axios';
import fs from 'fs/promises';
import path from 'path';

const ALLEGRO_API_BASE = 'https://api.allegro.pl';
const ALLEGRO_AUTH_BASE = 'https://allegro.pl/auth/oauth';

let cachedToken: {
  token: string;
  expiresAt: Date;
} | null = null;

export async function getAllegroToken(): Promise<string> {
  const clientId = process.env.ALLEGRO_CLIENT_ID;
  const clientSecret = process.env.ALLEGRO_CLIENT_SECRET;

  if (!clientId || !clientSecret) {
    throw new Error('Allegro credentials not configured');
  }

  if (cachedToken && cachedToken.expiresAt > new Date()) {
    return cachedToken.token;
  }

  try {
    const response = await axios.post(
      `${ALLEGRO_AUTH_BASE}/token`,
      new URLSearchParams({
        grant_type: 'client_credentials',
      }),
      {
        auth: {
          username: clientId,
          password: clientSecret,
        },
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
        },
      }
    );

    const { access_token, expires_in } = response.data;
    const expiresAt = new Date(Date.now() + (expires_in - 60) * 1000);

    cachedToken = {
      token: access_token,
      expiresAt,
    };

    console.log('✅ Allegro token obtained, expires at:', expiresAt);
    return access_token;
  } catch (error: any) {
    console.error('❌ Failed to get Allegro token:', error.response?.data || error.message);
    throw new Error('Failed to authenticate with Allegro API');
  }
}

export async function searchProductByName(productName: string, userToken?: string): Promise<any> {
  const token = userToken || await getAllegroToken();

  try {
    const response = await axios.get(
      `${ALLEGRO_API_BASE}/sale/products`,
      {
        headers: {
          Authorization: `Bearer ${token}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        },
        params: {
          phrase: productName,
          'language': 'pl-PL',
          limit: 1
        }
      }
    );

    return response.data;
  } catch (error: any) {
    console.error(`❌ Failed to search product "${productName}":`, error.response?.data || error.message);
    throw new Error(`Failed to search product: ${error.message}`);
  }
}

export async function fetchPaymentRefunds(orderId: string, userToken: string): Promise<any[]> {
  try {
    const response = await axios.get(
      `${ALLEGRO_API_BASE}/payments/refunds`,
      {
        headers: {
          Authorization: `Bearer ${userToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        },
        params: {
          'order.id': orderId,
        }
      }
    );

    return response.data.refunds || [];
  } catch (error: any) {
    console.error(`❌ Failed to fetch payment refunds for order ${orderId}:`, error.response?.data || error.message);
    return [];
  }
}

export async function getAllegroOfferDetails(offerId: string, userToken: string): Promise<any> {
  try {
    const response = await axios.get(
      `${ALLEGRO_API_BASE}/sale/product-offers/${offerId}`,
      {
        headers: {
          Authorization: `Bearer ${userToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        }
      }
    );

    return response.data;
  } catch (error: any) {
    console.error(`❌ Failed to get offer details for ${offerId}:`, error.response?.data || error.message);
    return null;
  }
}

export async function getUserAllegroToken(): Promise<string | null> {
  try {
    const { getAllegroConnectionFromPostgres } = await import('./postgres.js');
    const connection = await getAllegroConnectionFromPostgres();
    
    if (!connection || !connection.access_token) {
      console.log('⚠️ No user Allegro token found in database');
      return null;
    }
    
    return connection.access_token;
  } catch (error) {
    console.error('❌ Failed to get user Allegro token:', error);
    return null;
  }
}

export async function searchSellerOffers(ean: string, productName: string): Promise<any> {
  // Endpoint /sale/offers wymaga user tokenu OAuth (nie client_credentials)
  const userToken = await getUserAllegroToken();
  
  if (!userToken) {
    console.log('⚠️ Cannot search seller offers - no user OAuth token');
    return null;
  }
  
  try {
    // Wyszukaj w aktywnych ofertach sprzedawcy
    const response = await axios.get(
      `${ALLEGRO_API_BASE}/sale/offers`,
      {
        headers: {
          Authorization: `Bearer ${userToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        },
        params: {
          'ean': ean,
          'publication.status': 'ACTIVE',
          limit: 20
        }
      }
    );
    
    const offers = response.data.offers || [];
    console.log(`🔍 Found ${offers.length} seller offers for EAN: ${ean}`);
    
    // Zwróć pierwszą ofertę która ma zdjęcia
    const offerWithImages = offers.find((o: any) => o.images && o.images.length > 0);
    return offerWithImages || (offers.length > 0 ? offers[0] : null);
  } catch (error: any) {
    console.error(`❌ Failed to search seller offers for EAN "${ean}":`, error.response?.data || error.message);
    return null;
  }
}

export async function downloadProductImage(
  productName: string,
  externalId: string,
  userToken?: string
): Promise<string | null> {
  try {
    const token = userToken || await getAllegroToken();
    
    let imageUrl: string | null = null;
    let searchMethod = '';
    
    // STRATEGIA 1: Szukaj w ofertach sprzedawcy (najlepsze dla produktów custom)
    try {
      const offer = await searchSellerOffers(externalId, productName);
      
      if (offer && offer.images && offer.images.length > 0) {
        imageUrl = offer.images[0].url;
        searchMethod = 'seller offer by EAN';
        console.log(`✅ Found image in seller offer (EAN: ${externalId})`);
      }
    } catch (error) {
      console.log(`⚠️ Seller offer search failed, trying catalog...`);
    }
    
    // STRATEGIA 2: Jeśli nie znaleziono w ofertach, szukaj w katalogu produktów
    if (!imageUrl) {
      try {
        const catalogResponse = await axios.get(
          `${ALLEGRO_API_BASE}/sale/products`,
          {
            headers: {
              Authorization: `Bearer ${token}`,
              Accept: 'application/vnd.allegro.public.v1+json',
            },
            params: {
              'phrase': externalId,
              'language': 'pl-PL',
              limit: 5
            }
          }
        );
        
        const products = catalogResponse.data.products || [];
        const exactMatch = products.find((p: any) => {
          const productEans = p.eans || [];
          return productEans.includes(externalId);
        });
        
        if (exactMatch && exactMatch.images && exactMatch.images.length > 0) {
          imageUrl = exactMatch.images[0].url;
          searchMethod = 'product catalog by EAN';
          console.log(`✅ Found image in product catalog (EAN: ${externalId})`);
        }
      } catch (error) {
        console.log(`⚠️ Catalog search failed`);
      }
    }
    
    // STRATEGIA 3: Ostatnia szansa - szukaj po nazwie
    if (!imageUrl) {
      try {
        const nameSearchResult = await searchProductByName(productName, token);
        const products = nameSearchResult.products || [];
        
        if (products.length > 0 && products[0].images && products[0].images.length > 0) {
          imageUrl = products[0].images[0].url;
          searchMethod = 'product catalog by name';
          console.log(`✅ Found image by product name`);
        }
      } catch (error) {
        console.log(`⚠️ Name search failed`);
      }
    }

    if (!imageUrl) {
      console.log(`⚠️ No images found for EAN "${externalId}" or name "${productName}"`);
      return null;
    }
    
    console.log(`📦 Found image using: ${searchMethod}`);
    
    const imageResponse = await axios.get(imageUrl, {
      responseType: 'arraybuffer',
    });

    const productsDir = path.join(process.cwd(), 'attached_assets', 'products');
    await fs.mkdir(productsDir, { recursive: true });

    const filename = `${externalId}_1.jpg`;
    const filepath = path.join(productsDir, filename);

    await fs.writeFile(filepath, imageResponse.data);
    console.log(`✅ Image saved: ${filename}`);

    return filename;
  } catch (error: any) {
    console.error(`❌ Failed to download image for product "${productName}":`, error.message);
    return null;
  }
}

export async function downloadProductImages(
  items: Array<{ productName: string; externalId: string }>,
  userToken?: string
): Promise<{ success: number; failed: number }> {
  let success = 0;
  let failed = 0;

  for (const item of items) {
    try {
      const result = await downloadProductImage(item.productName, item.externalId, userToken);
      if (result) {
        const { updateProductImageUrl } = await import('./postgres.js');
        await updateProductImageUrl(item.externalId, result);
        success++;
      } else {
        failed++;
      }
      await new Promise((resolve) => setTimeout(resolve, 200));
    } catch (error) {
      failed++;
    }
  }

  return { success, failed };
}

export async function downloadImageFromUrl(
  imageUrl: string,
  externalId: string
): Promise<string | null> {
  try {
    if (!imageUrl) {
      console.log(`⚠️ No image URL provided for product ${externalId}`);
      return null;
    }

    const imageResponse = await axios.get(imageUrl, {
      responseType: 'arraybuffer',
      timeout: 10000,
    });

    const productsDir = path.join(process.cwd(), 'attached_assets', 'products');
    await fs.mkdir(productsDir, { recursive: true });

    const filename = `${externalId}_1.jpg`;
    const filepath = path.join(productsDir, filename);

    await fs.writeFile(filepath, imageResponse.data);
    console.log(`✅ Image saved: ${filename}`);

    return filename;
  } catch (error: any) {
    console.error(`❌ Failed to download image for product ${externalId}:`, error.message);
    return null;
  }
}

export async function downloadProductImagesFromDatabase(
  items: Array<{ imageUrl: string | null; externalId: string }>
): Promise<{ success: number; failed: number; skipped: number }> {
  let success = 0;
  let failed = 0;
  let skipped = 0;

  const itemsWithUrls = items.filter(item => item.imageUrl);
  console.log(`📊 Processing ${itemsWithUrls.length}/${items.length} products with image URLs`);

  for (const item of itemsWithUrls) {
    try {
      const result = await downloadImageFromUrl(item.imageUrl!, item.externalId);
      if (result) {
        success++;
      } else {
        failed++;
      }
      
      await new Promise((resolve) => setTimeout(resolve, 100));
    } catch (error) {
      console.error(`❌ Error processing product ${item.externalId}:`, error);
      failed++;
    }
  }

  skipped = items.length - itemsWithUrls.length;
  if (skipped > 0) {
    console.log(`⏭️ Skipped ${skipped} products without image URLs`);
  }

  return { success, failed, skipped };
}

export async function fetchAllegroPaymentOperations(
  userToken: string,
  dateFrom?: string,
  dateTo?: string
): Promise<any> {
  try {
    const params: any = {
      limit: 100,
    };

    if (dateFrom) {
      params['occurredAt.gte'] = dateFrom;
    }
    if (dateTo) {
      params['occurredAt.lte'] = dateTo;
    }

    const response = await axios.get(
      `${ALLEGRO_API_BASE}/payments/payment-operations`,
      {
        headers: {
          Authorization: `Bearer ${userToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        },
        params
      }
    );

    return response.data;
  } catch (error: any) {
    console.error('❌ Failed to fetch Allegro payment operations:', error.response?.data || error.message);
    throw new Error(`Failed to fetch payment operations: ${error.message}`);
  }
}

export async function fetchAllegroBalance(userToken: string): Promise<any> {
  try {
    const response = await axios.get(
      `${ALLEGRO_API_BASE}/payments/payment-operations`,
      {
        headers: {
          Authorization: `Bearer ${userToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        },
        params: {
          limit: 100
        }
      }
    );

    const operations = response.data.paymentOperations || [];
    
    const walletMap = new Map<string, any>();
    
    for (const op of operations) {
      const operator = op.wallet?.paymentOperator || 'UNKNOWN';
      const walletType = op.wallet?.type || 'UNKNOWN';
      const currency = op.wallet?.balance?.currency || 'PLN';
      const key = `${operator}-${walletType}-${currency}`;
      
      if (!walletMap.has(key)) {
        walletMap.set(key, op);
      }
    }

    const balancesByWallet: {
      available: { [currency: string]: number };
      pending: { [currency: string]: number };
      holds: { [currency: string]: number };
    } = {
      available: {},
      pending: {},
      holds: {}
    };

    walletMap.forEach((op, key) => {
      const walletType = op.wallet?.type;
      const currency = op.wallet?.balance?.currency || 'PLN';
      const amount = parseFloat(op.wallet?.balance?.amount || '0');

      if (walletType === 'AVAILABLE') {
        balancesByWallet.available[currency] = (balancesByWallet.available[currency] || 0) + amount;
      } else if (walletType === 'WAITING') {
        balancesByWallet.pending[currency] = (balancesByWallet.pending[currency] || 0) + amount;
      }
    });

    console.log('💰 Final balances:', JSON.stringify(balancesByWallet, null, 2));

    return {
      available: balancesByWallet.available,
      pending: balancesByWallet.pending,
      holds: balancesByWallet.holds
    };
  } catch (error: any) {
    console.error('❌ Failed to fetch Allegro balance:', error.message);
    throw error;
  }
}

export async function fetchAllegroFees(userToken: string, days: number = 30): Promise<any> {
  try {
    const endDate = new Date();
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);

    // Pobierz WSZYSTKIE wpisy z paginacją (Allegro limit = 100 na stronę)
    let allEntries: any[] = [];
    let offset = 0;
    const limit = 100;
    let hasMore = true;

    while (hasMore) {
      const response = await axios.get(
        `${ALLEGRO_API_BASE}/billing/billing-entries`,
        {
          headers: {
            Authorization: `Bearer ${userToken}`,
            Accept: 'application/vnd.allegro.public.v1+json',
          },
          params: {
            'occurredAt.gte': startDate.toISOString(),
            'occurredAt.lte': endDate.toISOString(),
            limit,
            offset
          }
        }
      );

      const entries = response.data.billingEntries || [];
      allEntries = allEntries.concat(entries);
      
      console.log(`📄 Pobrano ${entries.length} wpisów billing (offset: ${offset}, total: ${allEntries.length})`);
      
      // Jeśli otrzymano mniej niż limit, to koniec
      if (entries.length < limit) {
        hasMore = false;
      } else {
        offset += limit;
      }
    }

    console.log(`✅ Łącznie pobrано ${allEntries.length} wpisów billing za ostatnie ${days} dni`);
    
    // Filtruj tylko wpisy z dzisiejszego dnia (od północy)
    const today = new Date();
    const midnight = new Date(today.getFullYear(), today.getMonth(), today.getDate(), 0, 0, 0, 0);
    
    const todayEntries = allEntries.filter(entry => {
      const occurredAt = entry.occurredAt ? new Date(entry.occurredAt) : null;
      return occurredAt && occurredAt >= midnight;
    });
    
    console.log(`🕐 Filtrowanie tylko opłat z dzisiejszego dnia (od ${midnight.toLocaleString('pl-PL')})`);
    console.log(`📊 Opłat z dzisiaj: ${todayEntries.length} / ${allEntries.length} (wszystkich)`);
    
    const entries = todayEntries;
    
    const fees = {
      commission: 0,
      listing: 0,
      ads: 0,
      delivery: 0,
      refunds: 0,
      other: 0,
      total: 0
    };

    // Liczniki (ile sztuk każdego typu)
    const counts = {
      commission: 0,
      listing: 0,
      ads: 0,
      adsBreakdown: { // Rozpisanie reklam
        NSP: 0,  // Kampanie Ads
        ADS: 0,  // Inne reklamy
        PRO: 0,  // Promocje
        FHB: 0,  // Pakiet Promo
        FSF: 0   // Oferty wyróżnione
      },
      delivery: 0,
      refunds: 0,
      other: 0
    };

    for (const entry of entries) {
      const typeId = entry.type?.id || '';
      const typeName = entry.type?.name || '';
      const amount = Math.abs(parseFloat(entry.value?.amount || '0'));
      const occurredAt = entry.occurredAt ? new Date(entry.occurredAt).toLocaleString('pl-PL') : 'brak daty';
      
      if (amount === 0) continue;

      console.log(`💸 [${occurredAt}] Fee entry: ${typeId} (${typeName}) = ${amount} PLN`);

      switch (typeId) {
        case 'SUC':
          fees.commission += amount;
          counts.commission++;
          break;
        case 'LIS':
          fees.listing += amount;
          counts.listing++;
          break;
        case 'NSP': // Opłata za kampanię Ads
        case 'ADS':
        case 'PRO':
        case 'FHB':
        case 'FSF':
          fees.ads += amount;
          counts.ads++;
          // Zlicz każdy typ reklamy osobno
          if (typeId === 'NSP') counts.adsBreakdown.NSP++;
          else if (typeId === 'ADS') counts.adsBreakdown.ADS++;
          else if (typeId === 'PRO') counts.adsBreakdown.PRO++;
          else if (typeId === 'FHB') counts.adsBreakdown.FHB++;
          else if (typeId === 'FSF') counts.adsBreakdown.FSF++;
          break;
        case 'HB1':
        case 'DEL':
        case 'DPD':
        case 'SHP':
        case 'DTR': // Opłata dodatkowa za dostawę DPD
          fees.delivery += amount;
          counts.delivery++;
          break;
        case 'REF':
        case 'RET':
          fees.refunds += amount;
          counts.refunds++;
          break;
        case 'PAD':
          fees.other += amount;
          counts.other++;
          break;
        default:
          fees.other += amount;
          counts.other++;
          console.log(`⚠️  Unknown fee type: ${typeId} (${typeName})`);
      }
      
      fees.total += amount;
    }

    return {
      fees,
      counts,
      period: {
        startDate: startDate.toISOString(),
        endDate: endDate.toISOString(),
        days
      }
    };
  } catch (error: any) {
    console.error('❌ Failed to fetch Allegro fees:', error.message);
    throw error;
  }
}
