import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { ArrowLeft, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { ProductionCarrierForm, carrierFormSchema, type CarrierFormData } from "@/features/production-carriers/production-carrier-form";
import type { ProductionCarrier, ProductionCarrierGroup } from "@shared/schema";
import { useEffect } from "react";

export default function ProductionCarrierEdit() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const { data: carrier, isLoading } = useQuery<ProductionCarrier>({
    queryKey: ["/api/production/carriers", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/carriers/${id}`);
      if (!res.ok) throw new Error("Failed to fetch carrier");
      return res.json();
    },
  });

  const { data: groups = [] } = useQuery<ProductionCarrierGroup[]>({
    queryKey: ["/api/production/carriers/groups"],
  });

  const form = useForm<CarrierFormData>({
    resolver: zodResolver(carrierFormSchema),
    defaultValues: {
      carrierGroupId: 0,
      code: "",
      name: "",
      barcode: "",
      status: "available",
      capacity: null,
      capacityUnit: "",
      notes: "",
      isActive: true,
    },
  });

  useEffect(() => {
    if (carrier) {
      form.reset({
        carrierGroupId: carrier.carrierGroupId,
        code: carrier.code,
        name: carrier.name,
        barcode: carrier.barcode ?? "",
        status: carrier.status ?? "available",
        capacity: carrier.capacity ? Number(carrier.capacity) : null,
        capacityUnit: carrier.capacityUnit ?? "",
        notes: carrier.notes ?? "",
        isActive: carrier.isActive ?? true,
      });
    }
  }, [carrier, form]);

  const updateMutation = useMutation({
    mutationFn: (data: CarrierFormData) => {
      return apiRequest(`/api/production/carriers/${id}`, "PATCH", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers", id] });
      toast({ title: "Nośnik został zaktualizowany" });
      navigate(`/production/carriers/${id}`);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować nośnika",
        variant: "destructive",
      });
    },
  });

  const handleCancel = () => {
    navigate(`/production/carriers/${id}`);
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="flex justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-carrier-edit" />
        </div>
      </div>
    );
  }

  if (!carrier) {
    return (
      <div className="container mx-auto p-6">
        <Card>
          <CardContent className="p-8 text-center text-muted-foreground">
            Nie znaleziono nośnika
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center gap-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={handleCancel}
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-page-title">
            Edycja nośnika
          </h1>
          <p className="text-muted-foreground font-mono">
            {carrier.code}
          </p>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane nośnika</CardTitle>
        </CardHeader>
        <CardContent>
          <ProductionCarrierForm
            form={form}
            onSubmit={(data) => updateMutation.mutate(data)}
            mutation={updateMutation}
            groups={groups}
            mode="edit"
            onCancel={handleCancel}
          />
        </CardContent>
      </Card>
    </div>
  );
}
