import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Loader2, Package, FolderOpen } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";

interface ProductAddon {
  id: number;
  addonType: string;
  name: string;
  code: string | null;
  description: string | null;
  htmlTemplate: string | null;
  isActive: boolean;
}

interface AccessoryGroup {
  id: number;
  name: string;
  code: string;
  description: string | null;
  category: string;
  itemCount?: number;
}

type PickerItem = ProductAddon | AccessoryGroup;

interface AddonPickerDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSelect: (addon: ProductAddon | { code: string; name: string; isGroup: true }) => void;
}

const ADDON_TYPE_LABELS: Record<string, string> = {
  fabric: "Tkanina",
  board: "Płyta",
  certificate: "Certyfikat",
  accessory: "Akcesoria",
  component: "Komponent",
  accessory_groups: "Grupy akcesoriów",
};

function isAccessoryGroup(item: PickerItem): item is AccessoryGroup {
  return 'category' in item;
}

export function AddonPickerDialog({
  open,
  onOpenChange,
  onSelect,
}: AddonPickerDialogProps) {
  const [selectedItem, setSelectedItem] = useState<PickerItem | null>(null);
  const [selectedType, setSelectedType] = useState<string>("all");

  // Fetch active addons
  const { data: addons, isLoading: addonsLoading } = useQuery<ProductAddon[]>({
    queryKey: ["/api/product-addons"],
    enabled: open,
  });

  // Fetch accessory groups
  const { data: accessoryGroups, isLoading: groupsLoading } = useQuery<AccessoryGroup[]>({
    queryKey: ["/api/accessory-groups"],
    enabled: open,
  });

  const isLoading = addonsLoading || groupsLoading;

  const activeAddons = addons?.filter((a) => a.isActive) || [];
  const activeGroups = accessoryGroups?.filter((g) => g) || [];
  
  // Combine items
  const allItems: PickerItem[] = [...activeAddons, ...activeGroups];
  
  const filteredItems =
    selectedType === "all"
      ? allItems
      : selectedType === "accessory_groups"
      ? activeGroups
      : activeAddons.filter((a) => a.addonType === selectedType);

  const addonTypes = Array.from(new Set(activeAddons.map((a) => a.addonType)));
  const hasGroups = activeGroups.length > 0;

  const handleSelect = () => {
    if (selectedItem) {
      if (isAccessoryGroup(selectedItem)) {
        // For accessory groups, return special object with tag
        onSelect({
          code: selectedItem.code,
          name: selectedItem.name,
          isGroup: true,
        });
      } else {
        // For regular addons, return as before
        onSelect(selectedItem);
      }
      onOpenChange(false);
      setSelectedItem(null);
    }
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-3xl max-h-[80vh]">
        <DialogHeader>
          <DialogTitle>Wybierz dodatek do wstawienia</DialogTitle>
          <DialogDescription>
            Wybierz dodatek aby wstawić jego treść HTML do opisu produktu
          </DialogDescription>
        </DialogHeader>

        {isLoading ? (
          <div className="flex items-center justify-center py-12">
            <Loader2 className="h-8 w-8 animate-spin" />
          </div>
        ) : allItems.length === 0 ? (
          <div className="text-center text-muted-foreground py-12">
            Brak dostępnych elementów
          </div>
        ) : (
          <>
            <Tabs value={selectedType} onValueChange={setSelectedType}>
              <TabsList className={`grid mb-4 ${hasGroups ? 'grid-cols-7' : 'grid-cols-6'}`}>
                <TabsTrigger value="all">Wszystkie</TabsTrigger>
                {hasGroups && (
                  <TabsTrigger value="accessory_groups">
                    {ADDON_TYPE_LABELS.accessory_groups}
                  </TabsTrigger>
                )}
                {addonTypes.map((type) => (
                  <TabsTrigger key={type} value={type}>
                    {ADDON_TYPE_LABELS[type] || type}
                  </TabsTrigger>
                ))}
              </TabsList>

              <TabsContent value={selectedType}>
                <ScrollArea className="h-[400px] w-full">
                  <div className="space-y-2 p-4">
                    {filteredItems.map((item) => {
                      const isGroup = isAccessoryGroup(item);
                      const itemId = `${isGroup ? 'group' : 'addon'}-${item.id}`;
                      const isSelected = selectedItem?.id === item.id;
                      
                      return (
                        <div
                          key={itemId}
                          className={`p-4 border rounded-md cursor-pointer transition-all ${
                            isSelected
                              ? "border-primary bg-accent"
                              : "hover:bg-accent/50"
                          }`}
                          onClick={() => setSelectedItem(item)}
                          data-testid={`item-option-${itemId}`}
                        >
                          <div className="flex items-start gap-3">
                            {isGroup ? (
                              <FolderOpen className="h-5 w-5 text-muted-foreground mt-1" />
                            ) : (
                              <Package className="h-5 w-5 text-muted-foreground mt-1" />
                            )}
                            <div className="flex-1 min-w-0">
                              <div className="flex items-center gap-2">
                                <h4 className="font-medium truncate">{item.name}</h4>
                                <Badge variant="outline" className="text-xs">
                                  {isGroup 
                                    ? `Grupa (${item.itemCount || 0} elementów)` 
                                    : ADDON_TYPE_LABELS[(item as ProductAddon).addonType] || (item as ProductAddon).addonType
                                  }
                                </Badge>
                              </div>
                              {isGroup ? (
                                <>
                                  <p className="text-xs text-muted-foreground mt-1">
                                    Tag: <code className="bg-muted px-1">{'{{akcesorium-' + item.code + '}}'}</code>
                                  </p>
                                  {item.description && (
                                    <p className="text-sm text-muted-foreground mt-2">
                                      {item.description}
                                    </p>
                                  )}
                                </>
                              ) : (
                                <>
                                  {(item as ProductAddon).code && (
                                    <p className="text-xs text-muted-foreground mt-1">
                                      Kod: {(item as ProductAddon).code}
                                    </p>
                                  )}
                                  {(item as ProductAddon).description && (
                                    <p className="text-sm text-muted-foreground mt-2">
                                      {(item as ProductAddon).description}
                                    </p>
                                  )}
                                  {(item as ProductAddon).htmlTemplate && (
                                    <div className="mt-2 p-2 bg-muted rounded text-xs">
                                      <div
                                        className="line-clamp-2"
                                        dangerouslySetInnerHTML={{
                                          __html: (item as ProductAddon).htmlTemplate!.substring(0, 100),
                                        }}
                                      />
                                    </div>
                                  )}
                                </>
                              )}
                            </div>
                          </div>
                        </div>
                      );
                    })}
                  </div>
                </ScrollArea>
              </TabsContent>
            </Tabs>

            <DialogFooter>
              <Button
                type="button"
                variant="outline"
                onClick={() => onOpenChange(false)}
                data-testid="button-cancel-addon-picker"
              >
                Anuluj
              </Button>
              <Button
                type="button"
                onClick={handleSelect}
                disabled={!selectedItem}
                data-testid="button-insert-addon"
              >
                {selectedItem && isAccessoryGroup(selectedItem) ? 'Wstaw tag grupy' : 'Wstaw dodatek'}
              </Button>
            </DialogFooter>
          </>
        )}
      </DialogContent>
    </Dialog>
  );
}
