# 🐛 Problem: Niepoprawna Data Zamówienia w Odoo

**Data:** 2025-10-12  
**Status:** 🔴 AKTYWNY PROBLEM  
**Priorytet:** WYSOKI ⚠️

---

## 📋 Opis Problemu

### Problem główny:
**Data w polu `date_order` w Odoo jest niepoprawna - zamiast daty faktycznego złożenia zamówienia, wpisywana jest data przesłania/synchronizacji zamówienia do Odoo.**

### Dowód problemu:
Zrzut ekranu z Odoo pokazuje zamówienie S02945:
- **Order Date w Odoo:** 12 paź. 12:19
- **Oczekiwana data:** Data faktycznego złożenia zamówienia (z `commerce.orders.order_date`)

---

## 🔍 Ustalenia

### Co wiemy:

1. **Dane w PostgreSQL (`commerce.orders`)**
   - Pole: `order_date` 
   - Format: TIMESTAMP (UTC)
   - Źródła danych:
     - Allegro: `boughtAt` lub `updatedAt` (UTC)
     - Shoper: `date` lub `confirm_date` (konwertowane z Europe/Warsaw na UTC)

2. **Pobieranie z PostgreSQL**
   - Query: `TO_CHAR(o.order_date, 'YYYY-MM-DD HH24:MI:SS')::text as "orderDate"`
   - Lokalizacja: `server/odoo-sync.ts` linia 193
   - Format wyjściowy: String `YYYY-MM-DD HH24:MI:SS`

3. **Tworzenie zamówienia w Odoo (CREATE)**
   - Plik: `server/odoo-client.ts` linie 374-397
   - Kod:
     ```typescript
     const odooDateFormat = order.orderDate 
       ? (typeof order.orderDate === 'string' ? order.orderDate : order.orderDate.toISOString().replace('T', ' ').replace(/\.\d{3}Z$/, ''))
       : new Date().toISOString().replace('T', ' ').replace(/\.\d{3}Z$/, '');
     
     const orderData: OdooOrderData = {
       date_order: odooDateFormat,  // ← TUTAJ zapisywane
       ...
     };
     ```
   - ✅ Weryfikacja: TAK (linie 417-430)

4. **Aktualizacja zamówienia w Odoo (UPDATE)**
   - Plik: `server/odoo-sync.ts` linie 336-342
   - Kod:
     ```typescript
     if (order.orderDate) {
       updateData.date_order = typeof order.orderDate === 'string' 
         ? order.orderDate 
         : order.orderDate.toISOString().replace('T', ' ').replace(/\.\d{3}Z$/, '');
     }
     ```
   - ❌ Weryfikacja: NIE MA (brak sprawdzenia czy date_order zostało zapisane)

---

## ❓ Hipotezy do zbadania

### Hipoteza 1: Problem z fallbackiem w CREATE
```typescript
// Jeśli order.orderDate jest null/undefined, używana jest AKTUALNA data:
: new Date().toISOString().replace('T', ' ').replace(/\.\d{3}Z$/, '');
```
**Pytanie:** Czy `order.orderDate` może być `null/undefined` mimo że istnieje w bazie?

### Hipoteza 2: Problem z typem danych
**Pytanie:** Czy string z `TO_CHAR()` jest poprawnie przekazywany przez wszystkie warstwy?

### Hipoteza 3: Problem ze strefą czasową
**Pytanie:** Czy konwersja UTC → lokalny czas Odoo działa poprawnie?

---

## 🔬 Do zbadania

- [ ] Sprawdzić logi: czy `order.orderDate` ma wartość czy jest undefined?
- [ ] Sprawdzić: czy TO_CHAR zwraca poprawną datę z PostgreSQL?
- [ ] Sprawdzić: co faktycznie Odoo zapisuje w `date_order`?
- [ ] Sprawdzić: czy problem występuje w CREATE, UPDATE czy obu?

---

## 📝 Historia pracy nad problemem

### 2025-10-12 - Identyfikacja problemu
- Zauważono że Order Date w Odoo pokazuje datę synchronizacji zamiast daty zamówienia
- Przeanalizowano kod CREATE i UPDATE
- Zidentyfikowano brak weryfikacji w UPDATE
- Stworzono ten dokument

### 2025-10-12 10:31 - Analiza danych dla zamówienia #643 (S02945)

**Zbadano zamówienie #643:**

#### 1. Dane w PostgreSQL `commerce.orders`:
```sql
order_number: 643
source: ALLEGRO
source_order_id: ebc78111-a753-11f0-a5d7-371746f07634
order_date: 2025-10-12 10:13:04.898        ✅ POPRAWNA DATA ZAMÓWIENIA
created_at: 2025-10-12 10:13:23.112617     (zapisane do bazy)
updated_at: 2025-10-12 10:19:08.366046     (ostatnia aktualizacja)
odoo_order_id: 2836
```

#### 2. Dane w `odoo_sync_logs`:
```sql
operation: create
status: success
created_at: 2025-10-12 10:19:08            (wysłane do Odoo)
request_payload zawiera:
  "orderDate": "2025-10-12 10:13:04"       ✅ POPRAWNA DATA
  "date_order": BRAK!                      ❌ NIE MA W LOGU
```

#### 3. Co Odoo pokazuje:
- **Order Date w Odoo:** 12 paź. 12:19
- **To jest:** 10:19 UTC + 2h (strefa czasowa) = 12:19 lokalne
- **Czyli Odoo ma:** 2025-10-12 10:19:00 (czas synchronizacji, nie zamówienia!)

---

## 🎯 **KLUCZOWE ODKRYCIE:**

### Problem z logowaniem:
W `server/odoo-sync.ts` linia 363:
```typescript
const logPayload = operation === 'create' ? order : updateData;
```

**Logujemy:** obiekt `order` (który ma `orderDate`)  
**Wysyłamy do Odoo:** obiekt `orderData` (który ma `date_order`)

❌ **To wprowadza w błąd!** Nie widzimy co faktycznie idzie do Odoo!

### Główna hipoteza:
**W `createSaleOrder()` jest fallback:**
```typescript
const odooDateFormat = order.orderDate 
  ? (typeof order.orderDate === 'string' ? order.orderDate : ...)
  : new Date().toISOString()...  // ← UŻYWA BIEŻĄCEJ DATY!
```

**Pytanie:** Czy `order.orderDate` jakoś się "gubi" i używany jest fallback `new Date()`?

### Możliwe przyczyny:
1. ❓ `order.orderDate` jest `null/undefined` mimo że jest w bazie
2. ❓ Problem z typem - TO_CHAR zwraca string, ale coś go psuje
3. ❓ Błąd w parsowaniu/przekazywaniu przez warstwy
4. ❓ Odoo sam nadpisuje `date_order` podczas zapisu

---

## 🔬 Co trzeba zbadać dalej:

- [x] Sprawdzić dane w commerce.orders ✅ (order_date jest poprawne)
- [x] Sprawdzić logi synchronizacji ✅ (orderDate w payload jest poprawne)
- [ ] **KRYTYCZNE:** Dodać debug log PRZED wysłaniem do Odoo żeby zobaczyć co jest w `orderData.date_order`
- [ ] Sprawdzić czy `order.orderDate` jest string czy Date w momencie tworzenia `odooDateFormat`
- [ ] Zweryfikować co faktycznie Odoo ma zapisane w bazie (może przez Odoo shell)

---

## 🔄 Następne kroki

1. ✅ ~~Sprawdzić logi z ostatniej synchronizacji~~
2. ✅ ~~Zweryfikować wartości w bazie danych~~
3. ✅ ~~Dodać debug logi w `createSaleOrder()`~~
4. **TERAZ:** Uruchomić synchronizację testową i sprawdzić logi
5. Naprawić problem
6. Dodać weryfikację w UPDATE

---

### 2025-10-12 10:35 - Dodano debug logi

**Zmiany w `server/odoo-client.ts` (linie 380-396):**

Dodano szczegółowe logowanie:
```typescript
// 🔍 DEBUG: Sprawdź wartość order.orderDate
console.log(`🔍 [DEBUG orderDate] #${order.orderNumber}:`, {
  value: order.orderDate,
  type: typeof order.orderDate,
  isNull: order.orderDate === null,
  isUndefined: order.orderDate === undefined,
  isFalsy: !order.orderDate,
  isDate: order.orderDate instanceof Date
});

// 🔍 DEBUG: Sprawdź czy użyto fallbacku
const usedFallback = !order.orderDate;
console.log(`📅 [ODOO DATE] #${order.orderNumber}: ${odooDateFormat} ${usedFallback ? '⚠️ FALLBACK (new Date())' : '✅ (from order_date)'}`);
```

**Co to pokaże:**
- ✅ Dokładną wartość `order.orderDate`
- ✅ Typ danych (string/Date/undefined/null)
- ✅ Czy używany jest fallback `new Date()` (⚠️)
- ✅ Potwierdzi czy data pochodzi z `order_date` czy jest bieżąca

---

### 2025-10-12 10:35 - Testowanie debug logów

**Status:**
- ✅ Dodano zamówienie #643 do kolejki `odoo_sync_queue` (ID: 5907, operation: update)
- ⏳ Oczekiwanie na przetworzenie przez scheduled sync (1 min interval)
- 🔒 Advisory lock blokuje równoległe przetwarzanie (`pg_try_advisory_lock`)
- 📊 Endpoint /api/odoo/sync/manual wymaga autentykacji (401)

**Obserwacje:**
- System Odoo sync działa z intervalem 1 minuty
- Jest blokada advisory lock która zapobiega równoczesnemu przetwarzaniu
- Kolejne próby synchronizacji: pomijane z komunikatem "Inna instancja processSyncQueue() już działa"

**Następne kroki:**
- Poczekać na automatyczną synchronizację (~1 min)
- Sprawdzić logi czy pojawiły się debug informacje
- Zweryfikować czy fallback `new Date()` jest używany

---

### 2025-10-12 10:39 - KLUCZOWE USTALENIE ✅

**🔴 POTWIERDZONO ZAKRES PROBLEMU:**

1. **UPDATE operacje (aktualizacja istniejących zamówień):**
   - ✅ **DZIAŁA POPRAWNIE**
   - Data `order_date` jest prawidłowo aktualizowana
   - Kod w `server/odoo-sync.ts` linie 336-342 działa jak powinien

2. **CREATE operacje (tworzenie nowych zamówień):**
   - ❌ **PROBLEM WYSTĘPUJE**
   - Nowe zamówienie zapisywane jest z datą **dodania/synchronizacji** zamiast `order_date`
   - Problem jest w `server/odoo-client.ts` funkcji `createSaleOrder()`
   
**WNIOSEK:**
Problem NIE jest w bazie danych (tam `order_date` jest poprawne).  
Problem NIE jest w SQL query (TO_CHAR zwraca poprawną datę).  
Problem JEST w logice CREATE - prawdopodobnie używany jest fallback `new Date()` zamiast wartości z `order.orderDate`.

**Status testów:**
- Zamówienie #643 czeka na przetworzenie
- Serwer został zrestartowany (advisory lock został zwolniony)
- Debug logi pokażą dokładną przyczynę przy następnej synchronizacji

---

### 2025-10-12 10:58 - ROZWIĄZANIE ZAGADKI ✅

**🔍 Odkrycia użytkownika:**
Użytkownik zauważył że od godz. 12:00 dodano kilka zamówień (#642-#646) ale nie ma logów CREATE - zasugerował że może nie ma mechanizmu logowania.

**✅ USTALENIA:**

1. **Operacje w queue vs logi:**
   - Queue: wszystkie są `operation='update'` ✅
   - Logi: pierwsze wywołanie zapisane jako `'create'` ❌
   
2. **Przyczyna rozbieżności** (linia 254 w `odoo-sync.ts`):
   ```typescript
   const operation = hasOdooId ? 'update' : 'create';
   ```
   - Kod **ignoruje** operation z queue
   - Sam decyduje na podstawie obecności `odoo_order_id`
   - Jeśli brak `odoo_order_id` → wywołuje `createSaleOrder()` i loguje jako 'create'

3. **Zamówienia #642-646 faktycznie zostały UTWORZONE (CREATE):**
   ```
   #642: CREATE o 10:19:04 (przed debug logami 10:35) ❌
   #643: CREATE o 10:19:08 (przed debug logami 10:35) ❌  
   #644: CREATE o 10:46:00 (PO debug logach 10:35!) ✅
   #645: CREATE o 10:55:58 (PO debug logach 10:35!) ✅
   #646: CREATE o 10:51:02 (PO debug logach 10:35!) ✅
   ```

4. **Dlaczego nie ma debug logów dla #644, #645, #646:**
   - Serwer był offline/restart między 10:39-10:56
   - Brak plików logów z tego okresu
   - Logi CREATE zniknęły podczas restartu

**🎯 WNIOSEK KOŃCOWY:**

Problem NIE jest w mechanizmie logowania - debug logi działają prawidłowo!  
Problem był w tym że serwer był offline gdy zamówienia #644-646 były tworzone.

**GŁÓWNY PROBLEM NADAL AKTUALNY:**
- UPDATE działa ✅ - data jest poprawnie aktualizowana
- CREATE ma problem ❌ - data zapisywana jako czas synchronizacji zamiast order_date

**Potrzebujemy:**
1. Nowego zamówienia CREATE gdy serwer działa
2. Wtedy debug logi pokażą czy używany jest fallback `new Date()`

---

### 2025-10-12 11:02 - ZNALEZIONO PROBLEM ZE STREFAMI CZASOWYMI! 🕐

**🔍 Odkrycie użytkownika:**
Użytkownik zauważył że OMS pokazuje poprawny czas (12:49, 12:54, 12:44, 12:13) ale w bazie jest "minus 2 godziny" (10:49, 10:54, 10:44, 10:13).

**✅ ANALIZA:**

1. **PostgreSQL:**
   - Typ kolumny: `TIMESTAMP WITHOUT TIME ZONE` (brak informacji o strefie!)
   - Wartości w bazie: `2025-10-12 10:13:04` (UTC) ✅ POPRAWNE

2. **API (server/routes.ts linia 1525):**
   - Zwraca: `orderDate: order.order_date`
   - Format: `"2025-10-12 10:13:04"` (string BEZ "Z" na końcu!)

3. **Frontend (client/src/components/orders-table.tsx linia 396, 681):**
   ```typescript
   format(new Date(o.orderDate), "HH:mm", { locale: pl })
   ```
   - JavaScript `new Date("2025-10-12 10:13:04")` **traktuje to jako LOKALNY czas!**
   - W Polsce (UTC+2): pokazuje 10:13 LOCAL jako 12:13 (10:13 + 2h) ❌

**🎯 POPRAWNA DIAGNOZA (po korekcie użytkownika):**

**Zamówienie #643 - PRAWIDŁOWE WARTOŚCI:**
- **Złożone w Allegro:** 12:13 (czas polski lokalny) ✅
- **Baza PostgreSQL:** `2025-10-12 10:13:04` (UTC) ✅ **POPRAWNE!**
- **OMS wyświetla:** 12:13 ✅ **POPRAWNE!** (konwertuje 10:13 UTC → 12:13 lokalnie)
- **Odoo ma zapisane:** `2025-10-12 10:19:00` (UTC) ❌ **BŁĄD!**
- **Odoo wyświetla:** 12:19 ❌ **BŁĄD!** (10:19 UTC + 2h = 12:19 lokalnie)

**WNIOSEK:**
- ✅ Baza OMS działa POPRAWNIE - zapisuje UTC
- ✅ Frontend OMS działa POPRAWNIE - wyświetla czas lokalny
- ❌ **JEDYNY PROBLEM: Odoo CREATE zapisuje czas synchronizacji zamiast order_date!**

**CO SIĘ STAŁO:**
1. Zamówienie złożone: **12:13** (czas polski)
2. Zapisane w bazie: **10:13 UTC** ✅ (12:13 - 2h = 10:13 UTC)
3. Synchronizacja do Odoo: **10:19 UTC** (6 minut później)
4. Odoo zapisało: **10:19 UTC** ❌ (czas synchronizacji zamiast 10:13 UTC!)
5. Odoo wyświetla: **12:19** lokalnie ❌ (10:19 + 2h)

**GŁÓWNY PROBLEM:**
❌ Odoo CREATE zapisuje `new Date()` (czas synchronizacji) zamiast `order.orderDate`

---

**Kontynuacja pracy poniżej...**
