import { Pool, PoolClient } from "pg";
import type { ProductionRouting, InsertProductionRouting } from "@shared/schema";

export async function getRoutings(pool: Pool): Promise<ProductionRouting[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_routings 
    ORDER BY code ASC
  `);
  
  return result.rows.map(row => ({
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    productId: row.product_id,
    productType: row.product_type,
    defaultWorkCenterId: row.default_work_center_id,
    totalEstimatedTime: row.total_estimated_time,
    isActive: row.is_active,
    version: row.version,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getRoutingById(pool: Pool, id: number): Promise<ProductionRouting | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_routings WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    productId: row.product_id,
    productType: row.product_type,
    defaultWorkCenterId: row.default_work_center_id,
    totalEstimatedTime: row.total_estimated_time,
    isActive: row.is_active,
    version: row.version,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createRouting(pool: Pool, data: InsertProductionRouting): Promise<ProductionRouting> {
  const result = await pool.query(`
    INSERT INTO production.production_routings 
    (code, name, description, product_id, product_type, default_work_center_id, 
     total_estimated_time, is_active, version, notes)
    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
    RETURNING *
  `, [
    data.code,
    data.name,
    data.description ?? null,
    data.productId ?? null,
    data.productType ?? null,
    data.defaultWorkCenterId ?? null,
    data.totalEstimatedTime ?? 0,
    data.isActive ?? true,
    data.version ?? 1,
    data.notes ?? null,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    productId: row.product_id,
    productType: row.product_type,
    defaultWorkCenterId: row.default_work_center_id,
    totalEstimatedTime: row.total_estimated_time,
    isActive: row.is_active,
    version: row.version,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateRouting(pool: Pool, id: number, data: Partial<InsertProductionRouting>): Promise<ProductionRouting | null> {
  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.code !== undefined) {
    updates.push(`code = $${paramIndex++}`);
    values.push(data.code);
  }
  if (data.name !== undefined) {
    updates.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }
  if (data.description !== undefined) {
    updates.push(`description = $${paramIndex++}`);
    values.push(data.description);
  }
  if (data.productId !== undefined) {
    updates.push(`product_id = $${paramIndex++}`);
    values.push(data.productId);
  }
  if (data.productType !== undefined) {
    updates.push(`product_type = $${paramIndex++}`);
    values.push(data.productType);
  }
  if (data.defaultWorkCenterId !== undefined) {
    updates.push(`default_work_center_id = $${paramIndex++}`);
    values.push(data.defaultWorkCenterId);
  }
  if (data.totalEstimatedTime !== undefined) {
    updates.push(`total_estimated_time = $${paramIndex++}`);
    values.push(data.totalEstimatedTime);
  }
  if (data.isActive !== undefined) {
    updates.push(`is_active = $${paramIndex++}`);
    values.push(data.isActive);
  }
  if (data.version !== undefined) {
    updates.push(`version = $${paramIndex++}`);
    values.push(data.version);
  }
  if (data.notes !== undefined) {
    updates.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (updates.length === 0) {
    return getRoutingById(pool, id);
  }

  updates.push(`updated_at = NOW()`);
  values.push(id);

  const result = await pool.query(`
    UPDATE production.production_routings 
    SET ${updates.join(", ")}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);

  if (result.rows.length === 0) return null;

  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    productId: row.product_id,
    productType: row.product_type,
    defaultWorkCenterId: row.default_work_center_id,
    totalEstimatedTime: row.total_estimated_time,
    isActive: row.is_active,
    version: row.version,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteRouting(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(`
    DELETE FROM production.production_routings WHERE id = $1
  `, [id]);
  
  return result.rowCount !== null && result.rowCount > 0;
}

export async function recalculateTotalTime(client: PoolClient, routingId: number): Promise<number> {
  const result = await client.query(`
    SELECT COALESCE(SUM(estimated_time + COALESCE(setup_time, 0)), 0) AS total
    FROM production.production_routing_operations
    WHERE routing_id = $1
  `, [routingId]);
  
  const total = parseFloat(result.rows[0]?.total || '0');
  
  await client.query(`
    UPDATE production.production_routings
    SET total_estimated_time = $1, updated_at = NOW()
    WHERE id = $2
  `, [total, routingId]);
  
  return total;
}

export async function seedDefaultRoutings(pool: Pool): Promise<{ routingsInserted: number; operationsInserted: number }> {
  // Get work center IDs from database
  const workCenters = await pool.query(`
    SELECT id, code FROM production.production_work_centers
    WHERE code IN ('PILA-MW', 'OKL-FALIT', 'WIERT-M1', 'KOMPL', 'PAKOW')
  `);

  const workCenterMap = new Map(workCenters.rows.map(wc => [wc.code, wc.id]));

  // Seed routing
  const routingResult = await pool.query(`
    INSERT INTO production.production_routings 
    (code, name, description, product_type, is_active, version)
    VALUES 
      ('ROUT-SZAFKA-01', 'Produkcja szafki standardowej', 'Standardowa marszruta produkcji szafki meblowej', 'FURNITURE_CABINET', true, 1)
    ON CONFLICT (code) DO NOTHING
    RETURNING id
  `);

  let routingsInserted = routingResult.rowCount || 0;
  let operationsInserted = 0;

  if (routingsInserted > 0 && routingResult.rows.length > 0) {
    const routingId = routingResult.rows[0].id;

    // Seed operations for this routing
    const operationsResult = await pool.query(`
      INSERT INTO production.production_routing_operations
      (routing_id, sequence, code, name, description, work_center_id, estimated_time, setup_time, allow_parallel, instructions, quality_check_required, is_active)
      VALUES
        ($1, 1, 'OP-010', 'Cięcie płyt', 'Cięcie płyt meblowych na wymiar według specyfikacji', $2, 15.00, 5.00, false, 'Sprawdź wymiary przed cięciem. Użyj piły z odpowiednim ostrzem.', true, true),
        ($1, 2, 'OP-020', 'Oklejanie krawędzi', 'Oklejanie krawędzi płyt okleiną melamininową', $3, 20.00, 10.00, false, 'Ustaw temperaturę oklejarki na 200°C. Sprawdź przyczepność okleiny.', true, true),
        ($1, 3, 'OP-030', 'Wiercenie otworów', 'Wiercenie otworów pod zawiasy i półki', $4, 10.00, 3.00, false, 'Użyj szablonu do zaznaczenia pozycji otworów. Głębokość 12mm.', true, true),
        ($1, 4, 'OP-040', 'Kompletowanie elementów', 'Kompletowanie wszystkich elementów szafki wraz z okuciami', $5, 30.00, 0.00, false, 'Sprawdź kompletność według listy BOM. Upewnij się że wszystkie elementy są bez uszkodzeń.', true, true),
        ($1, 5, 'OP-050', 'Pakowanie', 'Pakowanie gotowej szafki w karton z zabezpieczeniem', $6, 10.00, 0.00, false, 'Użyj folii bąbelkowej do zabezpieczenia narożników. Dołącz instrukcję montażu.', false, true)
    `, [
      routingId,
      workCenterMap.get('PILA-MW'),
      workCenterMap.get('OKL-FALIT'),
      workCenterMap.get('WIERT-M1'),
      workCenterMap.get('KOMPL'),
      workCenterMap.get('PAKOW'),
    ]);

    operationsInserted = operationsResult.rowCount || 0;

    // Recalculate total time (using pooled client)
    const client = await pool.connect();
    try {
      await recalculateTotalTime(client, routingId);
    } finally {
      client.release();
    }
  } else {
    // Routing already exists, check if we need to seed operations
    const existingRouting = await pool.query(`
      SELECT id FROM production.production_routings WHERE code = 'ROUT-SZAFKA-01'
    `);

    if (existingRouting.rows.length > 0) {
      const routingId = existingRouting.rows[0].id;
      
      // Check if operations exist
      const existingOps = await pool.query(`
        SELECT COUNT(*) as count FROM production.production_routing_operations WHERE routing_id = $1
      `, [routingId]);

      if (parseInt(existingOps.rows[0].count) === 0) {
        // Seed operations
        const operationsResult = await pool.query(`
          INSERT INTO production.production_routing_operations
          (routing_id, sequence, code, name, description, work_center_id, estimated_time, setup_time, allow_parallel, instructions, quality_check_required, is_active)
          VALUES
            ($1, 1, 'OP-010', 'Cięcie płyt', 'Cięcie płyt meblowych na wymiar według specyfikacji', $2, 15.00, 5.00, false, 'Sprawdź wymiary przed cięciem. Użyj piły z odpowiednim ostrzem.', true, true),
            ($1, 2, 'OP-020', 'Oklejanie krawędzi', 'Oklejanie krawędzi płyt okleiną melamininową', $3, 20.00, 10.00, false, 'Ustaw temperaturę oklejarki na 200°C. Sprawdź przyczepność okleiny.', true, true),
            ($1, 3, 'OP-030', 'Wiercenie otworów', 'Wiercenie otworów pod zawiasy i półki', $4, 10.00, 3.00, false, 'Użyj szablonu do zaznaczenia pozycji otworów. Głębokość 12mm.', true, true),
            ($1, 4, 'OP-040', 'Kompletowanie elementów', 'Kompletowanie wszystkich elementów szafki wraz z okuciami', $5, 30.00, 0.00, false, 'Sprawdź kompletność według listy BOM. Upewnij się że wszystkie elementy są bez uszkodzeń.', true, true),
            ($1, 5, 'OP-050', 'Pakowanie', 'Pakowanie gotowej szafki w karton z zabezpieczeniem', $6, 10.00, 0.00, false, 'Użyj folii bąbelkowej do zabezpieczenia narożników. Dołącz instrukcję montażu.', false, true)
        `, [
          routingId,
          workCenterMap.get('PILA-MW'),
          workCenterMap.get('OKL-FALIT'),
          workCenterMap.get('WIERT-M1'),
          workCenterMap.get('KOMPL'),
          workCenterMap.get('PAKOW'),
        ]);

        operationsInserted = operationsResult.rowCount || 0;

        // Recalculate total time (using pooled client)
        const client = await pool.connect();
        try {
          await recalculateTotalTime(client, routingId);
        } finally {
          client.release();
        }
      }
    }
  }

  return { routingsInserted, operationsInserted };
}
