import { Pool } from "pg";
import type { InsertProductionMaterialMovement, ProductionMaterialMovement } from "@shared/schema";

export class MaterialMovementsService {
  /**
   * Get all material movements with optional filtering
   */
  async getAllMovements(pool: Pool, filters?: {
    productionOrderId?: number;
    carrierId?: number;
    operatorId?: number;
    materialType?: string;
    materialCode?: string;
    status?: string;
    sourceLocationId?: number;
    targetLocationId?: number;
    startDate?: Date;
    endDate?: Date;
    movementBatchId?: string;
    limit?: number;
    offset?: number;
  }): Promise<ProductionMaterialMovement[]> {
    let query = `
      SELECT * FROM production.production_material_movements
      WHERE 1=1
    `;
    const params: any[] = [];
    let paramIndex = 1;

    if (filters?.productionOrderId) {
      query += ` AND production_order_id = $${paramIndex++}`;
      params.push(filters.productionOrderId);
    }

    if (filters?.carrierId) {
      query += ` AND carrier_id = $${paramIndex++}`;
      params.push(filters.carrierId);
    }

    if (filters?.operatorId) {
      query += ` AND operator_id = $${paramIndex++}`;
      params.push(filters.operatorId);
    }

    if (filters?.materialType) {
      query += ` AND material_type = $${paramIndex++}`;
      params.push(filters.materialType);
    }

    if (filters?.materialCode) {
      query += ` AND material_code ILIKE $${paramIndex++}`;
      params.push(`%${filters.materialCode}%`);
    }

    if (filters?.status) {
      query += ` AND status = $${paramIndex++}`;
      params.push(filters.status);
    }

    if (filters?.sourceLocationId) {
      query += ` AND source_location_id = $${paramIndex++}`;
      params.push(filters.sourceLocationId);
    }

    if (filters?.targetLocationId) {
      query += ` AND target_location_id = $${paramIndex++}`;
      params.push(filters.targetLocationId);
    }

    if (filters?.startDate) {
      query += ` AND movement_date >= $${paramIndex++}`;
      params.push(filters.startDate);
    }

    if (filters?.endDate) {
      query += ` AND movement_date <= $${paramIndex++}`;
      params.push(filters.endDate);
    }

    if (filters?.movementBatchId) {
      query += ` AND movement_batch_id = $${paramIndex++}`;
      params.push(filters.movementBatchId);
    }

    query += ` ORDER BY movement_date DESC`;

    if (filters?.limit) {
      query += ` LIMIT $${paramIndex++}`;
      params.push(filters.limit);
    }

    if (filters?.offset) {
      query += ` OFFSET $${paramIndex++}`;
      params.push(filters.offset);
    }

    const result = await pool.query(query, params);
    return result.rows as ProductionMaterialMovement[];
  }

  /**
   * Get a single movement by ID
   */
  async getMovementById(pool: Pool, id: number): Promise<ProductionMaterialMovement | null> {
    const result = await pool.query(
      `SELECT * FROM production.production_material_movements WHERE id = $1`,
      [id]
    );
    return result.rows[0] as ProductionMaterialMovement || null;
  }

  /**
   * Create a new material movement
   */
  async createMovement(pool: Pool, data: InsertProductionMaterialMovement): Promise<ProductionMaterialMovement> {
    const result = await pool.query(
      `INSERT INTO production.production_material_movements (
        movement_batch_id, material_type, material_code, material_name,
        quantity, quantity_rejected, unit,
        source_location_id, target_location_id, carrier_id,
        production_order_id, work_order_id, work_center_id, routing_operation_id,
        operator_id, scanned_barcode, scan_method,
        status, movement_type, notes, metadata, movement_date
      ) VALUES (
        $1, $2, $3, $4, $5, $6, $7, $8, $9, $10,
        $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21, $22
      ) RETURNING *`,
      [
        data.movementBatchId || null,
        data.materialType,
        data.materialCode,
        data.materialName || null,
        data.quantity,
        data.quantityRejected || 0,
        data.unit || 'szt',
        data.sourceLocationId || null,
        data.targetLocationId || null,
        data.carrierId || null,
        data.productionOrderId || null,
        data.workOrderId || null,
        data.workCenterId || null,
        data.routingOperationId || null,
        data.operatorId || null,
        data.scannedBarcode || null,
        data.scanMethod || null,
        data.status || 'planned',
        data.movementType || null,
        data.notes || null,
        data.metadata ? JSON.stringify(data.metadata) : null,
        data.movementDate || new Date(),
      ]
    );
    return result.rows[0] as ProductionMaterialMovement;
  }

  /**
   * Batch create multiple movements (e.g., from scanning)
   * Uses transaction to ensure atomicity - all succeed or all fail
   */
  async createBatchMovements(pool: Pool, movements: InsertProductionMaterialMovement[]): Promise<ProductionMaterialMovement[]> {
    const batchId = `BATCH-${Date.now()}`;
    const client = await pool.connect();
    
    try {
      await client.query('BEGIN');
      
      const results: ProductionMaterialMovement[] = [];
      
      for (const movement of movements) {
        const result = await client.query(
          `INSERT INTO production.production_material_movements (
            movement_batch_id, material_type, material_code, material_name,
            quantity, quantity_rejected, unit,
            source_location_id, target_location_id, carrier_id,
            production_order_id, work_order_id, work_center_id, routing_operation_id,
            operator_id, scanned_barcode, scan_method,
            status, movement_type, notes, metadata, movement_date
          ) VALUES (
            $1, $2, $3, $4, $5, $6, $7, $8, $9, $10,
            $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21, $22
          ) RETURNING *`,
          [
            batchId,
            movement.materialType,
            movement.materialCode,
            movement.materialName || null,
            movement.quantity,
            movement.quantityRejected || 0,
            movement.unit || 'szt',
            movement.sourceLocationId || null,
            movement.targetLocationId || null,
            movement.carrierId || null,
            movement.productionOrderId || null,
            movement.workOrderId || null,
            movement.workCenterId || null,
            movement.routingOperationId || null,
            movement.operatorId || null,
            movement.scannedBarcode || null,
            movement.scanMethod || null,
            movement.status || 'planned',
            movement.movementType || null,
            movement.notes || null,
            movement.metadata ? JSON.stringify(movement.metadata) : null,
            movement.movementDate || new Date(),
          ]
        );
        results.push(result.rows[0] as ProductionMaterialMovement);
      }
      
      await client.query('COMMIT');
      return results;
    } catch (error) {
      await client.query('ROLLBACK');
      throw error;
    } finally {
      client.release();
    }
  }

  /**
   * Update movement status
   */
  async updateMovementStatus(pool: Pool, id: number, status: string): Promise<ProductionMaterialMovement> {
    const result = await pool.query(
      `UPDATE production.production_material_movements 
       SET status = $1, updated_at = NOW()
       WHERE id = $2
       RETURNING *`,
      [status, id]
    );
    return result.rows[0] as ProductionMaterialMovement;
  }

  /**
   * Update a movement - handles all mutable columns
   */
  async updateMovement(pool: Pool, id: number, data: Partial<InsertProductionMaterialMovement>): Promise<ProductionMaterialMovement> {
    const fields: string[] = [];
    const values: any[] = [];
    let paramIndex = 1;

    // Material identification
    if (data.materialType !== undefined) {
      fields.push(`material_type = $${paramIndex++}`);
      values.push(data.materialType);
    }
    if (data.materialCode !== undefined) {
      fields.push(`material_code = $${paramIndex++}`);
      values.push(data.materialCode);
    }
    if (data.materialName !== undefined) {
      fields.push(`material_name = $${paramIndex++}`);
      values.push(data.materialName);
    }
    
    // Quantity tracking
    if (data.quantity !== undefined) {
      fields.push(`quantity = $${paramIndex++}`);
      values.push(data.quantity);
    }
    if (data.quantityRejected !== undefined) {
      fields.push(`quantity_rejected = $${paramIndex++}`);
      values.push(data.quantityRejected);
    }
    if (data.unit !== undefined) {
      fields.push(`unit = $${paramIndex++}`);
      values.push(data.unit);
    }
    
    // Location flow
    if (data.sourceLocationId !== undefined) {
      fields.push(`source_location_id = $${paramIndex++}`);
      values.push(data.sourceLocationId);
    }
    if (data.targetLocationId !== undefined) {
      fields.push(`target_location_id = $${paramIndex++}`);
      values.push(data.targetLocationId);
    }
    
    // Carrier and operator
    if (data.carrierId !== undefined) {
      fields.push(`carrier_id = $${paramIndex++}`);
      values.push(data.carrierId);
    }
    if (data.operatorId !== undefined) {
      fields.push(`operator_id = $${paramIndex++}`);
      values.push(data.operatorId);
    }
    
    // Work order context
    if (data.productionOrderId !== undefined) {
      fields.push(`production_order_id = $${paramIndex++}`);
      values.push(data.productionOrderId);
    }
    if (data.workOrderId !== undefined) {
      fields.push(`work_order_id = $${paramIndex++}`);
      values.push(data.workOrderId);
    }
    if (data.workCenterId !== undefined) {
      fields.push(`work_center_id = $${paramIndex++}`);
      values.push(data.workCenterId);
    }
    if (data.routingOperationId !== undefined) {
      fields.push(`routing_operation_id = $${paramIndex++}`);
      values.push(data.routingOperationId);
    }
    
    // Scanning metadata
    if (data.scannedBarcode !== undefined) {
      fields.push(`scanned_barcode = $${paramIndex++}`);
      values.push(data.scannedBarcode);
    }
    if (data.scanMethod !== undefined) {
      fields.push(`scan_method = $${paramIndex++}`);
      values.push(data.scanMethod);
    }
    
    // Movement metadata
    if (data.status !== undefined) {
      fields.push(`status = $${paramIndex++}`);
      values.push(data.status);
    }
    if (data.movementType !== undefined) {
      fields.push(`movement_type = $${paramIndex++}`);
      values.push(data.movementType);
    }
    if (data.movementDate !== undefined) {
      fields.push(`movement_date = $${paramIndex++}`);
      values.push(data.movementDate);
    }
    if (data.notes !== undefined) {
      fields.push(`notes = $${paramIndex++}`);
      values.push(data.notes);
    }
    if (data.metadata !== undefined) {
      fields.push(`metadata = $${paramIndex++}`);
      values.push(data.metadata ? JSON.stringify(data.metadata) : null);
    }
    if (data.movementBatchId !== undefined) {
      fields.push(`movement_batch_id = $${paramIndex++}`);
      values.push(data.movementBatchId);
    }

    if (fields.length === 0) {
      throw new Error('No fields to update');
    }

    fields.push(`updated_at = NOW()`);
    values.push(id);

    const result = await pool.query(
      `UPDATE production.production_material_movements 
       SET ${fields.join(', ')}
       WHERE id = $${paramIndex}
       RETURNING *`,
      values
    );

    return result.rows[0] as ProductionMaterialMovement;
  }

  /**
   * Delete a movement
   */
  async deleteMovement(pool: Pool, id: number): Promise<void> {
    await pool.query(
      `DELETE FROM production.production_material_movements WHERE id = $1`,
      [id]
    );
  }

  /**
   * Get movement flow for a production order (chronological)
   */
  async getProductionOrderFlow(pool: Pool, productionOrderId: number): Promise<ProductionMaterialMovement[]> {
    const result = await pool.query(
      `SELECT * FROM production.production_material_movements
       WHERE production_order_id = $1
       ORDER BY movement_date ASC`,
      [productionOrderId]
    );
    return result.rows as ProductionMaterialMovement[];
  }

  /**
   * Get movements by carrier (track carrier history)
   */
  async getCarrierHistory(pool: Pool, carrierId: number): Promise<ProductionMaterialMovement[]> {
    const result = await pool.query(
      `SELECT * FROM production.production_material_movements
       WHERE carrier_id = $1
       ORDER BY movement_date DESC
       LIMIT 100`,
      [carrierId]
    );
    return result.rows as ProductionMaterialMovement[];
  }

  /**
   * Get current location inventory (materials currently in location)
   */
  async getLocationInventory(pool: Pool, locationId: number): Promise<any[]> {
    const result = await pool.query(
      `SELECT 
        material_code,
        material_name,
        material_type,
        SUM(CASE WHEN target_location_id = $1 THEN quantity ELSE 0 END) as quantity_in,
        SUM(CASE WHEN source_location_id = $1 THEN quantity ELSE 0 END) as quantity_out,
        SUM(CASE WHEN target_location_id = $1 THEN quantity ELSE 0 END) - 
        SUM(CASE WHEN source_location_id = $1 THEN quantity ELSE 0 END) as current_quantity
       FROM production.production_material_movements
       WHERE (source_location_id = $1 OR target_location_id = $1)
         AND status = 'completed'
       GROUP BY material_code, material_name, material_type
       HAVING SUM(CASE WHEN target_location_id = $1 THEN quantity ELSE 0 END) - 
              SUM(CASE WHEN source_location_id = $1 THEN quantity ELSE 0 END) > 0`,
      [locationId]
    );
    return result.rows;
  }
}

export const materialMovementsService = new MaterialMovementsService();
