import type { Express } from "express";
import { pool } from "../../postgres";
import { requirePermission } from "../../auth";

export function registerCutPatternRunRoutes(app: Express) {
  // GET /api/production/cut-pattern-runs - Lista wszystkich instancji produkcyjnych
  app.get("/api/production/cut-pattern-runs", requirePermission("view_production"), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          cpr.id,
          cpr.run_number,
          cpr.template_id,
          cpr.template_code,
          cpr.template_name,
          cpr.plan_line_id,
          cpr.status,
          cpr.scheduled_at,
          cpr.started_at,
          cpr.completed_at,
          cpr.quantity_multiplier,
          cpr.notes,
          cpr.metadata,
          cpr.is_active,
          cpr.created_at,
          cpr.updated_at,
          COALESCE(COUNT(cpri.id), 0) as items_count,
          COALESCE(SUM(cpri.quantity_requested), 0) as total_quantity,
          COALESCE(SUM(CASE WHEN cpri.status = 'zaplanowana' THEN cpri.quantity_requested ELSE 0 END), 0) as planned_quantity,
          COALESCE(SUM(CASE WHEN cpri.status = 'w_produkcji' THEN cpri.quantity_requested ELSE 0 END), 0) as in_production_quantity,
          COALESCE(SUM(CASE WHEN cpri.status = 'wykonana' THEN cpri.quantity_requested ELSE 0 END), 0) as completed_quantity
        FROM production.cut_pattern_runs cpr
        LEFT JOIN production.cut_pattern_run_items cpri ON cpri.run_id = cpr.id AND cpri.is_active = true
        WHERE cpr.is_active = true
        GROUP BY cpr.id
        ORDER BY cpr.scheduled_at DESC NULLS LAST, cpr.created_at DESC
      `);
      
      res.json(result.rows);
    } catch (error: any) {
      console.error("❌ Error fetching cut pattern runs:", error);
      res.status(500).json({ 
        error: "Błąd pobierania instancji produkcyjnych",
        details: error.message 
      });
    }
  });

  // GET /api/production/cut-pattern-runs/:id - Szczegóły run z formatkami
  app.get("/api/production/cut-pattern-runs/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID" });
      }
      
      const runResult = await pool.query(`
        SELECT 
          cpr.id,
          cpr.run_number,
          cpr.template_id,
          cpr.template_code,
          cpr.template_name,
          cpr.plan_line_id,
          cpr.status,
          cpr.scheduled_at,
          cpr.started_at,
          cpr.completed_at,
          cpr.quantity_multiplier,
          cpr.notes,
          cpr.metadata,
          cpr.is_active,
          cpr.created_at,
          cpr.updated_at
        FROM production.cut_pattern_runs cpr
        WHERE cpr.id = $1 AND cpr.is_active = true
      `, [id]);
      
      if (runResult.rows.length === 0) {
        return res.status(404).json({ error: "Instancja produkcyjna nie znaleziona" });
      }
      
      const run = runResult.rows[0];
      
      const itemsResult = await pool.query(`
        SELECT 
          id,
          template_item_id,
          item_name,
          length,
          width,
          thickness,
          board_material_id,
          edging_material_id,
          board_code,
          edging_code,
          color_code,
          edge1,
          edge2,
          edge3,
          edge4,
          quantity_requested,
          quantity_completed,
          position_metadata,
          status,
          notes,
          sort_order,
          is_active,
          created_at,
          updated_at
        FROM production.cut_pattern_run_items
        WHERE run_id = $1 AND is_active = true
        ORDER BY sort_order ASC, item_name ASC
      `, [id]);
      
      res.json({
        ...run,
        items: itemsResult.rows,
        items_count: itemsResult.rows.length,
        total_quantity: itemsResult.rows.reduce((sum, item) => sum + (item.quantity_requested || 0), 0),
        completed_quantity: itemsResult.rows.reduce((sum, item) => sum + (item.quantity_completed || 0), 0)
      });
    } catch (error: any) {
      console.error("❌ Error fetching run details:", error);
      res.status(500).json({ 
        error: "Błąd pobierania szczegółów",
        details: error.message 
      });
    }
  });

  // POST /api/production/cut-pattern-templates/:id/runs - Utwórz instancję produkcyjną z szablonu
  app.post("/api/production/cut-pattern-templates/:templateId/runs", requirePermission("manage_production"), async (req, res) => {
    const client = await pool.connect();
    
    try {
      const templateId = parseInt(req.params.templateId);
      
      if (isNaN(templateId)) {
        return res.status(400).json({ error: "Nieprawidłowe ID szablonu" });
      }
      
      const {
        scheduledAt,
        quantityMultiplier = 1,
        planLineId,
        notes,
        metadata
      } = req.body;
      
      await client.query('BEGIN');
      
      // Pobierz szablon z formatkami
      const templateResult = await client.query(`
        SELECT 
          id, code, name, board_length, board_width, board_thickness, kerf,
          image_url, opis
        FROM production.cut_pattern_templates
        WHERE id = $1 AND is_active = true
      `, [templateId]);
      
      if (templateResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Szablon nie znaleziony" });
      }
      
      const template = templateResult.rows[0];
      
      // Pobierz formatki szablonu
      const itemsResult = await client.query(`
        SELECT *
        FROM production.cut_pattern_template_items
        WHERE template_id = $1 AND is_active = true
        ORDER BY sort_order ASC
      `, [templateId]);
      
      if (itemsResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Szablon nie zawiera żadnych formatek" });
      }
      
      // Generuj unikalny numer run (RUN-YYYY-NNNN)
      const year = new Date().getFullYear();
      const countResult = await client.query(`
        SELECT COUNT(*) as count
        FROM production.cut_pattern_runs
        WHERE run_number LIKE $1
      `, [`RUN-${year}-%`]);
      
      const nextNum = (parseInt(countResult.rows[0].count) + 1).toString().padStart(4, '0');
      const runNumber = `RUN-${year}-${nextNum}`;
      
      // Utwórz run
      const runResult = await client.query(`
        INSERT INTO production.cut_pattern_runs (
          run_number, template_id, template_code, template_name, plan_line_id,
          scheduled_at, quantity_multiplier, notes, metadata, created_by
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING id
      `, [
        runNumber, templateId, template.code, template.name, planLineId,
        scheduledAt, quantityMultiplier, notes,
        metadata ? JSON.stringify(metadata) : null,
        req.user?.id || null
      ]);
      
      const runId = runResult.rows[0].id;
      
      // Skopiuj formatki z szablonu do run (snapshot)
      for (const item of itemsResult.rows) {
        const quantityRequested = Math.round(item.quantity_default * quantityMultiplier);
        
        await client.query(`
          INSERT INTO production.cut_pattern_run_items (
            run_id, template_item_id, item_name, length, width, thickness,
            board_material_id, edging_material_id, board_code, edging_code, color_code,
            edge1, edge2, edge3, edge4, quantity_requested, position_metadata, notes, sort_order
          ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19)
        `, [
          runId, item.id, item.item_name, item.length, item.width, item.thickness,
          item.board_material_id, item.edging_material_id, item.board_code, item.edging_code, item.color_code,
          item.edge1, item.edge2, item.edge3, item.edge4, quantityRequested,
          item.position_metadata, item.notes, item.sort_order
        ]);
      }
      
      // Zwiększ usage_count szablonu i ustaw last_used_at
      await client.query(`
        UPDATE production.cut_pattern_templates
        SET usage_count = usage_count + 1, last_used_at = NOW(), updated_at = NOW()
        WHERE id = $1
      `, [templateId]);
      
      await client.query('COMMIT');
      
      console.log(`✅ Created production run ${runNumber} from template ${template.code} with ${itemsResult.rows.length} items`);
      
      res.json({ 
        success: true, 
        message: `Instancja produkcyjna ${runNumber} została utworzona`,
        runId,
        runNumber
      });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error creating production run:", error);
      res.status(500).json({ 
        error: "Błąd tworzenia instancji produkcyjnej",
        details: error.message 
      });
    } finally {
      client.release();
    }
  });

  // POST /api/production/cut-pattern-runs/:id/start - Rozpocznij produkcję
  app.post("/api/production/cut-pattern-runs/:id/start", requirePermission("manage_production"), async (req, res) => {
    const client = await pool.connect();
    
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID" });
      }
      
      await client.query('BEGIN');
      
      // Sprawdź czy run istnieje i jest w statusie planowany
      const runCheck = await client.query(
        `SELECT id, run_number, status FROM production.cut_pattern_runs WHERE id = $1 AND is_active = true`,
        [id]
      );
      
      if (runCheck.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Instancja produkcyjna nie znaleziona" });
      }
      
      if (runCheck.rows[0].status !== 'planowany') {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: `Run już ma status: ${runCheck.rows[0].status}` });
      }
      
      // Zmień status run
      await client.query(`
        UPDATE production.cut_pattern_runs
        SET status = 'w_produkcji', started_at = NOW(), updated_at = NOW()
        WHERE id = $1
      `, [id]);
      
      // Zmień status wszystkich formatek
      await client.query(`
        UPDATE production.cut_pattern_run_items
        SET status = 'w_produkcji', updated_at = NOW()
        WHERE run_id = $1 AND status = 'zaplanowana' AND is_active = true
      `, [id]);
      
      await client.query('COMMIT');
      
      console.log(`✅ Started production run ${runCheck.rows[0].run_number}`);
      
      res.json({ success: true, message: "Produkcja rozpoczęta" });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error starting production run:", error);
      res.status(500).json({ 
        error: "Błąd rozpoczynania produkcji",
        details: error.message 
      });
    } finally {
      client.release();
    }
  });

  // POST /api/production/cut-pattern-runs/:id/complete - Zakończ produkcję
  app.post("/api/production/cut-pattern-runs/:id/complete", requirePermission("manage_production"), async (req, res) => {
    const client = await pool.connect();
    
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID" });
      }
      
      await client.query('BEGIN');
      
      const runCheck = await client.query(
        `SELECT id, run_number, status FROM production.cut_pattern_runs WHERE id = $1 AND is_active = true`,
        [id]
      );
      
      if (runCheck.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Instancja produkcyjna nie znaleziona" });
      }
      
      if (runCheck.rows[0].status !== 'w_produkcji') {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: `Run musi być w produkcji aby go zakończyć. Obecny status: ${runCheck.rows[0].status}` });
      }
      
      // Zmień status run
      await client.query(`
        UPDATE production.cut_pattern_runs
        SET status = 'wykonany', completed_at = NOW(), updated_at = NOW()
        WHERE id = $1
      `, [id]);
      
      // Zmień status wszystkich formatek i ustaw quantity_completed
      await client.query(`
        UPDATE production.cut_pattern_run_items
        SET 
          status = 'wykonana',
          quantity_completed = quantity_requested,
          updated_at = NOW()
        WHERE run_id = $1 AND status = 'w_produkcji' AND is_active = true
      `, [id]);
      
      await client.query('COMMIT');
      
      console.log(`✅ Completed production run ${runCheck.rows[0].run_number}`);
      
      res.json({ success: true, message: "Produkcja zakończona" });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error completing production run:", error);
      res.status(500).json({ 
        error: "Błąd kończenia produkcji",
        details: error.message 
      });
    } finally {
      client.release();
    }
  });

  // PATCH /api/production/cut-pattern-runs/:id - Edytuj run
  app.patch("/api/production/cut-pattern-runs/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID" });
      }
      
      const {
        scheduledAt,
        notes,
        metadata
      } = req.body;
      
      const result = await pool.query(`
        UPDATE production.cut_pattern_runs
        SET 
          scheduled_at = COALESCE($1, scheduled_at),
          notes = COALESCE($2, notes),
          metadata = COALESCE($3, metadata),
          updated_at = NOW()
        WHERE id = $4 AND is_active = true
        RETURNING run_number
      `, [
        scheduledAt,
        notes,
        metadata ? JSON.stringify(metadata) : null,
        id
      ]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Instancja produkcyjna nie znaleziona" });
      }
      
      console.log(`✅ Updated production run ${result.rows[0].run_number}`);
      
      res.json({ 
        success: true, 
        message: "Instancja produkcyjna zaktualizowana"
      });
    } catch (error: any) {
      console.error("❌ Error updating production run:", error);
      res.status(500).json({ 
        error: "Błąd aktualizacji",
        details: error.message 
      });
    }
  });

  // DELETE /api/production/cut-pattern-runs/:id - Usuń run (soft delete)
  app.delete("/api/production/cut-pattern-runs/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID" });
      }
      
      const result = await pool.query(`
        UPDATE production.cut_pattern_runs
        SET is_active = false, updated_at = NOW()
        WHERE id = $1 AND is_active = true
        RETURNING run_number
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Instancja produkcyjna nie znaleziona" });
      }
      
      console.log(`✅ Deleted production run ${result.rows[0].run_number}`);
      
      res.json({ 
        success: true, 
        message: "Instancja produkcyjna usunięta"
      });
    } catch (error: any) {
      console.error("❌ Error deleting production run:", error);
      res.status(500).json({ 
        error: "Błąd usuwania",
        details: error.message 
      });
    }
  });
}
